/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QSqlQuery>
#include <QSqlRecord>
#include <QDebug>
#include <QVariant>
#include <QPixmap>
#include <QSqlError>
#include <QFile>
#include "definitions.h"
#include "mapdatabase.h"
#include "maprepository.h"

//! Constructor.
MapDatabase::MapDatabase(MapRepository repository,
                         const QString &aMapImageDirectory) :
    dbType("QSQLITE"),
    connectionName("mapDb"),
    mapImageDirectory(aMapImageDirectory)
{
    db = QSqlDatabase::addDatabase(dbType, connectionName);

    // Set the path of the database file and the file name.
    QString dbFilename(aMapImageDirectory);
    dbFilename.append(repository.getDatabase());
    db.setDatabaseName(dbFilename);

    if (db.open()) {
        createTable();
        db.close();
    }
}

//! Destructor.
MapDatabase::~MapDatabase()
{
    db.close();
}

//! Creates database connection.
bool MapDatabase::createConnection()
{
    if (db.open()) {
        return true;
    }

    return false;
}

//! Creates a table of database for map tiles.
void MapDatabase::createTable()
{
    QSqlQuery query(db);

    query.exec("CREATE table maptiles (key varchar(30) primary key, "
               "zoom integer, tilex integer, tiley integer, pixmap "
               "varchar(30))");
}

//! Inserts a map tile to created database and saves the image of the map tile.
bool MapDatabase::insertMapTile(MapTile *tile)
{
    QPixmap pixmap(tile->pixmap());
    QString imagePath = mapImageDirectory;
    QString key((QString("%1").arg(tile->getKey())));
    key.replace(QString("/"), QString("_"));
    imagePath.append(key);

    if (!pixmap.save(imagePath, "PNG")) {
        qDebug() << "Failed to save image.";
    }

    QSqlQuery query(db);

    query.prepare("INSERT INTO maptiles (key, zoom, tilex, tiley, pixmap)"
                  "VALUES (?, ?, ?, ?, ?)");

    query.bindValue(0, tile->getKey());
    query.bindValue(1, tile->getZoomLevel());
    query.bindValue(2, tile->getMapXCoordinate());
    query.bindValue(3, tile->getMapYCoordinate());
    query.bindValue(4, imagePath);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Replace map tile in database.
bool MapDatabase::replaceMapTile(MapTile *tile)
{
    QPixmap pixmap(tile->pixmap());
    QString imagePath = mapImageDirectory;
    QString key((QString("%1").arg(tile->getKey())));
    key.replace(QString("/"), QString("_"));
    imagePath.append(key);

    if (!pixmap.save(imagePath, "PNG")) {
        
    }

    QSqlQuery query(db);

    query.prepare("REPLACE INTO maptiles (key, zoom, tilex, tiley, pixmap)"
                  "VALUES (?, ?, ?, ?, ?)");

    query.bindValue(0, tile->getKey());
    query.bindValue(1, tile->getZoomLevel());
    query.bindValue(2, tile->getMapXCoordinate());
    query.bindValue(3, tile->getMapYCoordinate());
    query.bindValue(4, imagePath);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Remove tile from database.
bool MapDatabase::removeMapTile(const QString &key)
{
    QSqlQuery query(db);
    QString str(key);
    str.replace(QString("_"), QString("/"));

    query.prepare("DELETE FROM maptiles WHERE key=?");

    query.bindValue(0, str);

    if (query.exec()) {
        return true;
    }

    return false;
}

//! Find a map tile in the database.
bool MapDatabase::findMapTile(const QString &key)
{
    QSqlQuery query(db);
    query.prepare("SELECT tilex FROM maptiles WHERE key=?");
    query.bindValue(0, key);
    query.exec();

    while (query.next()) {
        return true;
    }

    return false;
}

//! Returns a map tile matching to the key from database.
MapTile *MapDatabase::getMapTile(const QString &key)
{
    MapTile *tile = NULL;
    QSqlQuery query(db);
    query.prepare("SELECT pixmap FROM maptiles WHERE key=?");
    query.bindValue(0, key);
    query.exec();
    int fieldNo = query.record().indexOf("pixmap");

    while (query.next()) {
        QPixmap pm;

        if (pm.load(query.value(fieldNo).toString())) {
            if (!pm.isNull()) {
                tile = new MapTile();
                tile->setPixmap(pm);
            }
        }
    }

    return tile;
}

//! If database and map images are moved to a new directory, call this
void MapDatabase::changeDatabaseDirectory(const QString &newImageDirectory)
{
    QSqlQuery query(db);
    query.prepare("SELECT key, pixmap FROM maptiles");
    query.exec();

    while (query.next()) {
        QStringList oldImageName = query.value(1).toString().split("/");
        QString newImageName = QString(newImageDirectory)
                               .append(oldImageName.last());

        QSqlQuery query2(db);
        query2.prepare("UPDATE maptiles SET pixmap=? WHERE key=?");
        query2.bindValue(0, newImageName);
        query2.bindValue(1, query.value(0).toString());
        query2.exec();
    }
}

//! Prints the contents of the database. For testing purposes.
void MapDatabase::printDatabaseInformation()
{
    QSqlQuery query2(db);
    query2.prepare("SELECT key, zoom, tilex, tiley, pixmap FROM maptiles");
    while (query2.next()) {
        qDebug() << query2.value(0).toString();
        qDebug() << query2.value(1).toString();
        qDebug() << query2.value(2).toString();
        qDebug() << query2.value(3).toString();
        qDebug() << query2.value(4).toString();
    }
}
