/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __LOCATION_H__
#define __LOCATION_H__

/*!
 * Represents a map location. Contains the location information in
 * latitude/longitude format, the map x and y coordinates at the location
 * and the screen x and y coordinates where the tile will be drawn.
 */
class Location
{
public:
    //! Default constructor.
    Location();

    /*!
     * Constructs a Location object and initializes it with the given values.
     *
     * \param zoomLevel The map zoom level.
     * \param mapX The map x coordinate.
     * \param mapY The map y coordinate.
     * \param screenX The screen x coordinate.
     * \param screenY The screen y coordinate.
     */
    Location(qint32 zoomLevel, qint32 mapX, qint32 mapY,
             qint32 screenX, qint32 screenY);

    //! Destructor.
    ~Location();

    //! Returns the zoomLevel property.
    qint32 getZoomLevel() const { return zoomLevel; }

    //! Sets the zoomLevel property.
    void setZoomLevel(qint32 zoomLevel) { this->zoomLevel = zoomLevel; }

    //! Returns the longitude property.
    qreal getLongitude() const { return longitude; }

    //! Sets the longitude property.
    void setLongitude(qreal longitude) { this->longitude = longitude; }

    //! Returns the latitude property.
    qreal getLatitude() const { return latitude; }

    //! Sets the latitude property.
    void setLatitude(qreal latitude) { this->latitude = latitude; }

    //! Returns the mapX property.
    qint32 getMapX() const { return mapX; }

    //! Sets the mapY property.
    void setMapX(qint32 mapX) { this->mapX = mapX; }

    //! Returns the mapY property.
    qint32 getMapY() const { return mapY; }

    //! Sets the mapY property.
    void setMapY(qint32 mapY) { this->mapY = mapY; }

    //! Returns the screenX property.
    qint32 getScreenX() const { return screenX; }

    //! Sets the screenX property.
    void setScreenX(qint32 screenX) { this->screenX = screenX; }

    //! Returns the screenY property.
    qint32 getScreenY() const { return screenY; }

    //! Sets the screenY property.
    void setScreenY(qint32 screenY) { this->screenY = screenY; }

    /*!
     * \brief Get map tile x coordinate.
     *
     * Returns the x coordinate of the map tile at the
     * given longitude and zoom level.
     *
     * \param longitude The longitude.
     * \param The map zoom level.
     * \return The map tile x coordinate.
     */
    int getXTile(const double longitude, const int zoomLevel);

    /*!
     * \brief Get map tile y coordinate.
     *
     * Returns the y coordinate of the map tile at the
     * given latitude and zoom level.
     *
     * \param latitude The latitude.
     * \param zoomLevel The map zoom level.
     * \return The map tile y coordinate.
     */
    int getYTile(const double latitude, const int zoomLevel);

    /*!
     * \brief Get x pixel coordinate.
     *
     * Returns the x coordinate of the pixel at the
     * given longitude and zoom level.
     *
     * \param longitude The longitude.
     * \param zoomLevel The map zoom level.
     * \return The pixel x coordinate.
     */
    int getXPixel(const double longitude, const int zoomLevel);

    /*!
     * \brief Get y pixel coordinate.
     *
     * Returns the y coordinate of the pixel at the
     * given latitude and zoom level.
     *
     * \param latitude The latitude.
     * \param zoomLevel The map zoom level.
     * \return The pixel y coordinate.
     */
    int getYPixel(const double latitude, const int zoomLevel);

    /*!
     * Returns the latitude at the given map y coordinate and zoom level.
     *
     * \param mapTileY The y coordinate of the map tile.
     * \param zoomLevel The map zoom level.
     * \return The latitude at the map y coordinate.
     */
    double getLatitude(const int mapTileY, const int zoomLevel);

    /*!
     * Returns the longitude at the given map x coordinate and zoom level.
     *
     * \param mapTileX The x coordinate of the map tile.
     * \param zoomLevel The map zoom level.
     * \return The longitude at the map x coordinate.
     */
    double getLongitude(const int mapTileX, const int zoomLevel);

    //! Sets the value of maximumValidLongitude
    void setMaximumValidLongitude(qreal longitude)
        { this->maximumValidLongitude = longitude; }

    //! Sets the value of minimumValidLongitude
    void setMinimumValidLongitude(qreal longitude)
        { this->minimumValidLongitude = longitude; }

    //! Sets the value of maximumValidLatitude
    void setMaximumValidLatitude(qreal latitude)
        { this->maximumValidLatitude = latitude; }

    //! Sets the value of minimumValidLatitude
    void setMinimumValidLatitude(qreal latitude)
        { this->minimumValidLatitude = latitude; }

    /*!
     * \brief Whether set latitude and longitude values are valid
     *
     * Returns true if the set longitude and latitude values are within allowed
     * limits. Otherwise returns false. Note that if you haven't set the max
     * and min values yourself, the returned bool value will always be false.
     *
     * \return Whether set latitude and longitude values are valid
     */
    bool isLocationDataValid();

private:

    /*!
     * \brief Maximum valid value of longitude.
     *
     * Default value for this is zero. Set the value yourself if you need it.
     */
    qreal maximumValidLongitude;

    /*!
     * \brief Minimum valid value of longitude.
     *
     * Default value for this is zero. Set the value yourself if you need it.
     */
    qreal minimumValidLongitude;

    /*!
     * \brief Maximum valid value of latitude.
     *
     * Default value for this is zero. Set the value yourself if you need it.
     */
    qreal maximumValidLatitude;

    /*!
     * \brief Minimum valid value of latitude.
     *
     * Default value for this is zero. Set the value yourself if you need it.
     */
    qreal minimumValidLatitude;

    //! Pi.
    static const double pi = 3.14159265;

    //! Size of the tile, in pixels.
    static const int tileSize = 256;

    //! The zoom level at the location.
    qint32 zoomLevel;

    //! The longitude at the location.
    qreal longitude;

    //! The latitude at the location.
    qreal latitude;

    //! The map x coordinate at the location.
    qint32 mapX;

    //! The map y coordinate at the location.
    qint32 mapY;

    //! The screen x coordinate at the location.
    qint32 screenX;

    //! The screen y coordinate at the location.
    qint32 screenY;

    /*!
     * Returns the x position of the map tile at given longitude and zoom level.
     *
     * \param longitude The longitude in degrees.
     * \param zoomLevel The zoom level of the map.
     * \return The longitude at the map x coordinate.
     */
    double getXCoordinate(const double longitude, const int zoomLevel);

    /*!
     * Returns the y position of the map tile at given latitude and zoom level.
     *
     * \param latitude The latitude in degrees.
     * \param zoomLevel The zoom level of the map.
     * \return The latitude at the map y coordinate.
     */
    double getYCoordinate(const double latitude, const int zoomLevel);
};

#endif // __LOCATION_H__
