/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <cmath>
#include <QDebug>
#include <QtGlobal>
#include "location.h"

//! Default constructor.
Location::Location()
{
    maximumValidLongitude = 0;
    minimumValidLongitude = 0;
    maximumValidLatitude = 0;
    minimumValidLatitude = 0;
}

//! Constructs a Location object and initializes it with the given values.
Location::Location(qint32 zoomLevel, qint32 mapX, qint32 mapY,
                   qint32 screenX, qint32 screenY) :
        zoomLevel(zoomLevel),
        mapX(mapX),
        mapY(mapY),
        screenX(screenX),
        screenY(screenY)
{
    maximumValidLongitude = 0;
    minimumValidLongitude = 0;
    maximumValidLatitude = 0;
    minimumValidLatitude = 0;

}

//! Destructor.
Location::~Location()
{
}

//! Get map tile x coordinate.
int Location::getXTile(const double longitude, const int zoomLevel)
{
    return (int) getXCoordinate(longitude, zoomLevel);
}

//! Get map tile y coordinate.
int Location::getYTile(const double latitude, const int zoomLevel)
{
    return (int) getYCoordinate(latitude, zoomLevel);
}

//! Get x pixel coordinate.
int Location::getXPixel(const double longitude, const int zoomLevel)
{
    double x = getXCoordinate(longitude, zoomLevel);
    x -= (int) x;

    return (int)(tileSize * x);
}

//! Get y pixel coordinate.
int Location::getYPixel(const double latitude, const int zoomLevel)
{
    double y = getYCoordinate(latitude, zoomLevel);
    y -= (int) y;

    return (int)(tileSize * y);
}

//! Returns the x position of the map tile at given longitude and zoom level.
double Location::getXCoordinate(const double longitude, const int zoomLevel)
{
    double x = ((longitude + 180) / 360) * pow(2, zoomLevel);

    return x;
}

//! Returns the y position of the map tile at given latitude and zoom level.
double Location::getYCoordinate(const double latitude, const int zoomLevel)
{
    double y = 0.5 * log((1 + sin(latitude * pi / 180)) /
                         (1 - sin(latitude * pi / 180)));
    double coordinate = (1 - ((y + pi) / (2 * pi))) * pow(2, zoomLevel);

    return coordinate;
}

//! Returns the latitude at the given map y coordinate and zoom level.
double Location::getLatitude(const int mapTileY, const int zoomLevel)
{
    double y = 2 * pi * (1 - (mapTileY / pow(2, zoomLevel))) - pi;
    double latitude = atan(sinh(y)) * 180 / pi;

    return latitude;
}

//! Returns the longitude at the given map x coordinate and zoom level.
double Location::getLongitude(const int mapTileX, const int zoomLevel)
{
    double longitude = (mapTileX * 360 / pow(2, zoomLevel)) - 180;

    return longitude;
}

//! Whether set latitude and longitude values are valid
bool Location::isLocationDataValid()
{
    if (longitude <= maximumValidLongitude &&
        longitude >= minimumValidLongitude &&
        latitude <= maximumValidLatitude &&
        latitude >= minimumValidLatitude) {
        return true;
    }
    else {
        return false;
    }
}
