/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <stdio.h>
#include <QString>
#include "internalgpsn810.h"

#ifdef GPS_N810_IN_USE
#include "gpsstarter.h"
#include "gpsbt.h"  /* GPS BT package */
#include "gps.h"    /* for reading GPS data from daemon */

#ifdef GPS_TEST
#include <QDebug>
#endif

#ifdef N810_INTERNAL_GPS_TEST
#include <QDebug>
#endif


gpsbt_t btGpsContext = { {0, 0, 0, 0, 0, 0}, 0, 0 };  /* clearing the context is important! */

struct gps_data_t *gpsDaemon;

//! Constructor
InternalGpsN810::InternalGpsN810()
{
    firstStart = true;
    startingThread = NULL;
    startingThread = new GpsStarter();
    connect(startingThread, SIGNAL(gpsrunning(bool, QString)),
            this, SLOT(gpsRunning(bool, QString)));
    startingThread->start();
}

//! Destructor
InternalGpsN810::~InternalGpsN810()
{
    stopGps();
    delete startingThread;
}

//! Starts gps when called.
void InternalGpsN810::startGps()
{
    this->restartGps();
}

//! When starting or trying to start gps this function is called.
void InternalGpsN810::gpsRunning(bool run, QString error)
{
    const QString localAddress("127.0.0.1");
    const int numberBase = 10;

    if (run) {
        #ifdef N810_INTERNAL_GPS_TEST
        printf("\n\ngpsRunning slot\n\n");
        printf("\nGPS RUNNING SLOT");
        printf("\nGPS File: %s\n", btGpsContext.mgr.file);
        #endif

        QString port(DEFAULT_GPSD_PORT);
        gpsDaemon = gps_open(localAddress.toStdString().c_str(),
                             DEFAULT_GPSD_PORT);
        socket = new QTcpSocket();
        socket->connectToHost(localAddress, port.toUInt(NULL, numberBase),
                              QIODevice::ReadWrite);

        #ifdef N810_INTERNAL_GPS_TEST
        qDebug() << "PORT: " << DEFAULT_GPSD_PORT
                 << "     " << port.toUInt(NULL, numberBase);
        #endif

        connect(socket, SIGNAL(error(QAbstractSocket::SocketError)),
                this, SLOT(showError(QAbstractSocket::SocketError)));
        connect(socket, SIGNAL(stateChanged(QAbstractSocket::SocketState)),
                this, SLOT(readState(QAbstractSocket::SocketState)));
        connect(socket, SIGNAL(connected()), this, SLOT(socketConnected()));
        connect(socket, SIGNAL(readyRead()), this, SLOT(readData()));

    }
    else {
        #ifdef N810_INTERNAL_GPS_TEST
        qDebug() << "Trying to quit thread";
        #endif

        emit errorOccured(error);
    }
}

//! When some error in socket connection occurs this function is called.
void InternalGpsN810::showError(QAbstractSocket::SocketError socketError)
{
    #ifdef N810_INTERNAL_GPS_TEST
    qDebug() << "Socket error: " << socketError ;
    #endif

    emit errorOccured(tr("Socket error: %1").arg(socketError));
}

//! When socket is connected this is called.
void InternalGpsN810::socketConnected()
{
    const char *lineBreak = "r+\r\n";

    #ifdef N810_INTERNAL_GPS_TEST
    qDebug() << "Socket connected";
    #endif

    socket->write(lineBreak, sizeof(lineBreak));
}

//! Reads data from the gps socket
void InternalGpsN810::readData()
{
    const QString newLineCharacter("\n");
    QString unparsedNmeaString(socket->readAll());
    QString parsedNmeaString;
    int count = unparsedNmeaString.count();
    QRegExp delimiter(newLineCharacter);

    #ifdef N810_INTERNAL_GPS_TEST
    qDebug() << "bytes available: " << socket->bytesAvailable();
    qDebug() << "ready read: ";
    qDebug() << unparsedNmeaString.toAscii();
    /*qDebug()<<"COUNT: "<<unparsedNmeaString.count("\n")
              << "     INDEX OF "<<unparsedNmeaString.indexOf(delimiter)
              <<"    sect: "<<unparsedNmeaString.section(delimiter, 0, 0) ;*/
    #endif

    for (int i = 0; i < count; i++) {
        parsedNmeaString.append(unparsedNmeaString.section(delimiter, 0, 0));
        parsedNmeaString.append(newLineCharacter);

        #ifdef N810_INTERNAL_GPS_TEST
        qDebug() << "NMEAs: ";
        qDebug() << "section: "
                 << unparsedNmeaString.section(delimiter, 0, 0) << endl;
        qDebug() << "delimiter: " << parsedNmeaString.indexOf(delimiter);
        #endif

        if (parsedNmeaString.length() > 1) {
            #ifdef GPS_TEST
            qDebug() << "NMEA string:" << parsedNmeaString;
            #endif

            emit nmeaSentenceReady(parsedNmeaString);
        }

        unparsedNmeaString.remove(0, unparsedNmeaString
                                  .indexOf(delimiter) + 1);
        parsedNmeaString.clear();
    }
}

//! This function is always called when sockets' state is changed
void InternalGpsN810::readState(QAbstractSocket::SocketState socketState)
{
    Q_UNUSED(socketState);

    #ifdef N810_INTERNAL_GPS_TEST
    qDebug() << "state changed: " << socket->state();
    #endif
}

//! restarts gps when called. Can also be used for starting the gps
void InternalGpsN810::restartGps()
{
    // if thread is not running, restart is allowed.
    if (!startingThread->isRunning()) {
        #ifdef N810_INTERNAL_GPS_TEST
        qDebug() << "RESTART CALLED";
        #endif

        stopGps();
        startingThread->start();
    }
    else if (!firstStart) {
        #ifdef TEST
        qDebug() << "THREAD IS RUNNING...RESTART IS NOT ALLOWED";
        #endif
        emit errorOccured(tr("Couldn't restart gps"));
    }

    firstStart = false;
}

//! stops gps when called.
void InternalGpsN810::stopGps()
{
    if (gpsmgr_is_gpsd_running(&btGpsContext.mgr, NULL,
                               GPSMGR_MODE_JUST_CHECK) == GPSMGR_RUNNING) {

        if (!startingThread->isRunning()) {
            #ifdef N810_INTERNAL_GPS_TEST
            qDebug() << "stop gps requested";
            #endif

            socket->close();
            delete socket;
            gpsbt_stop(&btGpsContext);
            gps_close(gpsDaemon);
            gpsmgr_stop(&btGpsContext.mgr);

            if (gpsmgr_is_gpsd_running(&btGpsContext.mgr, NULL,
                                       GPSMGR_MODE_JUST_CHECK) ==
                                       GPSMGR_RUNNING) {
                emit errorOccured(tr("Couldn't close GPS daemon. Maybe some "
                                     "other program use it or there might be "
                                     "a some other failure")) ;
            }// if gpsmgr_not_running
        }
    }

    #ifdef N810_INTERNAL_GPS_TEST
    if (gpsmgr_is_gpsd_running(&btGpsContext.mgr,
                               NULL, GPSMGR_MODE_JUST_CHECK) ==
                               GPSMGR_NOT_RUNNING) {
        qDebug() << "GPS STOPPED ...  restart or close application";
    }
    #endif
}

#endif // GPS_N810_IN_USE
