/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QMessageBox>
#include <QString>
#include "importpoisdialog.h"
#include "poidatabase.h"
#include "poicategoriesdialog.h"

//! Constructor
ImportPoisDialog::ImportPoisDialog(QVector<Poi> importedPois,
                                   QtMapper &mainWindow,
                                   QWidget *parent)
    : QDialog(parent), importedPois(importedPois), mainWindow(mainWindow)
{
    setupUi(this);

    poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    poiDb->getPoiCategories(categories);

    setData();

    QString infoText(tr("You have imported %1 POIs.\nSelect "
                        "category for POIs to be saved.")
                     .arg(importedPois.count()));
    labelInfo->setText(infoText);

    connect(pushButtonEditCategories, SIGNAL(clicked()),
            this, SLOT(showPoiCategoriesDialog()));
    connect(buttonBox, SIGNAL(accepted()), this, SLOT(savePois()));
    connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
}

//! Sets labels of POI categories to the combobox.
void ImportPoisDialog::setData()
{
    comboBoxCategory->clear();

    for (qint32 i = 0; i < categories.count(); i++) {
        comboBoxCategory->addItem(categories.at(i).getLabel());
    }

    comboBoxCategory->setCurrentIndex(0);
}

//! Shows Dialog POI categories.
void ImportPoisDialog::showPoiCategoriesDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    PoiCategoriesDialog pc(mainWindow);

    if (pc.exec() == QDialog::Accepted) {
        categories.clear();
        poiDb->getPoiCategories(categories);
        setData();
    }
}

//! Saves imported POIs to database.
void ImportPoisDialog::savePois()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    poiDb->getPoiCategories(categories);

    if (categories.count() <= 0) {
        QMessageBox mb;
        mb.setStandardButtons(QMessageBox::Ok);
        mb.setWindowTitle(tr("NOTIFICATION"));
        QString text(tr("You have to add and choose category "
                        "before inserting any POIs"));
        mb.setText(text);
        mb.exec();
        return;
    }

    bool similar = false;
    qint32 countSimilars = 0;

    poiDb->getAllPois(pois);

    for (qint32 i = 0; i < importedPois.count(); i++) {

        for(qint32 j = 0; j < pois.count(); j++) {

            if (importedPois[i].isSimilar(pois.at(j)) == true) {
                similar = true;
            }
        }

        if (similar == false) {

            importedPois[i].setCategory(categories
                                        .at(comboBoxCategory->currentIndex()));
            poiDb->insertPoi(importedPois.at(i));
        }
        else {
            similar = false;
            countSimilars++;
        }
    }

    if (countSimilars > 0) {
        QMessageBox mb(this);
        mb.setStandardButtons(QMessageBox::Ok);
        mb.setWindowTitle(tr("NOTIFICATION"));
        QString text(tr("Didn't save %1 POIs \nbecause there were similar "
                        "POIs in database.").arg(countSimilars));
        mb.setText(text);
        mb.exec();
    }
}
