/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "gpsstarter.h"

#ifdef GPS_N810_IN_USE

#include "internalgpsn810.h"
#include <stdio.h>
#include <QDebug>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include "gpsbt.h"  /* GPS BT package */
#include "gps.h"    /* for reading GPS data from daemon */
#include <errno.h>

const int maxErrorBufferLength = 255;
const int maxGpsDeviceLength = 64;

extern gpsbt_t btGpsContext; /* clearing the context is important! */

char gpsDevice[maxGpsDeviceLength+1];

//! Constructor
GpsStarter::GpsStarter() :
        gpsBtDebugLevel(0),
        gpsDaemonDebugLevel(0),
        gpsConnectionTryingTimeOut(2),
        gpsConnectionMaxTryOuts(10)
{
    #ifdef GPS_TEST
    qDebug() << "GpsStarter constructor...";
    #endif
}

//! Destructor
GpsStarter::~GpsStarter()
{
    #ifdef GPS_TEST
    qDebug() << "GpsStarter destructor...";
    #endif
}

//! Starts the gps manager.
void GpsStarter::run()
{
    #ifdef GPS_TEST
    qDebug() << "GpsStarter is running...";
    #endif

    char errbuf[maxErrorBufferLength+1] = {0};
    gpsBtState = -1;

    bool attemptToConnect = true;
    int tryCount = 0;

    while ((gpsBtState != 0) && attemptToConnect) {
        tryCount++;

        #ifdef GPS_TEST
        qDebug() << "Attempting gpsbt_start, gpsBtState is" << gpsBtState;
        qDebug() << "Try count:" << tryCount;
        #endif

        gpsBtState = gpsbt_start(gpsDevice[0] ? gpsDevice : NULL,
                                 gpsBtDebugLevel,
                                 gpsDaemonDebugLevel,
                                 0, /* port */
                                 errbuf, maxErrorBufferLength,
                                 0, /* timeout */
                                 &btGpsContext);

        sleep(gpsConnectionTryingTimeOut);

        if (gpsmgr_is_gpsd_running(&btGpsContext.mgr,
                                   NULL, GPSMGR_MODE_JUST_CHECK) ==
                                   GPSMGR_NOT_RUNNING) {

            #ifdef GPS_TEST
            qDebug() << "Gps startup failed, gps manager is not running...";
            #endif

            if (tryCount == gpsConnectionMaxTryOuts) {
                attemptToConnect = false;
            }
        }
        else if (gpsmgr_is_gpsd_running(&btGpsContext.mgr, NULL,
                                        GPSMGR_MODE_JUST_CHECK) ==
                                        GPSMGR_RUNNING) {
            #ifdef GPS_TEST
            qDebug() << "gps manager is running...";
            #endif
            attemptToConnect = false;
        }
    }

    if (gpsBtState < 0) {
        #ifdef GPS_TEST
        qDebug() << "Could not start gps manager...";
        printf("\nERROR: %s\n", strerror(errno));
        #endif
        emit gpsrunning(false, strerror(errno));
    }
    else {
        emit gpsrunning(true, NULL);
    }
}

#endif // GPS_N810_IN_USE
