/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __GPSDATA_H__
#define __GPSDATA_H__
#include <QtGlobal>
#include <QObject>

class QTime;
class QTimer;

/*!
 * Class GpsData stores all sorts of information received from the gps device,
 * and also includes a QTime object that stores the current time of the system.
 */

class GpsData : public QObject
{

    Q_OBJECT

public:

    //! Constructor
    /*!
     * Constructor sets all the class variables to zero, with the exception of
     * systemTime, which is set to current time given by the system, and fix
     * which is set to 1.
     * Constructor also starts up the timer that updates the system time
     * variable.
     */
    GpsData();

    //! Destructor
    ~GpsData();

    //! set the type of position fix
    /*!
     * \param fixValue The position fix
     */
    void setFix(int fixValue);

    //! returns the type of position fix
    /*!
     * \return fix class variable
     */
    int getFix() const;

    //! set the quality of position fix
    /*!
     * \param newFixQuality The fix quality
     */
    void setFixQuality(int newFixQuality);

    //! returns the quality of position fix
    /*!
     * \return fixQuality class variable
     */
    int getFixQuality() const;

    //! set the latitude
    /*!
     * If the old value is different than the new one, latitudeChanged
     * signal is emitted.
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param newLatitude GPS latitude in degrees
     */
    void setLatitude(qreal newLatitude);

    //! returns the latitude
    /*!
     * \return GPS latitude in degrees
     */
    qreal getLatitude() const;

    //! set the longitude
    /*!
     * If the old value is different than the new one, longitudeChanged
     * signal is emitted.
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param newLongitude GPS latitude in degrees
     */
    void setLongitude(qreal newLongitude);

    //! returns the longitude
    /*!
     * \return longitude class variable
     */
    qreal getLongitude() const;

    //! set current speed
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param speed Current speed in km/h
     */
    void setSpeed(qreal speed);

    //! returns current speed in km/h
    /*!
     * \return speed class variable
     */
    qreal getSpeed() const;

    //! set the maximum speed measured so far
    /*!
     * \param maxspeed Maximum speed measured so far
     */
    void setMaxSpeed(qreal maxspeed);

    //! returns the maximum speed measured so far
    /*!
     * \return maxspeed class variable
     */
    qreal getMaxSpeed() const;

    //! set the current heading, degrees from true north
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param heading Current heading, degrees from true north
     */
    void setHeading(qreal heading);

    //! returns current heading, degrees from true north
    /*!
     * \return heading class variable
     */
    qreal getHeading() const;

    //! Set the horizontal dilution of precision
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param horizontalDilutionOfPrecision The horizontal dilution of precision
     */
    void setHorizontalDilutionOfPrecision(qreal horizontalDilutionOfPrecision);

    //! Return the horizontal dilution of precision
    /*!
     * \return horizontalDilutionOfPrecision class variable
     */
    qreal getHorizontalDilutionOfPrecision() const;

    //! Set the position dilution of precision
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param positionDilutionOfPrecision The position dilution of precision
     */
    void setPositionDilutionOfPrecision(qreal positionDilutionOfPrecision);

    //! Return the position dilution of precision
    /*!
     * \return positionDilutionOfPrecision class variable
     */
    qreal getPositionDilutionOfPrecision() const;

    //! Set the vertical dilution of precision
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param verticalDilutionOfPrecision The vertical dilution of precision
     */
    void setVerticalDilutionOfPrecision(qreal verticalDilutionOfPrecision);

    //! Return the vertical dilution of precision
    /*!
     * \return verticalDilutionOfPrecision class variable
     */
    qreal getVerticalDilutionOfPrecision() const;

    //! set the amount of satellites in view
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param satellitesInView The number of satellites in view
     */
    void setSatellitesInView(int satellitesInView);

    //! returns the amount of satellites in view
    /*!
     * \return satellitesInView class variable
     */
    int getSatellitesInView() const;

    //! set the number of satellites currently in use
    /*!
     * valueChanged(const QString &, qreal) signal is always emitted when
     * this method is called.
     * \param satellitesInUse The number of satellites currently in use
     */
    void setSatellitesInUse(int satellitesInUse);

    //! returns the amount of satellites currently in use
    /*!
     * \return satellitesInUse class variable
     */
    int getSatellitesInUse() const;

    //! Set satellite numbers used for gps fix
    /*!
     * \param index Table index in satellitesForFix[] class variable
     * \param satelliteForFix The satellites PRN number
     */
    void setSatellitesForFix(int index, int satelliteForFix);

    //! return a particular satellite's PRN number
    /*!
     * \param index Table index in satellitesForFix[] class variable
     * \return PRN number for the requested satellite
     */
    int getSatellitesForFix(int) const;

    //! returns a pointer to QTime object holding current system time
    /*!
     * \return A Pointer to a QTime object that holds current system time
     */
    QTime *getSystemTime();


signals:

    //! A signal that is emitted when latitude class variable changes
    /*!
     * \param latitude The latitude class variable
     */
    void latitudeChanged(qreal latitude);

    //! A signal that is emitted when longitude class variable changes
    /*!
     * \param longitude The longitude class variable
     */
    void longitudeChanged(qreal longitude);

    //! A signal that is emitted when fixquality class variable changes
    /*!
     * \param fixquality The fixquality class variable
     */
    void fixQualityChanged(int fixquality);

    //! A signal that is emitted when one of the class variables is set
    /*!
     * \param const name Class variable's name as QString reference
     * \param qreal value The class variable's value
     */
    void valueChanged(const QString &name, qreal value);

    //! A signal that is emitted when systemTime variable changes
    /*!
     * \param time A reference to systemTime class variable
     */
    void timeChanged(QTime &time);

private slots:

    //! sets systemTime variable to current system time
    /*!
     * timeChanged(QTime &) signal is emitted whenever this
     * slot is called.
     */
    void updateTime();


private:

    //! number of satellites in view
    int satellitesInView;

    //! number of satellites in use
    int satellitesInUse;

    //! position fix
    /*!
     *  3D fix - values include: 1 = no fix, 2 = 2D, 3 = 3D
     */
    int fix;

    //! quality of position fix
    /*!
     * Fix qualities:
     *
     * 0 = invalid
     * 1 = GPS fix (SPS)
     * 2 = DGPS fix
     * 3 = PPS fix
     * 4 = Real Time Kinematic
     * 5 = Float RTK
     * 6 = estimated (dead reckoning) (2.3 feature)
     * 7 = Manual input mode
     * 8 = Simulation mode
     */
    int fixquality;

    //! Stores the numbers of the GPS satellites
    int satellitesForFix[12];

    //! heading, degrees from true north
    qreal heading;

    //! latitude in degrees
    qreal latitude;

    //! longitude in degrees
    qreal longitude;

    //! maximum measured speed, km/h
    qreal maxspeed;

    //! current speed, km/h
    qreal speed;

    //! vertical dilution of precision
    qreal verticalDilutionOfPrecision;

    //! position dilution of precision
    qreal positionDilutionOfPrecision;

    //! horizontal dilution of precision
    qreal horizontalDilutionOfPrecision;

    //! stores the system time
    /*!
     * systemTime object is set to update once every second
     * using the QTimer object timeUpdater
     */
    QTime *systemTime;

    //! a timer for updating the system time object
    QTimer *timeUpdater;

};

#endif // __GPSDATA_H__
