/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "gpsdata.h"

#include <QDebug>
#include <QTime>
#include <QTimer>


//! constructor
GpsData::GpsData()
{
    satellitesInView = 0;
    satellitesInUse = 0;

    fix = 1;
    fixquality = 0;

    satellitesForFix[12] = 0;

    heading = 0;
    latitude = 0;
    longitude = 0;
    maxspeed = 0;
    speed = 0;

    verticalDilutionOfPrecision = 0;
    positionDilutionOfPrecision = 0;
    horizontalDilutionOfPrecision = 0;

    systemTime = new QTime(QTime::currentTime());
    timeUpdater = new QTimer(this);
    connect(timeUpdater, SIGNAL(timeout()), this, SLOT(updateTime()));
    timeUpdater->start(1000);
}

//! Destructor
GpsData::~GpsData()
{
    delete systemTime;
}

//! set the type of position fix
void GpsData::setFix(int fixValue)
{
    this->fix = fixValue;
    emit valueChanged("fixValue", fixValue);
}

//! returns the type of position fix
int GpsData::getFix() const
{
    return fix;
}

//! set the quality of position fix
void GpsData::setFixQuality(int newFixQuality)
{
    if (newFixQuality != this->fixquality) {
        this->fixquality = newFixQuality;
        emit fixQualityChanged(newFixQuality);
    }
    emit valueChanged("fixquality", newFixQuality);
}

//! returns the quality of position fix
int GpsData::getFixQuality() const
{
    return fixquality;
}

//! set the latitude
void GpsData::setLatitude(qreal newLatitude)
{
    if (latitude != newLatitude) {
        this->latitude = newLatitude;
        emit latitudeChanged(latitude);
    }
    emit valueChanged("latitude", latitude);
}

//! returns the latitude
qreal GpsData::getLatitude() const
{
    return latitude;
}

//! set the longitude
void GpsData::setLongitude(qreal newLongitude)
{
    if (longitude != newLongitude) {
        this->longitude = newLongitude;
        emit longitudeChanged(longitude);
    }
    emit valueChanged("longitude", longitude) ;
}

//! returns the longitude
qreal GpsData::getLongitude() const
{
    return longitude;
}

//! set current speed
void GpsData::setSpeed(qreal speed)
{
    this->speed = speed;
    emit valueChanged("speed", speed);
}

//! returns current speed
qreal GpsData::getSpeed() const
{
    return speed;
}

//! set the maximum speed measured so far
void GpsData::setMaxSpeed(qreal maxspeed)
{
    this->maxspeed = maxspeed;
    emit valueChanged("maxspeed", maxspeed) ;
}

//! returns the maximum speed measured so far
qreal  GpsData::getMaxSpeed() const
{
    return maxspeed;
}

//! set the current heading, degrees from true north
void GpsData::setHeading(qreal heading)
{
    this->heading = heading;
    emit valueChanged("heading", heading);
}

//! returns current heading, degrees from true north
qreal GpsData::getHeading() const
{
    return heading;
}

//! Set the horizontal dilution of precision
void GpsData::setHorizontalDilutionOfPrecision
     (qreal horizontalDilutionOfPrecision)
{
    this->horizontalDilutionOfPrecision = horizontalDilutionOfPrecision;
    emit valueChanged("horizontalDilutionOfPrecision",
                      horizontalDilutionOfPrecision);
}

//! Return the horizontal dilution of precision
qreal GpsData::getHorizontalDilutionOfPrecision() const
{
    return horizontalDilutionOfPrecision;
}

//! Set the position dilution of precision
void GpsData::setPositionDilutionOfPrecision(qreal positionDilutionOfPrecision)
{
    this->positionDilutionOfPrecision = positionDilutionOfPrecision;
    emit valueChanged("positionDilutionOfPrecision",
                      positionDilutionOfPrecision);
}

//! Return the position dilution of precision
qreal GpsData::getPositionDilutionOfPrecision() const
{
    return positionDilutionOfPrecision;
}

//! Set the vertical dilution of precision
void GpsData::setVerticalDilutionOfPrecision(qreal verticalDilutionOfPrecision)
{
    this->verticalDilutionOfPrecision = verticalDilutionOfPrecision;
    emit valueChanged("verticalDilutionOfPrecision",
                      verticalDilutionOfPrecision);
}

//! Return the vertical dilution of precision
qreal GpsData::getVerticalDilutionOfPrecision() const
{
    return verticalDilutionOfPrecision;
}

//! set the amount of satellites in view
void GpsData::setSatellitesInView(int satellitesInView)
{
    this->satellitesInView = satellitesInView;
    emit valueChanged("satellitesInView", satellitesInView);
}

//! returns the amount of satellites in view
int GpsData::getSatellitesInView() const
{
    return satellitesInView;
}

//! set the number of satellites currently in use
void GpsData::setSatellitesInUse(int satellitesInUse)
{
    this->satellitesInUse = satellitesInUse;
    emit valueChanged("satellitesInUse", satellitesInUse);
}

//! returns the amount of satellites currently in use
int GpsData::getSatellitesInUse() const
{
    return satellitesInUse;
}

//! Set satellite numbers used for gps fix
void GpsData::setSatellitesForFix(int index, int satelliteForFix)
{
    this->satellitesForFix[index] = satelliteForFix;
}

//! return a particular satellite's PRN number
int GpsData::getSatellitesForFix(int index) const
{
    return satellitesForFix[index];
}

//! returns a pointer to QTime object holding current system time
QTime *GpsData::getSystemTime()
{
    return systemTime;
}

//! sets systemTime variable to current system time
void GpsData::updateTime()
{
    *systemTime = QTime::currentTime();
    emit timeChanged(*systemTime);
}
