/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef GPSBASE_H
#define GPSBASE_H

#include <QObject>

#include "definitions.h"

class GpsData;
class GpsServiceInterface;
class Point ;
class QProgressDialog;
class QString;
class QTextEdit;
class UnitConverter;

//! A global variable for storing satellite data
extern GpsSatelliteData gpsSatelliteDataArray[12];

/*!
 * Class GpsBase provides basic gps functionality
 * such as parsers for different type GPS-sentences
 */

class GpsBase : public QObject
{

    Q_OBJECT

private:

    //! Object for the Gps interface.
    GpsServiceInterface *gps;

    //! Object that holds gps data
    GpsData *gpsData;

    //! Holds a single time variable and a point on the map
    Point *gpsPoint;

    //! GMT offset in hours
    int gmtOffset;

    //! Some useful functions
    UnitConverter *converter;

    //! Reads the NMEA sentence
    /*!
     * This method reads the incoming NMEA sentence and calculates
     * the checksum. If the checksum is ok, gpsCheckNmeaType method
     * is called.
     *
      \param nmea The incoming NMEA sentence
     */
    void readNmea(QString nmea);

    //! checks the type of nmea sentence and calls the appropriate parser
    /*!
     * \param nmea The NMEA sentence
     */
    void gpsCheckNmeaType(QString &nmea);

    //! parses RMC type NMEA sentences
    /*!
     * \param sentence The NMEA sentence
     */
    void parseRMC(const QString &sentence);

    //! parses GGA type NMEA sentences
    /*!
     * \param sentence The NMEA sentence
     */
    void parseGGA(const QString &sentence);

    //! parses GSA type NMEA sentences
    /*!
     * \param sentence The NMEA sentence
     */
    void parseGSA(const QString &sentence);

    //! parses GSV type NMEA sentences
    /*!
     * \param sentence The NMEA sentence
     */
    void parseGSV(QString &sentence);

public slots:

    //! Checks the NMEA sentences for errors.
    /*!
     * If there are no errors, readNmea is called
     *
     * \param nmea The NMEA sentence
     */
    void checkMessage(QString nmea);

    //! Generates error message and calls Qmessagebox to show it for the user.
    /*!
     * \param error The error message from gps
     */
    void generateErrorMessage(QString error);

public:

    //! constructor
    GpsBase();

    //! destructor
    ~GpsBase();

    //! Sends stop signal to gps.
    void stopGps();

    //! Starts gps by sending nmea query for the gps device.
    void startGps();

    //! Returns the used gpsData object
    /*!
     * \return A reference to used gpsData object
     */
    GpsData &getGpsData();

    //! Returns the used gpsPoint object
    /*!
     * \return A reference to used gpsPoint object
     */
    Point &getGpsPoint();

};

#endif
