/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QDebug>
#include <QMessageBox>
#include <QProgressDialog>
#include <QString>
#include <QTextEdit>
#include <QTime>
#include <QtGlobal>

#include <math.h>

#include "gpsbase.h"
#include "gpsdata.h"
#include "gpssatellitedata.h"

#include "point.h"
#include "unitconverter.h"
#include "gpsserviceinterface.h"

#ifdef GPS_SERIAL_IN_USE
#include "gpsserial.h"
#endif

#ifdef GPS_N810_IN_USE
#include "internalgpsn810.h"
#endif

#ifdef GPS_EMULATOR_IN_USE
#include "gpsemulator.h"
#endif

//! A global variable for storing satellite data
GpsSatelliteData gpsSatelliteDataArray[maxSatellitesCount];

//! constructor
GpsBase::GpsBase()
{
    gmtOffset = 0;
    gpsData = new GpsData();
    gpsPoint = new Point();

#ifdef GPS_N810_IN_USE
    gps = new InternalGpsN810();
#endif

#ifdef GPS_SERIAL_IN_USE
    gps = new GpsSerial();
#endif

#ifdef GPS_EMULATOR_IN_USE
    gps = new GpsEmulator();
#endif

    connect(gps, SIGNAL(nmeaSentenceReady(QString)),
            this, SLOT(checkMessage(QString)));
    connect(gps, SIGNAL(errorOccured(QString)),
            this, SLOT(generateErrorMessage(QString)));
}

//! destructor
GpsBase::~GpsBase()
{
    delete gps;
    delete gpsPoint;
    delete gpsData;

}

//! Generates error message and calls Qmessagebox to show it for the user.
void GpsBase::generateErrorMessage(QString error)
{
    QMessageBox::warning(0, "GPS", tr("Error occured in GPS\n%1.").arg(error));
}

//! Starts gps by sending nmea query for the gps device.
void GpsBase::startGps()
{
    gps->startGps();
}

//! Sends stop signal to gps.
void GpsBase::stopGps()
{
    gps->stopGps();
}

//! Checks the NMEA sentences for errors.
void GpsBase::checkMessage(QString nmea)
{
    if (nmea.contains(messageErrorFromGps) == true) {
        QMessageBox::warning(0, "GPS", tr("Error occured in GPS"));
    }
    else {
        readNmea(nmea);
    }
}

//! reads the NMEA sentence
void GpsBase::readNmea(QString nmea)
{
    // see if the sentence's checksum is ok
    QString checksumString;
    bool checksumOk = false;
    int indexOfAsterisk;
    int startIndex;
    int checksum = 0;
   
    // search correct start index.
    startIndex = nmea.lastIndexOf("$") + 1 ;

    indexOfAsterisk = nmea.indexOf('*', 0);

    for (int i = startIndex; i < indexOfAsterisk; i++) {
        checksum = checksum ^ nmea.at(i).toAscii();
    }

    // copy the checksum to a string
    checksumString = nmea;
    checksumString.remove(0, indexOfAsterisk + 1);
    if (checksumString.lastIndexOf(DELIM) != -1) {
        checksumString.remove(checksumString.lastIndexOf(DELIM),
                              checksumString.length());
    }

    if (checksum == checksumString.toInt(0, 16)) {
        checksumOk = true;
        // since the checksum is ok, call gpsCheckNmeaType
        nmea.remove(nmea.indexOf('*'), nmea.length()) ;
        gpsCheckNmeaType(nmea);
    }
    else {
        // the checksum is not ok, do something if you want
    }

}

//! checks the type of nmea sentence and calls the appropriate parser
void GpsBase::gpsCheckNmeaType(QString &nmea)
{

    if (nmea.contains(nmeaTypeGSV)) {
        parseGSV(nmea.remove(0, nmea.indexOf(DELIM) + 1));
    }
    else if (nmea.contains(nmeaTypeRMC)) {
        parseRMC(nmea.remove(0, nmea.indexOf(DELIM) + 1));
    }
    else if (nmea.contains(nmeaTypeGGA)) {
        parseGGA(nmea.remove(0, nmea.indexOf(DELIM) + 1));
    }
    else if (nmea.contains(nmeaTypeGSA)) {
        parseGSA(nmea.remove(0, nmea.indexOf(DELIM) + 1));
    }

}

//! parses RMC type NMEA sentences
void GpsBase::parseRMC(const QString &sentence)
{
    /* Recommended Minimum Navigation Information C
    *  1) UTC Time
    *  2) Status, V=Navigation receiver warning A=Valid
    *  3) Latitude
    *  4) N or S
    *  5) Longitude
    *  6) E or W
    *  7) Speed over ground, knots
    *  8) Track made good, degrees true
    *  9) Date, ddmmyy
    * 10) Magnetic Variation, degrees
    * 11) E or W
    * 12) FAA mode indicator (NMEA 2.3 and later)
    * 13) Checksum
    */

    // This one is used as a temporary string storage
    QString *token = new QString;

    qreal gpsLatitude = 0;
    qreal gpsLongitude = 0;

    // Parse time from GPS-sentence
    token->append(sentence.section(",", RMCTimeIndex, RMCTimeIndex));

    if (!token->isEmpty()) {

        token->truncate(hourStringLength);
        int hours = token->toInt();

        gmtOffset = gpsData->getSystemTime()->hour() - hours;

        if (gmtOffset < -12) {
            gmtOffset += 24;
        }
        else if (gmtOffset > 12) {
            gmtOffset -= 24;
        }

        token->clear();
    }

    /*
     * We'll skip the status part from GPS-sentence. Uncomment the lines below
     * if its needed
     */
    //token->append(sentence.section(DELIM, RMCStatusIndex, RMCStatusIndex));

    //if (!token->isEmpty() && *token == validIndicatorString) {
    ///* Data is valid */
    //}
    //else {
    ///* Data is invalid */
    //}
    //token->clear();

    // Parse the latitude.
    token->append(sentence.section(DELIM, RMCLatitudeIndex, RMCLatitudeIndex));

    if (!token->isEmpty()) {

        int degrees = (int)(token->toDouble() / 100);
        qreal minutes = token->toDouble() - degrees * 100;
        gpsLatitude = degrees + minutes / 60;
        token->clear();
    }
        
    // Parse N or S.
    token->append(sentence.section(DELIM, RMCNorthSouthIndex,
                                   RMCNorthSouthIndex));

    if (*token == south) {
        gpsLatitude = -1 * gpsLatitude;
    }

    token->clear();

    gpsData->setLatitude(gpsLatitude);

    // Parse the longitude.
    token->append(sentence.section(DELIM, RMCLongitudeIndex,
                                   RMCLongitudeIndex));

    if (!token->isEmpty()) {

        int degrees = (int)(token->toDouble() / 100);
        qreal minutes = token->toDouble() - degrees * 100;
        gpsLongitude = degrees + minutes / 60;
        token->clear();
    }

    // Parse E or W.
    token->append(sentence.section(DELIM, RMCEastWestIndex,
                                   RMCEastWestIndex));

    if (token == west) {
        gpsLongitude = -1 * gpsLongitude;
    }

    token->clear();

    gpsData->setLongitude(gpsLongitude);

    // Parse speed over ground, knots.
    token->append(sentence.section(DELIM, RMCSpeedIndex, RMCSpeedIndex));

    if (!token->isEmpty()) {

        qreal speed = token->toDouble(); // speed is in knots

        // transform knots to km/h and store it in gpsData
        speed = speed * knotsInKmPerHour;
        gpsData->setSpeed(speed);

        if (gpsData->getFix() > 1) {
            if (gpsData->getMaxSpeed() < speed) {
                gpsData->setMaxSpeed(speed);
            }
        }

        token->clear();
    }

    // Parse heading, degrees from true north.
    token->append(sentence.section(DELIM, RMCCourseIndex, RMCCourseIndex));

    if (!token->isEmpty()) {

        gpsData->setHeading(token->toDouble());

        token->clear();
    }

    delete token;

}

//! parses GGA type NMEA sentences
void GpsBase::parseGGA(const QString &sentence)
{
    /* GGA          Global Positioning System Fix Data
     1. Fix Time
     2. Latitude
     3. N or S
     4. Longitude
     5. E or W
     6. Fix quality
                   0 = invalid
                   1 = GPS fix (SPS)
                   2 = DGPS fix
                   3 = PPS fix
                   4 = Real Time Kinematic
                   5 = Float RTK
                   6 = estimated (dead reckoning) (2.3 feature)
                   7 = Manual input mode
                   8 = Simulation mode
     7. Number of satellites being tracked
     8. Horizontal dilution of precision
     9. Altitude, Meters, above mean sea level
     10. Alt unit (meters)
     11. Height of geoid (mean sea level) above WGS84 ellipsoid
     12. unit
     13. (empty field) time in seconds since last DGPS update
     14. (empty field) DGPS station ID number
     15. the checksum data
     */
    QString *token = new QString;

    const int GGAPositionFix = 5;
    const int GGAHorizontalDilutionOfPrecision = 7;
    const int GGAAltitude = 8;

    /* Parse Fix quality */
    token->append(sentence.section(DELIM, GGAPositionFix, GGAPositionFix));
    if (!token->isEmpty()) {
        gpsData->setFixQuality(token->toInt(0, 10));
    }
    token->clear();

    /* Parse Horizontal dilution of precision */
    token->append(sentence.section(DELIM, GGAHorizontalDilutionOfPrecision,
                                   GGAHorizontalDilutionOfPrecision)) ;
    if (!token->isEmpty()) {
        gpsData->setHorizontalDilutionOfPrecision(token->toFloat());
    }
    token->clear() ;

    /* Altitude */
    token->append(sentence.section(DELIM, GGAAltitude, GGAAltitude)) ;
    if (!token->isEmpty()) {
        gpsPoint->setAltitude(token->toInt());
    }
    else {
        gpsPoint->setAltitude(0);
    }

    token->clear();
    delete(token);
}

//! parses GSA type NMEA sentences
void GpsBase::parseGSA(const QString &sentence)
{
    /* GPS DOP and active satellites
     *  1) Auto selection of 2D or 3D fix (M = manual)
     *  2) 3D fix - values include: 1 = no fix, 2 = 2D, 3 = 3D
     *  3) PRNs of satellites used for fix
     *     (space for 12)
     *  4) PDOP (dilution of precision)
     *  5) Horizontal dilution of precision (HDOP)
     *  6) Vertical dilution of precision (VDOP)
     *  7) Checksum
     */

    const int GSA3dFix = 1;
    const int GSAFirstSatellite = 2;
    const int GSALastSatellite = 13;
    const int GSAPositionDilutionOfPrecision = 14;
    const int GSAHorizontalDilutionOfPrecision = 15;
    const int GSAVerticalDilutionOfPrecision = 16;

    // This one is used as a temporary string storage
    QString *token = new QString;

    // number of satellites
    int satCount = 0;

    // Parse 3D fix.
    token->append(sentence.section(DELIM, GSA3dFix, GSA3dFix));

    if (!token->isEmpty()) {
        gpsData->setFix(token->toInt(0, 10));
        token->clear();
    }

    // Parse number of satellites
    for (int i = GSAFirstSatellite; i <= GSALastSatellite; i++) {

        token->append(sentence.section(DELIM, i, i));

        if (!token->isEmpty()) {
            satCount++;

            gpsData->setSatellitesForFix(i, token->toInt(0, 10));
            gpsData->setSatellitesInUse(satCount);

            token->clear();
        }
    }

    // Parse PDOP
    token->append(sentence.section(DELIM,
                                   GSAPositionDilutionOfPrecision,
                                   GSAPositionDilutionOfPrecision));

    if (!token->isEmpty()) {
        gpsData->setPositionDilutionOfPrecision(token->toFloat());
        token->clear();
    }

    // Parse HDOP
    token->append(sentence.section(DELIM, GSAHorizontalDilutionOfPrecision,
                                   GSAHorizontalDilutionOfPrecision));

    if (!token->isEmpty()) {
        gpsData->setHorizontalDilutionOfPrecision(token->toFloat());
        token->clear();
    }

    // Parse VDOP
    token->append(sentence.section(DELIM, GSAVerticalDilutionOfPrecision,
                                   GSAVerticalDilutionOfPrecision));

    if (!token->isEmpty()) {
        gpsData->setVerticalDilutionOfPrecision(token->toFloat());
        token->clear();
    }

    delete token;
}

//! parses GSV type NMEA sentences
void GpsBase::parseGSV(QString &sentence)
{
    /* Must be GSV - Satellites in view
     *  1) total number of messages <=> messageFields
     *  2) message sequence number
     *  3) satellites in view
     *  4) satellite number
     *  5) elevation in degrees (0-90)
     *  6) azimuth in degrees to true north (0-359)
     *  7) SNR in dB (0-99)
     *  more satellite infos like 4)-7)
     *  n) checksum
     */

    QString *token = new QString;

    int sequenceNumber = 0;

    /*
     * All section methods have parameter 0. There wasn't any
     * better way to do it so magic variable is used.
     */

    /* Since we have no need for the information about the total number of
     * messages, we remove it from the sentence. If it is needed, uncomment
     * the lines commented with '***' */

    //token->append(sentence.section(DELIM, 0, 0));    // ***
    sentence.remove(0, sentence.indexOf(DELIM) + 1);
    //if (!token->isEmpty()) {                         // ***
    //messageFields = token->toInt(0, 10);         // ***
    //}                                                // ***

    //token->clear();                                  // ***

    /* Parse sequence number. */
    token->append(sentence.section(DELIM, 0, 0));
    sentence.remove(0, sentence.indexOf(DELIM) + 1);

    if (!token->isEmpty()) {
        sequenceNumber = token->toInt(0, 10);
    }

    token->clear();

    /*
     * gpsSatelliteDataArray is "divided" into three parts. The part that is
     * used for storing parsed data depends on the sequence number.
     * Sequence 1 uses array indices 0-3, sequence 2 uses 4-7,
     * and sequence 3 uses 8-11.
     */
    const int firstArrayIndex = (sequenceNumber - 1) * 4;

    /*
     * if we are receiving a new sequence of messages, erase the old data
     * from the array
     */
    if (sequenceNumber == 1) {

        for (int i = 0; i < maxSatellitesCount; i++) {
            gpsSatelliteDataArray[i].reset();
        }
    }

    /* Parse number of satellites in view. */
    token->append(sentence.section(DELIM, 0, 0));
    sentence.remove(0, sentence.indexOf(DELIM) + 1);

    if (!token->isEmpty()) {
        gpsData->setSatellitesInView(token->toInt(0, 10));
    }

    if (gpsData->getSatellitesInView() > maxSatellitesCount) {
        gpsData->setSatellitesInView(maxSatellitesCount);
    }

    token->clear();

    int satelliteIndex = firstArrayIndex;

    /* Loop until there are no more satellites to parse. */
    while (!sentence.isEmpty()) {

        /* Get token for Satellite Number. */
        token->append(sentence.section(DELIM, 0, 0)) ;
        sentence.remove(0, sentence.indexOf(DELIM) + 1) ;

        if (!token->isEmpty()) {
            int prn = token->toInt(0, 10);
            gpsSatelliteDataArray[satelliteIndex].setPrn(prn);
        }

        token->clear() ;

        /* Get token for elevation in degrees (0-90). */
        token->append(sentence.section(DELIM, 0, 0));
        sentence.remove(0, sentence.indexOf(DELIM) + 1);
        if (!token->isEmpty()) {
            gpsSatelliteDataArray[satelliteIndex]
            .setElevation(token->toInt(0, 10));
        }

        token->clear();

        /* Get token for azimuth in degrees to true north (0-359). */
        token->append(sentence.section(DELIM, 0, 0));
        sentence.remove(0, sentence.indexOf(DELIM) + 1);
        if (!token->isEmpty()) {
            gpsSatelliteDataArray[satelliteIndex]
            .setAzimuth(token->toInt(0, 10)) ;
        }

        token->clear();

        /* Get token for SNR. */
        token->append(sentence.section(DELIM, 0, 0));

        if (token == sentence) {
            sentence.clear();
        }
        else {
            sentence.remove(0, sentence.indexOf(DELIM) + 1);
        }

        if (!token->isEmpty()) {
            gpsSatelliteDataArray[satelliteIndex].setSnr(token->toInt(0, 10));
            token->clear();
        }

        satelliteIndex++;
    }

    delete(token);
}


//! Returns the used gpsData object
GpsData &GpsBase::getGpsData()
{
    return *gpsData;
}


//! Returns the used gpsPoint object
Point &GpsBase::getGpsPoint()
{
    return *gpsPoint ;
}
