/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QMessageBox>
#include "editpoidialog.h"
#include "poidatabase.h"
#include "poicategoriesdialog.h"

//! Constructor
EditPoiDialog::EditPoiDialog(QtMapper &mainWindow, QWidget *parent)
    : QDialog(parent), mainWindow(mainWindow),
        minLatitude(-85.05113),
        maxLatitude(85.05113),
        minLongitude(-180.00000),
        maxLongitude(180.00000),
        maxDecimals(6)
{
    setupUi(this);
    this->lineEditLatitude->setValidator(new QDoubleValidator(minLatitude, maxLatitude,
                          maxDecimals, lineEditLatitude)) ;
    this->lineEditLongitude->setValidator(new QDoubleValidator(minLongitude, maxLongitude,
                          maxDecimals, lineEditLongitude)) ;

    connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    connect(pushButtonEditCategories, SIGNAL(clicked()),
            this, SLOT(showPoiCategoriesDialog()));
    connect(lineEditLabel, SIGNAL(textChanged(const QString &)),
            this, SLOT(enableOkButton(const QString &)));
    poiDb = mainWindow.getPoiDb();
}

//! Checks validation of the lat/lon input fields.
void EditPoiDialog::accept()
{
    if((this->lineEditLatitude->hasAcceptableInput()) &&
       this->lineEditLongitude->hasAcceptableInput()) {
            QDialog::accept() ;
    }
    else {
        QMessageBox::information(this, tr("Information"),
                   tr("Latitude and/or Longitude is in wrong format.\n"
                      "Latitude range: %1 - +%2\n"
                      "Longitude range: %3 - +%4")
                   .arg(minLatitude)
                   .arg(maxLatitude)
                   .arg(minLongitude)
                   .arg(maxLongitude),
                   QMessageBox::Ok) ;
    }
}

//! Sets labels of POI categories to the combobox
void EditPoiDialog::setData()
{
    comboBoxCategory->clear();

    for (qint32 i = 0; i < categories.count(); i++) {
        comboBoxCategory->addItem(categories.at(i).getLabel());
    }
    comboBoxCategory->setCurrentIndex(selectedIndex);
}

//! Shows Dialog POI categories
void EditPoiDialog::showPoiCategoriesDialog()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }    
    
	PoiCategoriesDialog pc(mainWindow);
	pc.pushButtonDelete->hide();
	selectedIndex = comboBoxCategory->currentIndex();
	
        if (pc.exec() == QDialog::Accepted) {
                categories.clear();
                poiDb->getPoiCategories(categories);
                setData();
        }
}

//! Destructor
EditPoiDialog::~EditPoiDialog()
{
}

//! Enable or disable the ok button if a text field is empty.
void EditPoiDialog::enableOkButton(const QString &text)
{
    if (text.isEmpty()) {
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
    }
    else {
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(true);
    }
}
