/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef __DOWNLOADEDPOISDIALOG_H__
#define __DOWNLOADEDPOISDIALOG_H__

#include "ui_downloadedpois.h"
#include "QtMapper.h"
#include <QVector>

class Poi;
class PoiCategory;
class QStandardItemModel;

/*!
* Class is for showing and manipulating downloaded POIs before saving to database.
*/
class DownloadedPoisDialog : public QDialog, public Ui::DownloadedPoisDialog
{
    Q_OBJECT

public:
	/*!
	* Constructor.
	* \param pois Vector of pois.
	* \param latitude Latitude of current location.
	* \param longitude Longitude of current location.
	* \param mainWindow Reference to QtMapper.
	* \param parent The parent widget (default = no parent).
	*/
    DownloadedPoisDialog(QVector<Poi> &pois,
                         qreal latitude,
                         qreal longitude,
                         QtMapper &mainWindow, QWidget *parent = 0);
    /*!
    * Destructor.
    */
    ~DownloadedPoisDialog();

private:

    //! Describes is there all pois selected in the list.
    bool selectAll ;

    //! Earth radius in kilometers
    const int earthRadiusInKilometres;

    //! width of 1. column of view.
    const int column1Width;
    //! width of 2. column of view.
    const int column2Width;
    //! width of 3. column of view.
    const int column3Width;
    //! width of 4. column of view.
    const int column4Width;
    //! width of 5. column of view.
    const int column5Width;

    //! Label of default category to downloaded POIs.
    const QString defaultCategoryLabel;
    //! Description of default category to downloaded POIs.
    const QString defaultCategoryDescription;
    //! Enabled value of default category to downloaded POIs.
    const quint32 defaultCategoryEnabled;

    //! Container for POI category data
    QVector<PoiCategory> categories;

    //! Container for POI data
    QVector<Poi> pois;

    /*!
     * Container for POI data for checking is there already similar POIs
     * in database
     */
    QVector<Poi> allPois;

    //! Model of POI data
    QStandardItemModel * model;

    //! Latitude of current location
    qreal latitude;

    //! Longitude of current location
    qreal longitude;
   	
    //! Reference to main window.
    QtMapper &mainWindow;

    //! Pointer to POI database.
    PoiDatabase *poiDb;

    /*!
    * Calculates distance from current location to POI's location.
    * \param startLatitude Latitude of starting point.
    * \param startLongitude Longitude of starting point.
    * \param endLatitude Latitude of ending point.
    * \param endLongitude Longitude of ending point.
    * \return distance or POI from current point.
    */
    qreal distanceBetweenCoordinates(qreal startLatitude, qreal startLongitude,
                                     qreal endLatitude, qreal endLongitude);
    /*!
    * Calculates bearing from current location to POI's location.
    * \param startLatitude Latitude of starting point.
    * \param startLongitude Longitude of starting point.
    * \param endLatitude Latitude of ending point.
    * \param endLongitude Longitude of ending point.
    * \return Bearing or POI from current point.
    */
    qreal bearing(qreal startLatitude, qreal startLongitude,
                  qreal endLatitude, qreal endLongitude);

private slots:

    /*!
       * Sets the data of POIs to view.
       */
    void setData();
    /*!
       * Saves POIs to database.
       */
    void savePois();
    /*!
    * Cleans POI data.
    */
    void cleanData();

    /*!
     * Marks all pois to selected or unselected in the list.
     */
    void selectPois() ;
};

#endif
