/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QAbstractItemView>
#include <QDebug>
#include <QMessageBox>
#include <QStandardItem>
#include <QStandardItemModel>
#include <math.h>
#include "definitions.h"
#include "poidatabase.h"
#include "poi.h"
#include "poicategory.h"
#include "downloadedpoisdialog.h"

//! Constructor
DownloadedPoisDialog::DownloadedPoisDialog(QVector<Poi> &pois,
                                           qreal latitude,
                                           qreal longitude,
                                           QtMapper &mainWindow,
                                           QWidget *parent) :
    QDialog(parent),
    selectAll(false),
    earthRadiusInKilometres(6371),
    column1Width(50),
    column2Width(150),
    column3Width(100),
    column4Width(100),
    column5Width(150),
    defaultCategoryLabel(tr("Downloaded")),
    defaultCategoryDescription(tr("Downloaded POIs")),
    defaultCategoryEnabled(1),
    pois(pois),
    latitude(latitude),
    longitude(longitude),
    mainWindow(mainWindow)

{
    setupUi(this);

    model = new QStandardItemModel(this);

    connect(pushButtonCancel, SIGNAL(clicked()), this, SLOT(reject()));
    connect(pushButtonSave, SIGNAL(clicked()), this, SLOT(savePois()));
    connect(pushButtonSelectAll, SIGNAL(clicked()), this, SLOT(selectPois())) ;
	poiDb = mainWindow.getPoiDb();
    cleanData();
    setData();
}

void DownloadedPoisDialog::selectPois()
{
    if (pois.count() > 0) {
        if (!selectAll) {
               for (int i=0; i < pois.count() ; ++i) {
                    model->item(i)->setCheckState(Qt::Checked) ;
               }
               pushButtonSelectAll->setText(tr("&Unselect All")) ;
               selectAll = true ;
           }
            else  if (selectAll) {
                for (int i=0; i < pois.count() ; ++i) {
                    model->item(i)->setCheckState(Qt::Unchecked) ;
               }
               pushButtonSelectAll->setText(tr("Select &All")) ;
               selectAll = false ;
           }
        }
}

//! Cleans POI data
void DownloadedPoisDialog::cleanData()
{
    for (qint32 i = 0; i < pois.count(); i++) {
        QString description(pois.at(i).getDescription());
        description.replace(QString("<br/>"), QString(", "));
        pois[i].setDescription(description);
    }
}

//!Sets the POI data to view
void DownloadedPoisDialog::setData()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }

	poiDb->getPoiCategories(categories);
	
	bool found = false;	
		
	for (qint32 i = 0; i < categories.count(); i++) {
	
		if(categories.at(i).getLabel() == defaultCategoryLabel) {
			
			for(qint32 j = 0; j < pois.count(); j++) {
				pois[j].setCategory(categories.at(i));		
            }

			found = true;
		} 
	}
	
	if (found == false) {
		PoiCategory category;
		category.setLabel(defaultCategoryLabel);
		category.setDescription(defaultCategoryDescription);
		category.setEnabled(defaultCategoryEnabled);
		poiDb->insertPoiCategory(category);
		category.setCategoryId(poiDb->getCategoryId(defaultCategoryLabel));
	 
		for(qint32 i = 0; i < pois.count(); i++) {
			pois[i].setCategory(category);		
		}
	}
				
	QStandardItem *item0 = new QStandardItem(QString(tr("Selected")));
	QStandardItem *item1 = new QStandardItem(QString(tr("Category")));
	QStandardItem *item2 = new QStandardItem(QString (tr("Distance (km)")));
	QStandardItem *item3 = new QStandardItem(QString (tr("Bearing")));
	QStandardItem *item4 = new QStandardItem(QString (tr("Label")));
	
	model->setHorizontalHeaderItem(0, item0);
	model->setHorizontalHeaderItem(2, item1);
	model->setHorizontalHeaderItem(3, item2);
	model->setHorizontalHeaderItem(4, item3);
	model->setHorizontalHeaderItem(1, item4);
		
	for (qint32 i = 0; i < pois.count(); i++) {
		QStandardItem *item0 = new QStandardItem(QString(""));
		item0->setCheckable(true);
                item0->setCheckState(Qt::Unchecked);
       
        QStandardItem *item1 =
            new QStandardItem(QString(pois.at(i).getCategory().getLabel()));

        QStandardItem *item2 = new QStandardItem(QString("%1")
                .arg(distanceBetweenCoordinates(latitude, longitude,
                                                pois.at(i).getLatitude(),
                                                pois.at(i).getLongitude())));

        QStandardItem *item3 = new QStandardItem(QString("%1")
                .arg(bearing(latitude, longitude,
                             pois.at(i).getLatitude(),
                             pois.at(i).getLongitude())));

        QStandardItem *item4 =
            new QStandardItem(QString(pois.at(i).getLabel()));

        item1->setEditable(false);
        item2->setEditable(false);
        item3->setEditable(false);
        item4->setEditable(false);

        model->setItem(i, 0, item0);
        model->setItem(i, 2, item1);
        model->setItem(i, 3, item2);
        model->setItem(i, 4, item3);
        model->setItem(i, 1, item4);
    }

    tableView->setModel(model);
    tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    tableView->setColumnWidth(0, column1Width);
    tableView->setColumnWidth(2, column2Width);
    tableView->setColumnWidth(3, column3Width);
    tableView->setColumnWidth(4, column4Width);
    tableView->setColumnWidth(1, column5Width);
}

//! Calculates distance from current location to POI's location
qreal DownloadedPoisDialog::distanceBetweenCoordinates(qreal startLatitude,
                                                       qreal startLongitude,
                                                       qreal endLatitude,
                                                       qreal endLongitude)
{
    // convert degrees to radians
    startLatitude = startLatitude * pi / 180;
    startLongitude = startLongitude * pi / 180;
    endLatitude = endLatitude * pi / 180;
    endLongitude = endLongitude * pi / 180;

    qreal latitudeDifference = endLatitude - startLatitude;
    qreal longitudeDifference = endLongitude - startLongitude;

    qreal temp = pow(sin(latitudeDifference / 2), 2) +
                 cos(startLatitude) * cos(endLatitude) *
                 pow(sin(longitudeDifference / 2), 2);

    qreal distance = earthRadiusInKilometres * 2 * atan2(sqrt(temp),
                     sqrt(1 - temp));


    return distance;
}

//! Calculates bearing from current location to POI's location
qreal DownloadedPoisDialog::bearing(qreal startLatitude,
                                    qreal startLongitude,
                                    qreal endLatitude,
                                    qreal endLongitude)
{
    // convert degrees to radians
    startLatitude = startLatitude * pi / 180;
    startLongitude = startLongitude * pi / 180;
    endLatitude = endLatitude * pi / 180;
    endLongitude = endLongitude * pi / 180;

    qreal longitudeDifference = endLongitude - startLongitude;

    // calculate the bearing
    qreal y = sin(longitudeDifference) * cos(endLatitude);
    qreal x = cos(startLatitude) * sin(endLatitude) - sin(startLatitude) *
              cos(endLatitude) * cos(longitudeDifference);
    qreal bearing = atan2(y, x);

    // convert bearing to degrees
    bearing = bearing * 180 / pi;

    return bearing;
}
//! Saves POIs to database
void DownloadedPoisDialog::savePois()
{
    if (pois.count() == 0) {
        this->accept();
    }

    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }

    poiDb->getAllPois(allPois);

    qint32 similars = 0;
    bool similar = false;

    for (qint32 i = 0; i < pois.count(); i++) {
        if (model->item(i)->checkState() == Qt::Checked) {
            if (allPois.count() == 0) {
                poiDb->insertPoi(pois.at(i));
            }
            else {
                for (qint32 j = 0; j < allPois.count(); j++) {
                    if (allPois.at(j).isSimilar(pois.at(i))) {
                        similars++;
                        similar = true;
                    }
                }

                if (similar == false) {
                    poiDb->insertPoi(pois.at(i));
                }
            }
            similar = false;
        }
    }

    // Notification of similar POIs
    if (similars > 0) {
        QMessageBox mb(this);
        mb.setStandardButtons(QMessageBox::Ok);
        mb.setWindowTitle(tr("NOTIFICATION"));
        QString text(QString(tr("There is %1 similar POIs already!"))
                     .arg(similars));
        mb.setText(text);
        mb.exec();
    }
    
    this->accept();
}

//! Destructor
DownloadedPoisDialog::~DownloadedPoisDialog()
{
}
