/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef DIAGRAMS_H
#define DIAGRAMS_H

#include <QFont>
//#include <QObject>
#include <QPoint>
#include <QRect>
#include <QWidget>
#include <QPainter>
#include <QString>
class Bar ;
class QPaintEvent;
class Plot ;
class SatelliteDrawing ;

/*!
 * Class handles bar and satellites' position drawings.
 */
class Diagrams : public QWidget
{
    Q_OBJECT

public:
    /**
     * Constructor
     * Takes four parameters.
     * @param 1 is count how many bars should create
     * for the satellites' signal noise ratio.
     * @param 2 is int type variable for x -position.
     * @param 3 is int type variable for y -position.
     * @param 4 is int type variable for width of needed drawing area.
     * @param 5 is int type variable for height of needed drawing area.
     */
    Diagrams(int countOfBars, int x, int y, int width, int height) ;

    /**
     * Overloaded Constructor
     * Takes two parameters.
     * @param 1 is count how many bars should create
     * for the satellites' signal noise ratio.
     * @param 2 is the QRect type variable which contains same parameters
     * than the previous constructor for the same use.
     */
    Diagrams(int countOfBars, QRect* rect) ;

    /**
    * Overloaded Constructor
    * Takes one parameter.
    * @param 1 is count how many bars should create
    * for the satellites' signal noise ratio.
    * It takes starting measures from the widgets' geometry.
    */
    Diagrams(int countOfBars) ;

    /**
     * Deletes all satelliteDrawing objects and Bar objects.
     */
    ~Diagrams() ;

    /**
     * Sets maximum value for the signal noise ratio. Default value is zero.
     * @param is the int type of variable.
     */
    void setMaximumValue(int maxValue) ;

    /**
     * return maximum value. If not defined returns zero.
     */
    int getMaximumValue() ;


    /**
     * Sets minimum value for the signal noise ratio. Default value is zero.
     * @param is the int type of variable.
     */
    void setMinimumValue(int minValue) ;

    /**
     * return minimum value. If not defined returns zero.
     */
    int getMinimumValue() ;

    /**
     * Sets azimuth value for the wanted satellite drawing.
     * @param1 is ID of the wanted satellite.
     * @param2 is azimuth value.
     */
    void setAzimuth(int id, int azimuth) ;

    /**
     * Sets elevation value for the wanted satellite drawing.
     * @param1 is ID of the wanted satellite.
     * @param2 is elevation value.
     */
    void setElevation(int id, int elevation) ;

    /**
     * Via this function new signal noise ratio is updated. Function calculates
     * percentage part of maximumvalue and sends it to correct bar which is
     * pointed by the ID.
     * @param 1 int type variable for bar id.
     * @param 2 int type variable for value of signal noise ratio.
     */
    void setValue(int id, int value) ;

    /**
     * @param is ID for the wanted bar or satellite.
     * return Satellite's ID number as QString.
     */
    QString getId(int idNumber) ;

    /**
     * return count of bars. Describes satellites count also.
     */
    int getCountOfBars() ;

    /**
     * Replace Id.
     * @param1 is ID of the satellite and bar.
     * @param2 is QString type for new Id.
     */
    void setId(int idNumber, QString &id) ;

protected:
    /**
     * This is virtual protected in parent class. Have to implement when
     * custom drawing used.
     * @param is event type.
     */
    void paintEvent(QPaintEvent* event) ;

private:
    /**
     * Creates bar and satellite objects. Called from all constructors.
     */
    void createBars() ;

    /**
     * Calculates drawing area for the each bar. Called always when
     * paintevent is called.
     */
    void updateBars() ;

    /**
     * Variables initialized here to avoid repetition in constructors.
     * Called from all constructors.
     */
    void init() ;

    /**
     * Takes measures from the QWidgets geometry and calculates new measures
     * for the drawing areas.
     */
    void measure() ;

    //! Satellite's SNR bars
    Bar *bars[12];

    //! Plot object draws the background image for satellite images
    Plot *plot;

    //! Satellite image objects
    SatelliteDrawing *sat[12];

    //! Maximum value of SNR
    int maxValue;

    //! Minimum value of SNR
    int minValue;

    //! Number of SNR bars
    int countOfBars;

    //! Width of drawing area
    int width;

    //! Height of drawing area
    int height;

    //! x-coordinate of drawing area
    int x;

    //! y-coordinate of drawing area
    int y;

    //! Used for displaying the SNR values on screen
    int value;

    //! Left side of the drawing area, for the SNR bars
    QRect drawAreaLeft;

    //! Right side of the drawing area, for the satellite images
    QRect drawAreaRight;

    //! Rectangle for the SNR bars
    QRect barRect;

    //! Font for header texts
    QFont headerFont;

    //! Coordinate for the SNR bar's header text
    QPoint headerPlaceLeft;

    //! Coordinate for the satellite images' header text
    QPoint headerPlaceRight;

    //! QPainter object for the SNR bar area
    QPainter painterLeft;

    //! QPainter object for the satellite drawing area
    QPainter painterRight;

};

#endif
