/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "diagrams.h"
#include "bar.h"
#include "plot.h"
#include "satelliteDrawing.h"

#include <QPaintEvent>
#include <QPoint>


//! Constructor, parameters as ints
Diagrams::Diagrams(int countOfBars, int x, int y, int width, int height)
{
    this->countOfBars = countOfBars;
    this->x = x;
    this->y = y;
    this->height = height;
    this->width = width;
    init();
    measure();
    this->createBars();
}

/*!
 * Overloaded constructor, previous constructor's last four parameters
 * inside QRect
 */
Diagrams::Diagrams(int countOfBars, QRect* rect)
{
    this->countOfBars = countOfBars;
    this->x = rect->x();
    this->y = rect->y();
    this->height = rect->height();
    this->width = rect->width();
    init();
    measure();
    this->createBars();
}

//! Overloaded Constructor, takes parameters from the widget's geometry
Diagrams::Diagrams(int countOfBars)
{
    this->countOfBars = countOfBars;
    measure();
    this->createBars();
}

//! Variables initialized here to avoid repetition in constructors
void Diagrams::init()
{
    this->maxValue = 0;
    this->minValue = 0;

    // Header Texts.
    headerFont = QFont("Arial", 8);
    headerFont.setBold(true);
    headerFont.setUnderline(true);
}

//! Deletes all satelliteDrawing objects and Bar objects
Diagrams::~Diagrams()
{
    for (int i = 0; i < this->countOfBars; i++) {
        delete bars[i];
        delete sat[i];
    }
}

/*!
 * Takes measures from the QWidget's geometry and calculates new measures
 * for the drawing areas.
 */
void Diagrams::measure()
{
    
    int geometryHeight = this->geometry().height();
    int geometryWidth = this->geometry().width();
    int geometryX = this->geometry().x();
    int geometryY = this->geometry().y();
    
    QFontMetrics headerMetrics(headerFont);
    int headerHeight = headerMetrics.height();
    
    headerPlaceLeft = QPoint(0, headerHeight);
    headerPlaceRight = QPoint(geometryWidth / 2, headerHeight);

    this->x = geometryX;
    this->y = geometryY + headerHeight;
    this->width = geometryWidth;
    this->height = geometryHeight - headerHeight;

    // measure drawing areas.
    drawAreaLeft.setX(geometryX);
    drawAreaLeft.setY(geometryY + headerHeight);
    drawAreaLeft.setBottom(geometryHeight - headerHeight);
    drawAreaLeft.setWidth(geometryWidth / 2);

    drawAreaRight.setX(geometryWidth / 2);
    drawAreaRight.setY(geometryY + headerHeight);
    drawAreaRight.setBottom(geometryHeight - headerHeight);
    drawAreaRight.setWidth(geometryWidth / 2);

    // Make bars height 70% of the drawarea.
    barRect.setX(geometryX);
    barRect.setY(geometryY + headerHeight);
    barRect.setBottom(drawAreaLeft.bottom());
    barRect.setWidth(drawAreaLeft.width() / countOfBars);
}

//! Sets maximum value for the signal noise ratio
void Diagrams::setMaximumValue(int maxValue)
{
    this->maxValue = maxValue;
}

//! return maximum value
int Diagrams::getMaximumValue()
{
    return this->maxValue;
}

//! Sets minimum value for the signal noise ratio
void Diagrams::setMinimumValue(int minValue)
{
    this->minValue = minValue;
}

//! return minimum value
int Diagrams::getMinimumValue()
{
    return this->minValue;
}

//! Via this function new signal noise ratio is updated
void Diagrams::setValue(int id, int value)
{
    if (value > maxValue) {
        value = maxValue;
    }
    else if (value < minValue) {
        value = minValue;
    }
    if (id <= countOfBars || id >= 0) {
        bars[id]->setNewValue((int)(((qreal)value / (qreal)maxValue)*100),
                              value);
    }
}

//! return count of bars. Describes satellites count also.
int Diagrams::getCountOfBars()
{
    return this->countOfBars;
}

/*!
 * Calculates drawing area for the each bar. Called always when paintevent
 * is called.
 */
void Diagrams::updateBars()
{
    int xpos = drawAreaLeft.x();
    for (int i = 0; i < countOfBars;i++) {
        bars[i]->setXpos(xpos);
        xpos += drawAreaLeft.width() / countOfBars;
    }
}

//! Creates bar and satellite objects. Called from all constructors.
void Diagrams::createBars()
{
    int xpos = 0;
    for (int i = 0; i < countOfBars;i++) {
        bars[i] = new Bar(barRect);
        xpos += drawAreaLeft.width() / countOfBars;
        sat[i] = new SatelliteDrawing(drawAreaRight);
    }
    plot = new Plot(drawAreaRight);
    this->updateBars();
}

/*!
 * This is virtual protected in parent class. Have to implement when
 * custom drawing is used.
 */
void Diagrams::paintEvent(QPaintEvent* event)
{
    this->measure();
    if (event->type() == QEvent::Paint) {

        //left side of dialog
        QString headerLeft(tr("Satellites' SNRs"));
        QString headerRight(tr("Satellites' Locations"));

        painterLeft.begin(this);
        painterLeft.setFont(headerFont);
        painterLeft.drawText(headerPlaceLeft, headerLeft);

        for (int i = 0; i < countOfBars;i++) {
            bars[i]->updateArea(barRect);
            this->updateBars();
            bars[i]->update(&painterLeft);
        }
        painterLeft.end();

        //right side of dialog
        painterRight.begin(this);

        plot->updateArea(drawAreaRight);
        plot->update(&painterRight);
        painterRight.setFont(headerFont);
        painterRight.setPen(Qt::black);
        painterRight.drawText(headerPlaceRight, headerRight);

        for (int i = 0;i < countOfBars;i++) {
            sat[i]->updateArea(plot->getDrawArea());
            sat[i]->update(&painterRight);
        }

        painterRight.end();
    }
}

//! Sets elevation value for the wanted satellite drawing.
void Diagrams::setElevation(int id, int elevation)
{
    sat[id]->setElevation(elevation);
}

//! Sets azimuth value for the wanted satellite drawing.
void Diagrams::setAzimuth(int id, int azimuth)
{
    sat[id]->setAzimuth(azimuth);
}

//! Replace Id
void Diagrams::setId(int idNumber, QString &id)
{
    bars[idNumber]->setId(id);
    sat[idNumber]->setId(id);
}

//! return Satellite's ID number as QString
QString Diagrams::getId(int idNumber)
{
    return bars[idNumber]->getId();
}
