/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "browsepoisdialog.h"
#include "poilistdialog.h"
#include "poidatabase.h"
#include <QDebug>
#include <QMessageBox>
#include <QButtonGroup>

//! Constructor
BrowsePoisDialog::BrowsePoisDialog(QtMapper &mainWindow,
                                   qreal latitude,
                                   qreal longitude,
                                   qint32 limit,
                                   bool validGpsDataAvailable,
                                   QWidget *parent) :
    QDialog(parent),
    mainWindow(mainWindow),
    latitude(latitude),
    longitude(longitude),
    limit(limit),
    buttonTextShow(tr("&Show...")),
    buttonTextSearch(tr("S&earch..."))
{
    setupUi(this);
    connect(pushButtonBrowse, SIGNAL(clicked()), this, SLOT(showDialog()));
    connect(pushButtonClose, SIGNAL(clicked()), this, SLOT(close()));

	poiDb = mainWindow.getPoiDb();
    
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
	poiDb->getPoiCategories(categories);
	
	comboBoxCategory->addItem(QString(tr("All")));		
			
	for (qint32 i = 0; i < categories.count(); i++) {
		comboBoxCategory->addItem(categories.at(i).getLabel());
		comboBoxCategory->setCurrentIndex(categories.at(i).getCategoryId());
	}
		
    comboBoxCategory->setCurrentIndex(0);

    comboBoxCategory->setEnabled(false);
	
	QButtonGroup* group = new QButtonGroup(this);

    connect(group, SIGNAL(buttonClicked(QAbstractButton *)),
            this, SLOT(radioButtonSelectionChanged(QAbstractButton *)));
            
    radioButtonUseGpsLocation->setChecked(false);
    radioButtonUseSearch->setChecked(false);
    radioButtonShowAll->setChecked(true);
    group->addButton(radioButtonUseGpsLocation);
    group->addButton(radioButtonUseSearch);
    group->addButton(radioButtonShowAll);

    if (!validGpsDataAvailable) {
        radioButtonUseGpsLocation->setEnabled(false);
    }
    this->validGpsDataAvailable = validGpsDataAvailable;
}


//! Checks which option is selected and calls appropriate function.
void BrowsePoisDialog::showDialog()
{
    if (radioButtonUseGpsLocation->isChecked() == true) {
        selectedOption = 0;
        showPoiListDialogByGpsLocation();
    } 
    else if (radioButtonUseSearch->isChecked() == true) {
        selectedOption = 1;
        showPoiListDialogBySearch();
    } 
    else {
        selectedOption = 2;
        showPoiListDialogAll();
    }
}


//! Shows dialog POI List when option use Gps location is selected.
void BrowsePoisDialog::showPoiListDialogByGpsLocation()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
	qint32 i = comboBoxCategory->currentIndex() - 1;
	
	pois = new QVector<Poi> ;
   
	//Get pois from certain category
	if (i >= 0 ) {
        poiDb->getPois(*pois, latitude, longitude, limit,
                      categories.at(i).getCategoryId());
                      
        if (pois->count() > 0) {

            PoiListDialog pl(selectedOption, mainWindow, latitude, longitude,
                             limit, categories.at(i).getCategoryId(),
                             validGpsDataAvailable, this);
            pl.exec();

        } 
        else {

            QMessageBox mb(this);
            mb.setStandardButtons(QMessageBox::Ok);
            mb.setWindowTitle(tr("No POIs found"));
            mb.setText(QString(tr("There are no POIs in that category.")));
            mb.exec();
        }
        //Get pois from all categories
    } 
    else {
        PoiListDialog pl(selectedOption, mainWindow, latitude, longitude,
                         limit, i, validGpsDataAvailable, this);
		pl.exec();
	}
		
	delete pois;
}


//! Shows dialog POI List when option use Search is selected.
void BrowsePoisDialog::showPoiListDialogBySearch()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }    
    
	pois = new QVector<Poi> ;
	
	poiDb->getPoisByKeyWord(*pois,lineEditKeyWord->text());
			
	if (pois->count() > 0 ) {
	
        PoiListDialog pl(selectedOption, lineEditKeyWord->text(), mainWindow,
                         latitude, longitude, validGpsDataAvailable,this);
        pl.exec();

    } 
    else {

        QMessageBox mb(this);
        mb.setStandardButtons(QMessageBox::Ok);
        mb.setWindowTitle(tr("No POIs found"));
        mb.setText(QString(tr("No POIs found with keyword \"%1\".")
                           .arg(lineEditKeyWord->text())));
        mb.exec();
    }

    delete pois;
}


//! Shows dialog POI List when option show all is selected.
void BrowsePoisDialog::showPoiListDialogAll()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
	pois = new QVector<Poi> ;
	
	poiDb->getAllPois(*pois);
	
    PoiListDialog pl(selectedOption, mainWindow, latitude, longitude,
                     validGpsDataAvailable, this);
    pl.exec();

    delete pois;
}


//! Destructor
BrowsePoisDialog::~BrowsePoisDialog()
{
}

//! Changes the button text when radio button selection changes.
void BrowsePoisDialog::radioButtonSelectionChanged(QAbstractButton *button)
{
    if (button == radioButtonUseGpsLocation) {
        pushButtonBrowse->setText(buttonTextShow);
        comboBoxCategory->setEnabled(true);
    }
    else if (button == radioButtonShowAll) {
        pushButtonBrowse->setText(buttonTextShow);
        comboBoxCategory->setEnabled(false);
    }
    else if (button == radioButtonUseSearch) {
        pushButtonBrowse->setText(buttonTextSearch);
        comboBoxCategory->setEnabled(false);
    }
}
