/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef BAR_H
#define BAR_H

#include <QFont>
#include <QLinearGradient>
#include <QPen>
#include <QPoint>
#include <QRect>

class QPainter;

/**
 * This class manages one bar of the diagram.
 */
class Bar
{
public:

    /**
     * Constructor.
     * @param1 int type for the x -position.
     * @param2 int type for the y -position.
     * @param3 int type for the width.
     * @param4 int type for the height.
     */
    Bar(int x, int y, int width, int height);

    /**
     * Same as the previous constructor but information is wrapped in QRect.
     * @param is QRect type.
     */
    Bar(QRect &rect);

    //! Destructor
    ~Bar() ;

    /**
     * Sets new x position.
     * @param is int type.
     */
    void setXpos(int x) ;

    /**
     * return x position.
     */
    int getXpos() ;


    /**
     * Sets new y position.
     * @param is int type.
     */
    void setYpos(int y) ;

    /**
     * return y position.
     */
    int getYpos() ;


    /**
     * Set the height of the bar
     * @param is int type.
     */
    void setHeight(int height) ;

    /**
     * return height of the bar
     */
    int getHeight() ;


    /**
     * Set the width of the bar
     * @param is int type.
     */
    void setWidth(int width) ;

    /**
     * return width of the bar
     */
    int getWidth() ;


    /**
     * Updates a new value for the bar and calculates right value for the bar.
     * @param1 percentage value.
     * @param2 is original value for the text use.
     */
    void setNewValue(int value, int displayedValue) ;


    /**
     * Called by paintevent. Draws bar and defines color and height for the
     * bar by value.
     * @param is QPainter type. Needed for the drawing.
     */
    void update(QPainter * painter) ;

    /**
     * Updates drawing area by the given parameters wrapped in the QRect.
     * @param is QRect type.
     */
    void updateArea(QRect &rect) ;

    /**
     * Same as the previous but given parameters are given separate.
     * @param is QRect type.
     */
    void updateArea(int x, int y, int width, int height);


    /**
     * Sets new ID name for the bar.
     * @param is QString type.
     */
    void setId(QString &id) ;

    /**
     * return bar's ID as QString.
     */
    QString getId() ;


private:

    /**
     * This function updates correct value ratio. This is needed when drawing
     * area size is changed.
     */
    void updateValue();

    /**
     * Measures bar by the given parameters. Always called when paintevent
     * is called.
     */
    void measure();

    //! ID string of the satellite
    QString id;

    //! QPen for drawing the bar's outline
    QPen outline;

    //! Qpen for drawing the letters
    QPen letters;

    //! QFont for the letters
    QFont font;

    //! Rectangle object for the bar
    QRect barRect;

    //! Coordinates for satellites ID number
    QPoint idTextPoints;

    //! Coordinates for the value text
    QPoint valueTextPoints;

    /*!
     * Coordinates for the top point of bar, needed for drawing bar's
     * coloured insides
     */
    QPoint colorPointTop;

    /*!
     * Coordinates for the bottom point of bar, needed for drawing bar's
     * coloured inside
     */
    QPoint colorPointBottom;

    //! Height, including bar and displayed texts
    int height;

    //! Width of drawing
    int width;

    //! on-screen x-coordinate of the bar
    int x;

    //! on-screen y-coordinate of the bar
    int y;

    //! Value displayed on the bar
    int displayedValue;

    //! Modified value for drawing the bar
    int value;

    //! Reference value for bar's size
    int origValue;

    //! Height of the bar
    int barHeight;

    //! Width of the bar
    int barWidth;

    //! Bar's maximum value
    int barMax;

    //! Bar's minumum value
    int barMin;

};

#endif
