/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include "bar.h"

#include <QFontMetrics>
#include <QPainter>


//! Constructor
Bar::Bar(int x, int y, int width, int height)
{
    this->width = width ;
    this->height = height ;
    this->x = x ;
    this->y = y ;
    this->value = 0 ;
    measure() ;
}

//! Constructor, with parameters given as a QRect object
Bar::Bar(QRect &rect)
{
    this->width = rect.width() ;
    this->height = rect.height() ;
    this->x = rect.x() ;
    this->y = rect.y() ;
    this->value = 0 ;
    measure() ;
}

//! Destructor
Bar::~Bar()
{
}

/*!
 * Measures bar by the given parameters. Always called when paintevent
 * is called.
 */
void Bar::measure()
{
    font.setFamily("Arial") ;

    // leave "frames" around of text.
    font.setPointSize((int)((qreal)width * 0.5));

    // leave space between other bars.
    this->width = (int)((qreal)this->width * 0.8);
    QFontMetrics fontMetrics(font);

    valueTextPoints.setX(x);
    valueTextPoints.setY(this->y + fontMetrics.height());

    idTextPoints.setX(this->x) ;
    idTextPoints.setY(this->y + this->height) ;

    barRect.setX(this->x) ;

    barMax = (idTextPoints.y() - fontMetrics.height()) - valueTextPoints.y() ;
    barMin = valueTextPoints.y() ;

    updateValue() ;

    barRect.setBottom(idTextPoints.y() - fontMetrics.height()) ;
    barRect.setY(barRect.bottom() - value) ;
    barRect.setWidth(this->width) ;

    colorPointTop.setX(barRect.x()) ;
    colorPointTop.setY(barRect.y()) ;
    colorPointBottom.setX(barRect.x()) ;
    colorPointBottom.setY(barRect.bottom()) ;
}

//! Updates drawing area, parameters given as separate ints
void Bar::updateArea(int x, int y, int width, int height)
{
    this->width = width ;
    this->height = height ;
    this->x = x ;
    this->y = y ;
    measure() ;
}

//! Updates drawing area, parameters given as a QRect object
void Bar::updateArea(QRect &rect)
{
    this->x = rect.x() ;
    this->y = rect.y() ;
    this->width = rect.width() ;
    this->height = rect.height() ;
    measure() ;
}

//! Sets new ID name for the bar
void Bar::setId(QString &id)
{
    this->id = id ;
}

//! return bar's ID as QString.
QString Bar::getId()
{
    return this->id ;
}

/*!
 * Called by paintevent. Draws bar and defines color and height for the bar
 * by value.
 */
void Bar::update(QPainter * painter)
{
    measure() ;
    QLinearGradient rectColor ;
    rectColor.setStart(colorPointTop) ;
    rectColor.setFinalStop(colorPointBottom);
    if (id.length() > 1 && value > 0) {
        painter->setPen(outline) ;

        if (displayedValue <= 10) {
            rectColor.setColorAt(1, Qt::black);
            rectColor.setColorAt(0, Qt::red);
            painter->setBrush(rectColor) ;
        } else if ((displayedValue <= 30) && (displayedValue > 10)) {
            rectColor.setColorAt(1, Qt::black);
            rectColor.setColorAt(0, Qt::yellow);
            painter->setBrush(rectColor) ;
        } else if ((displayedValue > 30) && (displayedValue < 50)) {
            rectColor.setColorAt(1, Qt::black);
            rectColor.setColorAt(0, Qt::blue);
            painter->setBrush(rectColor) ;
        } else if (displayedValue >= 50) {
            rectColor.setColorAt(1, Qt::black);
            rectColor.setColorAt(0, Qt::green);
            painter->setBrush(rectColor) ;
        } else {
            rectColor.setColorAt(1, Qt::black);
            rectColor.setColorAt(0, Qt::black);
            painter->setBrush(rectColor) ;
        }

        painter->drawRect(barRect) ;
        painter->setFont(font) ;

        QString valueString ;
        valueString.setNum(displayedValue) ;

        painter->drawText(valueTextPoints, valueString) ;
        painter->drawText(idTextPoints, id) ;
    }
}

//! Updates new value for the bar and calculates right value for the bar.
void Bar::setNewValue(int value, int displayedValue)
{
    this->origValue = value ;
    this->displayedValue = displayedValue ;
    if (value <= 100 || value >= 0) {
        this->value = (int)(((qreal)value / 100) * barMax);
    }
}

/*!
 * This function updates correct value ratio. This is needed when drawing area
 * size is changed.
 */
void Bar::updateValue()
{
    this->value = (int)(((qreal)origValue / 100) * barMax);
}

//! Sets new x position
void Bar::setXpos(int x)
{
    barRect.setX(x) ;
    this->x = x ;
}

//! return x position
int Bar::getXpos()
{
    return this->x ;
}

//! Sets new y position
void Bar::setYpos(int y)
{
    this->y = y ;
}

//! return y position
int Bar::getYpos()
{
    return this->y ;
}

//! Set the height of the bar
void Bar::setHeight(int height)
{
    this->height = height ;
}

//! return height of the bar
int Bar::getHeight()
{
    return this->height ;
}

//! Set the width of the bar
void Bar::setWidth(int width)
{
    this->width = width ;
}

//! return width of the bar
int Bar::getWidth()
{
    return this->width ;
}
