/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QMessageBox>
#include <QString>
#include <QVector>
#include "addpoidialog.h"
#include "poi.h"
#include "poidatabase.h"
#include "poicategoriesdialog.h"
#include <QDebug>

//! Constructor.
AddPoiDialog::AddPoiDialog(QtMapper &mainWindow, QWidget *parent)
    : QDialog(parent), mainWindow(mainWindow),
        minLatitude(-85.05113),
        maxLatitude(85.05113),
        minLongitude(-180.00000),
        maxLongitude(180.00000),
        maxDecimals(6),
        windowTitle("Add POI")
{
    setupUi(this);
    setWindowTitle(windowTitle);
    this->lineEditLatitude->setValidator(new QDoubleValidator(minLatitude, maxLatitude,
                                                              maxDecimals, lineEditLatitude)) ;
    this->lineEditLongitude->setValidator(new QDoubleValidator(minLongitude, maxLongitude,
                                                               maxDecimals, lineEditLongitude)) ;
    buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
    connect(lineEditLabel, SIGNAL(textChanged(const QString &)),
            this, SLOT(checkLabel(const QString &)));

    connect(buttonBox, SIGNAL(accepted()), this, SLOT(accept())) ;
    connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    connect(pushButtonEditCategories, SIGNAL(clicked()),
            this, SLOT(showPoiCategoriesDialog()));
            
    poiDb = mainWindow.getPoiDb();
    selectedIndex = 0;
}

void AddPoiDialog::accept()
{
    if((this->lineEditLatitude->hasAcceptableInput()) &&
       this->lineEditLongitude->hasAcceptableInput()) {
            QDialog::accept() ;
    }
    else {
        QMessageBox::information(this, tr("Information"),
                   tr("Latitude and/or Longitude is in wrong format.\n"
                      "Latitude range: %1 - +%2\n"
                      "Longitude range: %3 - +%4")
                   .arg(minLatitude)
                   .arg(maxLatitude)
                   .arg(minLongitude)
                   .arg(maxLongitude),
                   QMessageBox::Ok) ;
    }
}
//! Sets labels of POI categories to the combobox.
void AddPoiDialog::setData()
{
    comboBoxCategory->clear();

    for (qint32 i = 0; i < categories.count(); i++) {
        comboBoxCategory->addItem(categories.at(i).getLabel());
    }

    comboBoxCategory->setCurrentIndex(selectedIndex);
}

//! Shows Dialog POI categories.
void AddPoiDialog::showPoiCategoriesDialog()
{
	PoiCategoriesDialog pc(mainWindow);
	pc.pushButtonDelete->hide();
	selectedIndex = comboBoxCategory->currentIndex();
	
	if (pc.exec() == QDialog::Accepted) {
        setCategories();
    }
}

//! Destructor.
AddPoiDialog::~AddPoiDialog()
{
}

//! Set the contents of the latitude field.
void AddPoiDialog::setLatitudeContents(qreal latitude)
{
    lineEditLatitude->setText(QString("%1").arg(latitude));
}

//! Set the contents of the longitude field.
void AddPoiDialog::setLongitudeContents(qreal longitude)
{
    lineEditLongitude->setText(QString("%1").arg(longitude));
}

//! Gets the POI categories and puts them in the combo box.
void AddPoiDialog::setCategories()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    categories.clear();
    poiDb->getPoiCategories(categories);
    setData();
}

//! Adds a new POI to the database.
void AddPoiDialog::addPoiToDatabase()
{

        if (!poiDb->isConnected()) {
            poiDb->createConnection();
        }

        Poi poi;
        poi.setLabel(lineEditLabel->text());
        poi.setDescription(textEditDescription->toPlainText());
        poi.setLatitude(lineEditLatitude->text().toDouble());
        poi.setLongitude(lineEditLongitude->text().toDouble());

        QVector<Poi> allPois;

        bool similar = false;

        poiDb->getAllPois(allPois);

        for (qint32 i = 0; i < allPois.count(); i++) {
            if (poi.isSimilar(allPois.at(i))) {
                similar = true;
                break;
            }
        }

        if (similar == true) {
            QMessageBox::information(this,
                                     tr("Notice"),
                                     tr("A similar POI already exists!"),
                                     QMessageBox::Ok);
        }
        else {
            categories.clear();
            poiDb->getPoiCategories(categories);
            poi.setCategory(categories.at(comboBoxCategory->currentIndex()));
            poiDb->insertPoi(poi);
        }
}

//! Enables the dialog's OK-button.
void AddPoiDialog::checkLabel(const QString &poiName)
{
    if (poiName.isEmpty()) {
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(false);
    }
    else {
        buttonBox->button(QDialogButtonBox::Ok)->setEnabled(true);
    }
}
