/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#ifndef QTMAPPER_H
#define QTMAPPER_H

#include <QMainWindow>
#include <QPointer>
#include <QStack>
#include <QString>
#include <QVector>
#include "definitions.h"
#include "location.h"
#include "maprepository.h"
#include "ui_QtMapper.h"
#include "poi.h"
#include "trackpoint.h"

class CoordinateConverter;
class RouteDownloader;
class GpsBase;
class MapDatabase;
class MapDownloader;
class MapTile;
class MapTileCache;
class MapTileDownloadDialog;
class MapView;
class Location;
class PoiData;
class PoiDatabase;
class PoiDownloader;
class RouteData;
class SearchLocationData;
class SearchLocationDownloader;
class SearchLocationDialog;
class SearchView;
class QAction;
class QActionGroup;
class QGraphicsSceneDragDropEvent;
class QGraphicsSceneMouseEvent;
class QGraphicsSceneWheelEvent;
class QMenu;
class QLabel;
class QPixmap;
class QSettings;
class QToolBar;

/*!
 * The Qt Mapper application main window.
 */
class QtMapper : public QMainWindow, public Ui::QtMapper
{
    Q_OBJECT

public:
    /*!
     * Constructor.
     *
     * \param parent The parent widget. Default value is 0 (no parent).
     * \param f The window flags. Default value is 0.
     */
    QtMapper(QWidget *parent = 0, Qt::WFlags f = 0);

    //! Destructor.
    ~QtMapper();

    /*!
     * Centers the view on the location at the specified coordinates.
     *
     * \param longitude The longitude of the location to go to.
     * \param latitude The latitude of the location to go to.
     */
    void goToLocation(qreal longitude, qreal latitude);

    //! Get the current screen location.
    Location getCurrentMapLocation() {
        return currentMapLocation;
    }

    //! Get the current GPS location.
    Location getCurrentGpsLocation() {
        return currentGpsLocation;
    }

    /*!
    * Returns distance between given parameters in kilometers.
    */
    qreal distanceBetweenCoordinates(qreal startLatitude, qreal startLongitude,
                                     qreal endLatitude, qreal endLongitude);

    /*!
    * Returns Pointer to POI database.
    * \return Pointer to POI database.
    */
    PoiDatabase *getPoiDb() { return poiDb; }
    
    /*!
     * Goes to selected POI (from browse POIs) and shows the POI.
     * \param poi POI object.
     */
    void goToSelectedPoi(Poi poi);

private:
    QPointer<SearchView> searchView;

    //! Stores mouse's pixel coordinates when it is pressed
    QPoint mousePressScreenCoordinates;

    //! Stores mouse's pixel coordinates when it is released
    QPoint mouseReleaseScreenCoordinates;

    //! Whether user wants the quickmenu or not
    /*!
     * Default value for this is true. The value is changed when used clicks
     * "Show Quick Menu" in the "View" -menu.
     */
    bool userWantsQuickMenu;

    //! Time to wait before popping up the "Load default repositories?" message.
    const int delay;

    //! Window width.
    const int windowWidth;

    //! Window height.
    const int windowHeight;

    //! Longitude of default location
    const qreal defaultLongitude;

    //! Latitude of default location
    const qreal defaultLatitude;

    //! Default zoomlevel
    const qint32 defaultZoom;

    //! Minimum allowed latitude value
    const qreal minLatitude;

    //! Maximum allowed latitude value
    const qreal maxLatitude;

    //! Minimum allowed longitude value
    const qreal minLongitude;

    //! Maximum allowed longitude value
    const qreal maxLongitude;

    //! Maximum number of decimals to use with latitude/longitude values.
    const qint32 maxDecimals;

    // Status bar times for import POIs
    const int statusBarTimeFileSaved;
    const int statusBarTimeFileLoaded;

    //! Maximum number of results shown in POI list
    const int limitResultOfPois;

    //! Invalid latitude/longitude value
    const qreal invalidValue;

    //! Earth's radius, needed for calculating distances
    const int earthRadiusInKilometres;

    //! Checking interval
    const qint32 checkingInterval;

    //! Meters in kilometer
    const qint32 metersInKilometer;

    const qint32 trackingInterval;

    //! Maximum count of points of track.
    const qint32 maximumTrackSize;

    const qreal minimumTrackLineDistance;

    //! Tooltip text for gps fix indicator light, when fix is not valid
    const QString gpsFixInvalidTooltip;

    //! Tooltip text for gps fix indicator light, when fix is valid
    const QString gpsFixValidTooltip;

    //! Zoom level when Go to POI (browse POIs dialog)
    const qint32 zoomLevelOfGoToPoi;

    //! Toolbar icon size.
    const QSize toolBarIconSize;

    //! icon resources
    const QString lockedIconResource;
    const QString unlockedIconResource;

    //! The time between calls to the view centering function in milliseconds.
    const int centerViewInterval;

    //! Minimum size of area in pixels that must be drawn when using mouse zoom
    const int mouseZoomPixelThreshold;

    //! Maximum allowed zoomlevel
    const int maxZoomLevel;

    //! Minimum allowed zoomlevel
    const int minZoomLevel;

    //! Time to wait before calling goToSavedLocation, in milliseconds
    const int saveLocationDelay;

    //! Dialog for downloading maptiles from chosen area
    MapTileDownloadDialog *mapTileDownloadDialog;

    //! Pixmap for gps fix indicator light
    QPixmap *gpsFixIndicator;

    //! Label to hold gpsFixIndicator pixmap
    QLabel *gpsFixLabel;

    //! For storing the gps fix quality: no fix = 0, valid fix > 0
    int gpsFixQuality;

    //! whether mouse zooming feature is in use or not
    bool mouseZoomInUse;

    //! toggles downloading map images that user selects with mouse
    bool downloadMapImagesSelectedWithMouse;

    //! Whether view center's coordinates are updated when scrolling
    bool updateViewCenterCoordinates;

    //! x-index of the map tile in the upper left corner of mapscene
    int upperLeftCornerMapXIndex;

    //! y-index of the map tile in the upper left corner of mapscene
    int upperLeftCornerMapYIndex;

    //! longitude and latitude of the point in the center of the view
    QPointF centerOfScreen;

    //! x-index of the pixel in the center of the view
    int viewCenterXPixel;

    //! y-index of the pixel in the center of the view
    int viewCenterYPixel;

    //! coordinates of the starting point of mouse zooming
    QPointF mouseSelectionStartPoint;

    //! coordinates of the ending point of mouse zooming
    QPointF mouseSelectionEndPoint;
    
    //! If this is false, the application won't try to download images.
    bool mapImageDownloadingAllowed;

    //! Directory for map images
    QString mapImageDirectory;

    //! For storing different settings
    QSettings *settings;

    //! Maximum amount of space allowed for map images
    qint64 sizeLimitForMapImages;

    //! The number of images removed when space reserved for map images runs out
    int amountOfImagesToRemove;

    //! A GpsBase object for getting the location information.
    GpsBase *gps;

    //! Used for saving the map images to disk.
    MapDatabase *db;

    PoiDatabase *poiDb;
    //! Used for downloading map tiles.
    MapDownloader *md;

    //! Used for saving map images into memory cache.
    MapTileCache *cache;

    //! A list of POIs used in import POI Dialog.
    QVector<Poi> * pois;

    //! A map view widget, which is used as the MainWindow's central widget.
    MapView *view;

    //! The current screen location.
    Location currentMapLocation;

    //! The current GPS location.
    Location currentGpsLocation;

    //! A list of the map repositories used in the application.
    QVector<MapRepository> repositories;

    //! The currently selected repository.
    qint32 activeRepository;

    /*!
     * When downloading map images, the download requests are put in this stack.
     * A download request is carried in a Location object whose x, y and
     * and the zoom level properties are used to get the correct tile.
     */
    QStack<Location> downloadQueue;

    /*!
     * Indicates whether a download is currently in progress.
     * Used to determine when to start a new download.
     */
    bool downloadInProgress;

    //! The application's main tool bar.
    QToolBar *toolBar;

    //! A group for the tool actions.
    QActionGroup *toolActions;

    //! Mouse coordinates.
    QPoint previousMouseLocation;

    //! A Dialog for searching location
    QPointer<SearchLocationDialog> searchLocation;

    //! For downloading location data.
    QPointer<SearchLocationDownloader> searchLocationDownloader;

    //! Searched location information.
    QPointer<SearchLocationData> searchLocationData;

    //! For downloading POI data.
    QPointer<PoiDownloader> poiDownloader;

    //! Downloaded POI information.
    QPointer<PoiData> poiData;

    //! Last checked latitude for checkIsTherePoiNear.
    qreal lastLatitude;

    //! Last checked latitude for checkIsTherePoiNear.
    qreal lastLongitude;

    //! Keep track which POIs are alarmed after last view updating.
    QVector<Poi> alarmedPois;

    //! Count of showed/alarmed nearest POIs.
    qint32 countNearestPois;

    //! defines distance to alarm (km).
    qreal distanceToAlarm;

    //! Timer for checkIsTherePoiNear.
    QTimer *checkPoiTimer;

    //! Container for tracking data.
    QVector<TrackPoint> trackPoints;

    //! Timer for tracking.
    QTimer *trackingTimer;

    //! The move tool action.
    QAction *moveToolAct;

    //! The zoom tool action.
    QAction *zoomToolAct;

    //! Action for cancelling downloads
    QAction *cancelDownloadsAct;

    //! The zoom level for the route display.
    int routeZoomLevel;

    //! Used to periodically check if the current waypoint needs to be changed.
    QTimer *routeTimer;

    //! The time to wait between checking the waypoints in milliseconds.
    static const int routeCheckInterval = 1000;

    //! For downloading route data.
    QPointer<RouteDownloader> routeDownloader;

    //! Holds the current route information.
    QPointer<RouteData> currentRouteData;

    //! Advanced user mode on/off.
    bool advancedMode;

    //! Initialize the toolbar and the actions in it.
    void createToolbar();

    //! Connect all signals.
    void connectSignals();

    //! Set up actions.
    void createActions();

    /*!
     * \brief Get a map tile.
     *
     * Gets a map tile. First the memory cache is searched.
     * If the tile is not in the cache, it is searched for in the local
     * database. If the tile's not in the database either, the tile is queued
     * for download from the INTERNET.
     *
     * \param zoomLevel The zoom level of the tile.
     * \param mapX The x coordinate of the map tile.
     * \param mapY The y coordinate of the map tile.
     * \param screenX The x position (row) to draw the tile on screen.
     * \param screenY The y position (column) to draw the tile on screen.
     */
    MapTile *getTile(quint32 zoomLevel, quint32 mapX, quint32 mapY,
                     qint32 screenX, qint32 screenY);

    /*!
     * \brief Select tiles to display.
     *
     * Selects the map tiles to display after zooming or panning the view.
     * The number of tiles displayed is determined by the current zoom level.
     * On level 0 there is one tile and on level 1 there are four tiles.
     * On levels above 1 the amount of tiles displayed is defined
     * in definitions.h
     */
    void selectTiles();

    /*!
       * Draws location marker to current location.
    */
    void drawMarker();

    /*!
    * Draws marker to defined location.
    */
    void drawInfoMarker(qreal latitude, qreal longitude,
                        QString infoText, InfoType infoType);

    /*!
     * \brief Limits image cache size
     *
     * Checks the amount of space taken by the map images and removes
     * images if needed, starting with the oldest ones.
     *
     * \param byteLimit The maximum allowed space for images in bytes
     */
    void limitImageCache(qint64 byteLimit);

    //! Saves user settings
    void saveSettings();

    //! Loads user settings
    void loadSettings();

    //! zoom in and out use this method
    void zoom(qint32 zoom);

    //! Adds default POI categories if there is no categories in database
    void addDefaultPoiCategories();

    //! Draw lines between the line points along the route.
    void drawRouteLines();

    //! Resets the repository list.
    void resetRepositories();

private slots:
    //! Called when coordinateToolAct is triggered
    void coordinateToolActTriggered();

    //! Sets the mapview to location that was viewed before last shutdown
    void goToSavedLocation();

    /*!
     * Called when the centerOnLocation action is toggled.
     *
     * \param enable True enables view centering, false disables it.
     */
    void centerViewOnLocationToggled(bool enable);

    /*!
     * Enabling and disabling gps depending on the choice of the checkbox
     * in the GPS -dropdown menu.
     */
    void enableGps() ;

    //! Update view center pixels and their coordinates
    void setViewCenterCoordinates();

    //! Dowloads map images from area set in mapTileDownloadDialog
    void downloadThemTiles();

    //! Hides "Download map images" dialog, enables area selection with mouse
    void selectAreaButtonClicked();

    //! Updates the gps fix indicator light
    void gpsFixChanged(int);

    //! Called when mapview is scrolled to right edge
    void scrolledToRightEdge();

    //! Called when mapview is scrolled to left edge
    void scrolledToLeftEdge();

    //! Called when mapview is scrolled to top edge
    void scrolledToTopEdge();

    //! Called when mapview is scrolled to bottom edge
    void scrolledToBottomEdge();

    /*!
     * \brief Handle pending downloads.
     *
     * Checks the download queue. If there are downloads left, the next
     * download is taken from the queue and a download request is sent.
     */
    void handleDownloadQueue();

    /*!
     * \brief Save a downloaded map image.
     *
     * This slot is connected to the imageDownloaded signal of a MapDownloader
     * object, which is sent when the MapDownloader has finished downloading
     * an image. After the image is received from the downloader, it's inserted
     * into both the memory cache and the local database and added to the view.
     */
    void processDownloadedImage();

    /*!
     * \brief Handle a failed download.
     *
     * This slot is connected to the imageDownloadFailed slot of a
     * MapDownloader object, which is sent if a download failed for some reason.
     *
     * \param error A string containing the reason for a failed download.
     */
    void handleFailedDownload(QString error);

    /*!
     * \brief Cancel all downloads.
     *
     * Clears all pending download requests from the queue and aborts a
     * download in progress, if any.
     */
    void clearDownloadQueue();

    //! Zoom in the map view.
    void zoomIn();

    //! Zoom out the map view.
    void zoomOut();

    /*! \brief Pan the map view up.
     *
     *  Pans the map view up and returns true if succesful.
     *  Returns false if we're at the edge of the map.
     */
    bool panUp();

    /*! \brief Pan the map view down.
     *
     *  Pans the map view down and returns true if succesful.
     *  Returns false if we're at the edge of the map.
     */
    bool panDown();

    /*! \brief Pan the map view left.
     *
     *  Pans the map view left and returns true if succesful.
     *  Returns false if we're at the edge of the map.
     */
    bool panLeft();

    /*! \brief Pan the map view right.
     *
     *  Pans the map view right and returns true if succesful.
     *  Returns false if we're at the edge of the map.
     */
    bool panRight();

    //! Centers the view on the current GPS location.
    void goToCurrentGpsLocation();

    //! Opens the "Go to location" dialog.
    void openGoToLocationDialog();

    /*!
     * The latitude received from the GPS device has changed.
     *
     * \param latitude The new latitude.
     */
    void gpsLatitudeChanged(qreal latitude);

    /*!
     * The longitude received from the GPS device has changed.
     *
     * \param longitude The new longitude.
     */
    void gpsLongitudeChanged(qreal longitude);

    /*!
     * Some other value (speed etc.) received from the GPS device has changed.
     *
     * \param name The name of the changed value.
     * \param value The new value.
     */
    void gpsOtherValueChanged(const QString name, qreal value);

    /*!
     * \brief Displays the satellite details dialog.
     *
     * Displays the satellite details dialog, which shows the information
     * received from the gps device.
     */
    void gpsDetails();

    //! Displays the help dialog
    void help();

    //! Displays the about dialog
    void about();

    //! Displays the preferences dialog
    void preferences();

    //! Show/hide the quick menu bar.
    void showQuickMenu(bool show);

    /*!
     * The mouse wheel was moved in the map view area.
     *
     * \param e Information about the wheel movement.
     */
    void mouseWheelMovedInMapView(QGraphicsSceneWheelEvent *e);

    /*!
     * Shows Dialog POI categories.
     */
    void showPoiCategoriesDialog();

    /*!
    * Shows Dialog Browse POIs
    */
    void showBrowsePoisDialog();

    /*!
    * Shows Dialog Import POIs
    */
    void showImportPoiDialog();

    //! Shows the download indicator when there are ongoing downloads.
    void showDownloadIndicator();

    //! Hides the download indicator when all downloads are complete.
    void hideDownloadIndicator();

    //! Called when route data has finished downloading.
    void routeDownloadComplete();

    /*!
     * Called when route data downloading has failed.
     *
     * \param errorCode HTTP error code.
     * \param errorDescription Information about the error.
     */
    void routeDownloadFailed(int errorCode, const QString &errorDescription);

    /*!
     * Shows Dialog  search location.
     */
    void openSearchLocationDialog();

    //! Called when location download is complete
    void locationDownloadComplete();

    /*!
     * Shows error string if dowload fails
     *
     * \param errorCode HTTP error code.
     * \param errorDescription Information about the error.
     */
    void locationDownloadFailed(int errorCode, const QString &errorDescription);

    //! Download Location specified in search location dialog
    void goToSearchedLocation();

    //! Handle open search location buttons states
    void clearButtonState();

    /*!
     * One of the actions in the tool action group was triggered.
     *
     * \param act The action that was triggered.
     */
    void toolActionsClicked(QAction *act);

    //! Ask the user if he wants to load the default repositories.
    void loadDefaultRepositories();

    //! listens to mouse button presses in mapscene
    void mouseButtonPressedInMapScene(QGraphicsSceneMouseEvent *e);

    //! listens to mouse button releases in mapscene
    void mouseButtonReleasedInMapScene(QGraphicsSceneMouseEvent *e);

    //! slot for mouse zooming feature
    void mouseZoom();

    /*!
    * Checks is there any pois located to screen and draw them.
    */
    void drawPoiMarkers();

    //! Open the route download dialog.
    void openRouteDownloadDialog();

    //!Called when POIs are downloaded from internet.
    void poiDownloadComplete();

    /*!
     * Called if the POI downloading failed.
     *
     * \param errorCode HTTP error code.
     * \param errorDescription Information about the error.
     */
    void poiDownloadFailed(int errorCode, const QString &errorDescription);

    //! Shows download POIs dialog.
    void showDownloadPoisDialog();

    //! Shows downloaded POIs dialog
    void showDownloadedPoisDialog(QVector<Poi> downloadedPois);

    //! Displays the map image download dialog
    void downloadMapImagesFromSelectedAreaDialog();//(bool areaPainted);

    /*!
    * Checks if there POI near. Called when location changes.
    */
    void checkIsTherePoiNear();

    /*!
    * Change alarm state off or on.
    */
    void changeAlarmState(bool checked);

    /*!
     * Clears the current route data.
     * Called when the clearRouteAct action is triggered.
     */
    void clearRoute();

    //! Redraws all items in the map view.
    void refreshMapView();

    //! Called when the route destination has been reached.
    void destinationReached();

    //! Moves to the next waypoint along the route.
    void moveToNextWaypoint();

    //! Saves track point to vector.
    void storeTrackPoint();

    //! Change conditions of tracking.
    void enableTracking();

    //! Emitted when tracking is enabled or disabled.
    //void enableTracking(bool) ;

    /*!
     * \brief Check waypoints when drawing route.
     *
     * Called when the routeTimer sends its timeout() signal.
     * If the current location is within a set distance
     * from the current waypoint along the route, the
     * moveToNextWaypoint() function is called.
     */
    void checkWaypoint();

    //! Saves track to file.
    void saveTrack();

    //! Open track from xml file.
    void openTrack();

    //! Shows or hide track.
    void showTrack(bool checked);

    //! Draw lines between the track points among track.
    void drawTrackLines();

    //! Clears stored track (track clears also when quitting the application).
    void clearTrack();

    /*!
     * Add a POI at a location selected with the mouse.
     *
     * \param x The x coordinate at the mouse cursor.
     * \param y The y coordinate at the mouse cursor.
     */
    void addPoiFromMapCoordinates(int x, int y);

    /*!
     * Start/stop tracking when the quick menu tracking button is pressed.
     *
     * \param enable True starts tracking, false stops it.
     */
    void enableTrackingWithQuickMenuButton(bool enable);

    /*!
    * Move screen center point to nearest POI.
    */
    void goToNearestPoi();

    //! Centers the view on gps marker and sets view center coordinates
    void centerOnGpsMarker();

    /*!
    * Changes the state of show POIs.
    */
    void changeShowPoisState();
    

protected:
    //! Called whenever the QtMapper window is closed
    void closeEvent(QCloseEvent *e);
};

#endif
