/*
    Qt Mapper - A GPS map application
    Copyright (C) 2008  Ixonos Plc. Authors:

        Antero Lehtonen - antero.lehtonen@ixonos.com
        Atte Tihinen - atte.tihinen@ixonos.com
        Jaakko Putaala - jaakko.putaala@ixonos.com
        Teppo Pennanen - teppo.pennanen@ixonos.com

    Qt Mapper is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    Qt Mapper is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Qt Mapper; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
    USA.
*/

#include <QAction>
#include <QActionGroup>
#include <QDebug>
#include <QDir>
#include <QFileDialog>
#include <QFileInfo>
#include <QGraphicsItem>
#include <QGraphicsSceneWheelEvent>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QList>
#include <QLocale>
#include <QMessageBox>
#include <QMouseEvent>
#include <QNetworkReply>
#include <QPixmap>
#include <QPointer>
#include <QPushButton>
#include <QScrollBar>
#include <QString>
#include <QTimer>
#include <QToolBar>
#include <QVector>
#include <QSettings>
#include <QSound>
#include <QXmlDefaultHandler>
#include <QXmlSimpleReader>
#include <cmath>

#include "QtMapper.h"
#include "aboutdialog.h"
#include "addpoidialog.h"
#include "browsepoisdialog.h"
#include "definitions.h"
#include "downloadedpoisdialog.h"
#include "downloadpoisdialog.h"
#include "downloadroutedialog.h"
#include "gpsbase.h"
#include "gpsdata.h"
#include "gpxhandler.h"
#include "helpdialog.h"
#include "importpoisdialog.h"
#include "linepoint.h"
#include "mapdatabase.h"
#include "mapdownloader.h"
#include "maprepository.h"
#include "maptile.h"
#include "maptilecache.h"
#include "maptiledownloaddialog.h"
#include "mapscene.h"
#include "mapview.h"
#include "networkactivityindicator.h"
#include "poicategoriesdialog.h"
#include "poidata.h"
#include "poidatabase.h"
#include "poidownloader.h"
#include "preferencesdialog.h"
#include "routedata.h"
#include "routedownloader.h"
#include "satellitedetailsdialog.h"
#include "searchlocationdialog.h"
#include "searchlocationdownloader.h"
#include "searchview.h"
#include "trackingdatareader.h"
#include "trackingdatawriter.h"
#include "unitconverter.h"
#include "waypoint.h"

#define DEBUG

//! Constructor.
QtMapper::QtMapper(QWidget *parent, Qt::WFlags f) :
        QMainWindow(parent, f),
        delay(1000),
        windowWidth(mapTileWidth * mapTileColumns),
        windowHeight(mapTileHeight * mapTileRows + 90),
        defaultLongitude(25.5093),
        defaultLatitude(65.0101),
        defaultZoom(10),
        minLatitude(-85.05113),
        maxLatitude(85.05113),
        minLongitude(-180.00000),
        maxLongitude(180.00000),
        maxDecimals(6),
        statusBarTimeFileSaved(2000),
        statusBarTimeFileLoaded(2000),
        limitResultOfPois(50),
        invalidValue(-200.0),
        earthRadiusInKilometres(6371),
        checkingInterval(3000),
        metersInKilometer(1000),
        trackingInterval(1000),
        maximumTrackSize(10000),
        minimumTrackLineDistance(0.010),
        gpsFixInvalidTooltip(tr("GPS fix is invalid")),
        gpsFixValidTooltip(tr("GPS fix is valid")),
        zoomLevelOfGoToPoi(16),
        toolBarIconSize(QSize(24, 24)),
        lockedIconResource(":/resources/images/locked.png"),
        unlockedIconResource(":/resources/images/unlocked.png"),
        centerViewInterval(1000),
        mouseZoomPixelThreshold(20),
        maxZoomLevel(18),
        minZoomLevel(2),
        saveLocationDelay(5)
{
    setupUi(this);
    setWindowTitle(tr("Qt Mapper"));
    setIconSize(toolBarIconSize);

    view = new MapView(this);
    setCentralWidget(view);

    showTrackAct->setEnabled(false);
    clearTrackAct->setEnabled(false);
    saveTrackAct->setEnabled(false);

    loadSettings();

    md = new MapDownloader();

    searchView = new SearchView(this);
    searchView->setHidden(true);

    cache = new MapTileCache();

    if (repositories.isEmpty()) {
        db = NULL;
    }
    else {
        db = new MapDatabase(repositories.at(activeRepository),
                             mapImageDirectory);
        db->createConnection();
    }

    poiDb = new PoiDatabase;
    poiDb->createConnection();
        
    downloadInProgress = false;

    gps = new GpsBase();
    gps->startGps();

    previousMouseLocation = QPoint(0, 0);

    createActions();
    createToolbar();

    currentMapLocation.setZoomLevel(defaultZoom);
    currentMapLocation.setLatitude(defaultLatitude);
    currentMapLocation.setLongitude(defaultLongitude);
    currentMapLocation.setMapX(currentMapLocation
                               .getXTile(currentMapLocation.getLongitude(),
                                         currentMapLocation.getZoomLevel()));
    currentMapLocation.setMapY(currentMapLocation
                               .getYTile(currentMapLocation.getLatitude(),
                                         currentMapLocation.getZoomLevel()));

    currentGpsLocation = currentMapLocation;
    currentGpsLocation.setLatitude(invalidValue);
    currentGpsLocation.setLongitude(invalidValue);
    currentGpsLocation.setMapX((int)invalidValue);
    currentGpsLocation.setMapY((int)invalidValue);

    currentGpsLocation.setMaximumValidLongitude(maxLongitude);
    currentGpsLocation.setMinimumValidLongitude(minLongitude);
    currentGpsLocation.setMaximumValidLatitude(maxLatitude);
    currentGpsLocation.setMinimumValidLatitude(minLatitude);

    centerOfScreen.setX(defaultLongitude);
    centerOfScreen.setY(defaultLatitude);

    checkPoiTimer = new QTimer(this);

    trackingTimer = new QTimer(this);

    routeTimer = new QTimer(this);

    connectSignals();

    /*
     * If the map repository list is empty, set up a timer ask the user if he
     * wants to load the default map repositories after the application has
     * started up.
     */
    if (repositories.isEmpty()) {
        QTimer::singleShot(delay, this, SLOT(loadDefaultRepositories()));
        mapImageDownloadingAllowed = false;
    }
    else {
        mapImageDownloadingAllowed = true;
        selectTiles();
        goToLocation(defaultLongitude, defaultLatitude);
    }

    mouseZoomInUse = false;
    downloadMapImagesSelectedWithMouse = false;

    userWantsQuickMenu = true;

    addDefaultPoiCategories();

    mousePressScreenCoordinates.setX(0);
    mousePressScreenCoordinates.setY(0);
    mouseReleaseScreenCoordinates.setY(0);
    mouseReleaseScreenCoordinates.setX(0);
    gpsEnabledAct->setChecked(true);

    QTimer::singleShot(saveLocationDelay, this, SLOT(goToSavedLocation()));

    goToGpsLocationAct->setEnabled(false);
    goToNearestPoiAct->setEnabled(false);
}

//! Destructör.
QtMapper::~QtMapper()
{
    saveSettings();

    delete md;

    if (db->connected()) {
        db->closeConnection();
    }

    delete db;
    delete cache;
    delete poiDb;
    delete gpsFixIndicator;

    gps->stopGps();
    delete gps;
}

//! Save a downloaded map image.
void QtMapper::processDownloadedImage()
{
    MapTile *newTile = md->getDownloadedImage();

    if (newTile != NULL) {
        // Put the downloaded image into the cache.
        cache->insert(newTile);

        // Store the downloaded image in the map database.
        if (db != NULL) {
            if (!db->connected()) {
                db->createConnection();
            }

            db->insertMapTile(newTile);
        }

        // Show the downloaded image.
        view->addMapTile(newTile, newTile->getScreenXCoordinate(),
                         newTile->getScreenYCoordinate());
    }

    downloadInProgress = false;

    if (downloadQueue.isEmpty()) {
        // Make a QDir object from the map image directory
        QDir directory("Images/");
        directory.setSorting(QDir::Time | QDir::DirsLast | QDir::Reversed);

        // Make a list of the contents
        QFileInfoList mapImageFiles = directory.entryInfoList();
        qint64 totalSize = 0;

        // check the current size of images directory
        for (int i = 0; i < mapImageFiles.size(); i++) {
            totalSize += mapImageFiles.at(i).size();
        }

        /* if the total size of the images is greater than the set maximum,
         * delete some of the files */
        if (totalSize > sizeLimitForMapImages) {
            for (int i = 0; i < amountOfImagesToRemove; i++) {
                if (mapImageFiles.at(i).fileName() == "." ||
                        mapImageFiles.at(i).fileName() == "..") {
                    break;
                }

                // Remove the file and the corresponding database entry.
                directory.remove(mapImageFiles.at(i).fileName());

                if (db != NULL) {
                    if (!db->connected()) {
                        db->createConnection();
                    }

                    db->removeMapTile(mapImageFiles.at(i).fileName());
                }
            }
        }
    }
    handleDownloadQueue();
}

//! Initialize the toolbar and the actions in it.
void QtMapper::createToolbar()
{
    toolBar = addToolBar(tr("Main"));
    toolBar->setFloatable(false);

    toolBar->addActions(toolActions->actions());
    toolBar->addSeparator();

    centerOnLocationAct->setChecked(false);
    centerOnLocationAct->setIcon(QIcon(unlockedIconResource));
    toolBar->addAction(centerOnLocationAct);

    toolBar->addAction(goToAddressAct);
    toolBar->addAction(drawPoisAct);

    gpsFixIndicator = new QPixmap;
    gpsFixIndicator->load(gpsFixInvalidIndicatorResource);
    gpsFixLabel = new QLabel;
    gpsFixLabel->setPixmap(*gpsFixIndicator);
    gpsFixLabel->setToolTip(gpsFixInvalidTooltip);
    toolBar->addSeparator();
    toolBar->addWidget(gpsFixLabel);
    gpsFixQuality = 0;
}

//! Moves to the next waypoint along the route.
void QtMapper::moveToNextWaypoint()
{
    int next = view->nextWaypoint();

    if (next > view->getRouteEnd()) {
        clearRoute();
    }
    else {
        goToLocation(currentRouteData->getWaypoints().at(next)->getLongitude(),
                     currentRouteData->getWaypoints().at(next)->getLatitude());
        refreshMapView();
    }
}

//! Set up actions.
void QtMapper::createActions()
{
    toolActions = new QActionGroup(this);
    toolActions->setExclusive(true);

    moveToolAct = new QAction(QIcon(moveToolResource), tr("Move Tool"), this);
    moveToolAct->setCheckable(true);
    moveToolAct->setChecked(true);
    toolActions->addAction(moveToolAct);

    zoomToolAct = new QAction(QIcon(zoomToolResource), tr("Zoom Tool"), this);
    zoomToolAct->setCheckable(true);
    toolActions->addAction(zoomToolAct);

    cancelDownloadsAct = new QAction(QIcon(stopDownloadResource),
                                     tr("Cancel DLs"), this);
    toolActions->addAction(cancelDownloadsAct);
    cancelDownloadsAct->setEnabled(false);

    showQuickMenuAct->setChecked(true);

    //coordinateToolAct = new QAction(/*QIcon(),*/"Lat/Lon", this);
    //coordinateToolAct->setCheckable(true);
}

//! Connect all signals.
void QtMapper::connectSignals()
{
    // Connect the signals of the map downloader.
    connect(md, SIGNAL(imageDownloaded()),
            this, SLOT(processDownloadedImage()));
    connect(md, SIGNAL(imageDownloadFailed(QString)),
            this, SLOT(handleFailedDownload(QString)));
    connect(md, SIGNAL(allHttpRequestsComplete()),
            this, SLOT(hideDownloadIndicator()));
    connect(md, SIGNAL(httpRequestsPending()), this,
            SLOT(showDownloadIndicator()));

    // Connect the signals of the actions.
    connect(zoomInAct, SIGNAL(triggered()), this, SLOT(zoomIn()));
    connect(zoomOutAct, SIGNAL(triggered()), this, SLOT(zoomOut()));
    connect(panUpAct, SIGNAL(triggered()), this, SLOT(panUp()));
    connect(panDownAct, SIGNAL(triggered()), this, SLOT(panDown()));
    connect(panLeftAct, SIGNAL(triggered()), this, SLOT(panLeft()));
    connect(panRightAct, SIGNAL(triggered()), this, SLOT(panRight()));
    connect(goToGpsLocationAct, SIGNAL(triggered()),
            this, SLOT(goToCurrentGpsLocation()));
    connect(gpsDetailsAct, SIGNAL(triggered()), this, SLOT(gpsDetails()));
    connect(quitAct, SIGNAL(triggered()), this, SLOT(close()));
    connect(goToLatLonAct, SIGNAL(triggered()),
            this, SLOT(openGoToLocationDialog()));
    connect(helpAboutAct, SIGNAL(triggered()), this, SLOT(about()));
    connect(helpHelpAct, SIGNAL(triggered()), this, SLOT(help()));
    connect(preferencesAct, SIGNAL(triggered()), this, SLOT(preferences()));
    connect(poiCategoriesAct, SIGNAL(triggered()),
            this, SLOT(showPoiCategoriesDialog()));
    connect(poiBrowseAct, SIGNAL(triggered()),
            this, SLOT(showBrowsePoisDialog()));
    connect(poiImportAct, SIGNAL(triggered()),
            this, SLOT(showImportPoiDialog()));

    connect(toolActions, SIGNAL(triggered(QAction *)),
            this, SLOT(toolActionsClicked(QAction *)));
    
    connect(poiDownloadAct, SIGNAL(triggered()),
            this, SLOT(showDownloadPoisDialog()));
    connect(downloadRouteAct, SIGNAL(triggered()),
            this, SLOT(openRouteDownloadDialog()));
    connect(clearRouteAct, SIGNAL(triggered()), this, SLOT(clearRoute()));

    connect(downloadMapImagesAct, SIGNAL(triggered()),
            this, SLOT(downloadMapImagesFromSelectedAreaDialog()));

    connect(showQuickMenuAct, SIGNAL(toggled(bool)),
            this, SLOT(showQuickMenu(bool)));
    connect(coordinateToolAct, SIGNAL(triggered()),
            this, SLOT(coordinateToolActTriggered()));
    connect(showSpeedAct, SIGNAL(toggled(bool)), view, SLOT(enableSpeedDisplay(bool))) ;
    connect(enableTrackingAct, SIGNAL(triggered()),
            this, SLOT(enableTracking()));
    connect(saveTrackAct, SIGNAL(triggered()), this, SLOT(saveTrack()));
    connect(openTrackAct, SIGNAL(triggered()), this, SLOT(openTrack()));
    connect(showTrackAct, SIGNAL(toggled(bool)), this, SLOT(showTrack(bool)));
    connect(clearTrackAct, SIGNAL(triggered()), this, SLOT(clearTrack()));

    // Connect the signals of the gps object.
    connect(&gps->getGpsData(), SIGNAL(latitudeChanged(qreal)),
            this, SLOT(gpsLatitudeChanged(qreal)));
    connect(&gps->getGpsData(), SIGNAL(longitudeChanged(qreal)),
            this, SLOT(gpsLongitudeChanged(qreal)));
    connect(&gps->getGpsData(), SIGNAL(fixQualityChanged(int)),
            this, SLOT(gpsFixChanged(int)));

    // Connect the signals of the map view.
    connect(view->getScene(),
            SIGNAL(mouseWheelMoved(QGraphicsSceneWheelEvent *)),
            this, SLOT(mouseWheelMovedInMapView(QGraphicsSceneWheelEvent *)));
    connect(view, SIGNAL(mapSceneAtRight()),
            this, SLOT(scrolledToRightEdge()));
    connect(view, SIGNAL(mapSceneAtLeft()),
            this, SLOT(scrolledToLeftEdge()));
    connect(view, SIGNAL(mapSceneAtTop()),
            this, SLOT(scrolledToTopEdge()));
    connect(view, SIGNAL(mapSceneAtBottom()),
            this, SLOT(scrolledToBottomEdge()));

    connect(view, SIGNAL(lastWaypointReached()),
            this, SLOT(destinationReached()));
            
    connect(goToAddressAct, SIGNAL(triggered()),
            this, SLOT(openSearchLocationDialog()));


    // Connect signal of go to selected location
    connect(alarmOnAct, SIGNAL(toggled(bool)),
            this, SLOT(changeAlarmState(bool)));

    // Connect signal of draw POI marker
    connect(drawPoisAct, SIGNAL(triggered()), this, SLOT(changeShowPoisState()));
     
    // Connect signal for checking is there a POI near.
    connect(checkPoiTimer, SIGNAL(timeout()),
            this, SLOT(checkIsTherePoiNear()));

    // Connect signal for tracking.
    connect(trackingTimer, SIGNAL(timeout()), this, SLOT(storeTrackPoint()));

    connect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent *)),
            this,
            SLOT(mouseButtonPressedInMapScene(QGraphicsSceneMouseEvent *)));
    connect(view->getScene(),
            SIGNAL(mouseButtonReleased(QGraphicsSceneMouseEvent *)),
            this,
            SLOT(mouseButtonReleasedInMapScene(QGraphicsSceneMouseEvent *)));

    // Connect the quickmenu's signals
    connect(view, SIGNAL(zoomedIn()), this, SLOT(zoomIn())) ;
    connect(view, SIGNAL(zoomedOut()), this, SLOT(zoomOut()));
    connect(view, SIGNAL(poiSelected(int, int)),
            this, SLOT(addPoiFromMapCoordinates(int, int)));
    connect(view, SIGNAL(enableTracking(bool)),
            this, SLOT(enableTrackingWithQuickMenuButton(bool)));

    connect(routeTimer, SIGNAL(timeout()), this, SLOT(checkWaypoint()));

    connect(goToNearestPoiAct, SIGNAL(triggered()),
            this, SLOT(goToNearestPoi()));
    connect(toolActions, SIGNAL(triggered(QAction *)),
            this, SLOT(toolActionsClicked(QAction *)));

    connect(gpsEnabledAct, SIGNAL(triggered()), this, SLOT(enableGps()));
    connect(centerOnLocationAct, SIGNAL(toggled(bool)),
            this, SLOT(centerViewOnLocationToggled(bool)));

    connect(searchView, SIGNAL(loadStarted()),
            this, SLOT(showDownloadIndicator()));

    connect(&gps->getGpsData(), SIGNAL(valueChanged(const QString &, qreal)),
            this, SLOT(gpsOtherValueChanged(const QString &, qreal))) ;

}

void QtMapper::enableGps()
{
    if (gpsEnabledAct->isChecked()) {
        gps->startGps();
    }
    else if (!gpsEnabledAct->isChecked()) {
        gps->stopGps();
    }
}

//! Get a map tile.
MapTile *QtMapper::getTile(quint32 zoomLevel,
                           quint32 mapX, quint32 mapY,
                           qint32 screenX, qint32 screenY)
{
    MapTile *newTile = NULL;

    // Try to find the tile in the cache first.
    newTile = cache->find(zoomLevel, mapX, mapY, screenX, screenY);

    // If getting the tile from the cache failed, search the database.
    if (newTile == NULL) {
        QString key = QString("/%1/%2/%3").arg(zoomLevel).arg(mapX).arg(mapY);

        if (db != NULL) {
            if (!db->connected()) {
                db->createConnection();
            }

            newTile = db->getMapTile(key);
        }

        // If the tiles were not in the database either, try to download them.
        if (newTile == NULL || newTile->pixmap().isNull()) {
            downloadQueue.push(Location(zoomLevel, mapX, mapY,
                                        screenX, screenY));
            handleDownloadQueue();
        }
        else {
            newTile->setScreenXCoordinate(screenX);
            newTile->setScreenYCoordinate(screenY);
        }
    }

    // Return the new tile if it was found, NULL otherwise.
    return newTile;
}

//! Select tiles to display.
void QtMapper::selectTiles()
{
    if (!mapImageDownloadingAllowed) {
        return;
    }

    qint32 zoom = currentMapLocation.getZoomLevel();

    /*
     * Minimum zoom level (zoomed out).
     * At this zoom level the map is made of only one tile.
     */
    if (zoom == 0) {
        MapTile *tile = getTile(0, 0, 0, 0, 0);

        /*
         * If the tile was found in the cache or the database,
         * put it on screen.
         * Otherwise the tile will be shown later after it's downloaded.
         */
        if (tile != NULL) {
            view->addMapTile(tile, tile->getScreenXCoordinate(),
                             tile->getScreenYCoordinate());
        }

        upperLeftCornerMapXIndex = 0;
        upperLeftCornerMapYIndex = 0;

    }

    // At zoom level 1 the map contains four tiles.
    else if (zoom == 1) {
        MapTile *tiles[tilesDisplayedAtFirstZoomLevel];

        tiles[0] = getTile(zoom, 0, 0, 0, 0);
        tiles[1] = getTile(zoom, 1, 0, 1, 0);
        tiles[2] = getTile(zoom, 0, 1, 0, 1);
        tiles[3] = getTile(zoom, 1, 1, 1, 1);

        for (int i = 0; i < tilesDisplayedAtFirstZoomLevel; i++) {
            if (tiles[i] != NULL) {
                view->addMapTile(tiles[i], tiles[i]->getScreenXCoordinate(),
                                 tiles[i]->getScreenYCoordinate());
            }
        }

        upperLeftCornerMapXIndex = 0;
        upperLeftCornerMapYIndex = 0;

    }
    /*
     * At higher zoom levels the map contains more tiles than
     * can be displayed on screen at once.
     */
    else {
        MapTile *tiles[tilesDisplayedAtHigherZoomLevels];

        // Get the current tile and the tiles surrounding it.
        qint32 currentXTile = currentMapLocation.getMapX();
        qint32 currentYTile = currentMapLocation.getMapY();

        int numberOfRows = mapTileRows;
        int numberOfColumns = mapTileColumns;

        int column, row;
        int tileIndex = 0;

        int x = currentXTile - numberOfColumns / 2;
        int y = currentYTile - numberOfRows / 2;

        upperLeftCornerMapXIndex = x;
        upperLeftCornerMapYIndex = y;

        for (row = 0; row < numberOfRows; row++, y++) {
            for (column = 0; column < numberOfColumns; column++, x++) {

                if (x < 0 || y < 0 || x >= pow(2, zoom) || y >= pow(2, zoom)) {
                    tiles[tileIndex] = NULL;
                }
                else {
                    tiles[tileIndex] = getTile(zoom, x, y, column, row);
                }
                tileIndex++;
            }
            x -= numberOfColumns;
        }

        for (int i = 0; i < tilesDisplayedAtHigherZoomLevels; i++) {
            if (tiles[i] != NULL) {
                view->addMapTile(tiles[i], tiles[i]->getScreenXCoordinate(),
                                 tiles[i]->getScreenYCoordinate());
            }
        }

    }

    drawMarker();

    if (view->routeDisplayEnabled()) {
        drawRouteLines();
    }
}

//! Draws location marker to current location.
void QtMapper::drawMarker()
{
    qreal longitude = currentGpsLocation.getLongitude();
    qreal latitude = currentGpsLocation.getLatitude();

    if (longitude > maxLongitude || longitude < minLongitude ||
        latitude > maxLatitude || latitude < minLatitude) {
        return;
    }

    // If the current GPS location is in the view, draw the marker.
    int mapX = currentMapLocation.getMapX();
    int mapY = currentMapLocation.getMapY();

    int gpsX = currentGpsLocation.getMapX();
    int gpsY = currentGpsLocation.getMapY();

    int zoomLevel = currentGpsLocation.getZoomLevel();

    int xPixel = 0;
    int yPixel = 0;

    if (zoomLevel == 0) {
        xPixel = currentGpsLocation.getXPixel(longitude, zoomLevel);
        yPixel = currentGpsLocation.getYPixel(latitude, zoomLevel);

        view->drawLocationMarker(xPixel, yPixel, zoomLevel);
    }

    else if (zoomLevel == 1) {
        xPixel = currentGpsLocation.getXPixel(longitude, zoomLevel)
                 + (gpsX - 1) * 256;
        yPixel = currentGpsLocation.getYPixel(latitude, zoomLevel)
                 + (gpsY - 1) * 256;

        view->drawLocationMarker(xPixel, yPixel, zoomLevel);
    }

    else { // higher zoomlevels

        int numberOfRows = mapTileRows;
        int numberOfColumns = mapTileColumns;

        // First check if the correct row is displayed on screen
        for (int row = mapY - numberOfRows / 2;
                 row <= mapY + numberOfRows / 2;
                 row++) {

            // if the correct row was found...
            if (row == gpsY) {

                // ...see if we have the right column
                for (int column = mapX - numberOfColumns / 2;
                         column <= mapX + numberOfColumns / 2;
                         column++) {

                    /* ...and if the correct column was found,
                     * draw the marker according to current zoomlevel*/
                    if (column == gpsX) {

                        xPixel = currentGpsLocation.
                                 getXPixel(longitude, zoomLevel)
                                 + (column - mapX +
                                    numberOfColumns / 2 - 1) * 256;
                        yPixel = currentGpsLocation.
                                 getYPixel(latitude, zoomLevel)
                                 + (row - mapY + numberOfRows / 2 - 1) * 256;

                        view->drawLocationMarker(xPixel, yPixel, zoomLevel);
                    }
                }
            }
        }
    }
}

//Draws marker to defined location
void QtMapper::drawInfoMarker(qreal latitude, qreal longitude, QString infoText,
                              InfoType infoType)
{
    int zoomLevel = currentMapLocation.getZoomLevel();
    // If the current GPS location is in the view, draw the marker.
    int mapX = currentMapLocation.getMapX();
    int mapY = currentMapLocation.getMapY();

    int poiX = currentMapLocation.getXTile(longitude, zoomLevel);
    int poiY = currentMapLocation.getYTile(latitude, zoomLevel);


    int xPixel = 0;
    int yPixel = 0;

    if (zoomLevel == 0) {
        xPixel = currentGpsLocation.getXPixel(longitude, zoomLevel);
        yPixel = currentGpsLocation.getYPixel(latitude, zoomLevel);

        view->showLocationInfo(xPixel, yPixel, zoomLevel, infoText, infoType);
    }

    else if (zoomLevel == 1) {
        xPixel = currentGpsLocation.getXPixel(longitude, zoomLevel)
                 + (poiX - 1) * 256;
        yPixel = currentGpsLocation.getYPixel(latitude, zoomLevel)
                 + (poiY - 1) * 256;

        view->showLocationInfo(xPixel, yPixel, zoomLevel, infoText, infoType);
    }

    else { // higher zoomlevels

        int numberOfRows = mapTileRows;
        int numberOfColumns = mapTileColumns;

        // First check if the correct row is displayed on screen
        for (int row = mapY - numberOfRows / 2;
                row <= mapY + numberOfRows / 2;
                row++) {

            // if the correct row was found...
            if (row == poiY) {

                // ...see if we have the right column
                for (int column = mapX - numberOfColumns / 2;
                        column <= mapX + numberOfColumns / 2;
                        column++) {

                    /* ...and if the correct column was found, draw the marker
                    * according to current zoomlevel*/
                    if (column == poiX) {

                        xPixel = currentGpsLocation.getXPixel(longitude,
                                                              zoomLevel)
                                 + (column - mapX + numberOfColumns / 2 - 1) *
                                 256;
                        yPixel = currentGpsLocation.getYPixel(latitude,
                                                              zoomLevel)
                                 + (row - mapY + numberOfRows / 2 - 1) * 256;

                        view->showLocationInfo(xPixel, yPixel, zoomLevel,
                                               infoText, infoType);

                    }

                }

            }

        }

    }
}

//! Handle a failed download.
void QtMapper::handleFailedDownload(QString error)
{
    const int statusBarMessageTimeout = 5000;
    const QString errorRequestAborted(tr("Request aborted"));
    QString errorMessage(tr("Failed to download image data:\n"));
    errorMessage.append(error);

    // An http error has occurred.
    if (error != errorRequestAborted) {
        statusBar()->showMessage(errorMessage, statusBarMessageTimeout);
    }
}

//! Handle pending downloads.
void QtMapper::handleDownloadQueue()
{
    if (!downloadInProgress) {
        if (!downloadQueue.isEmpty()) {
            Location location = downloadQueue.pop();
            qint32 zoom = location.getZoomLevel();

            md->downloadMapTile(repositories.at(activeRepository), zoom,
                                location.getMapX(), location.getMapY(),
                                location.getScreenX(), location.getScreenY());
            downloadInProgress = true;

            if (!view->getNetworkActivityIndicator()->isActive()) {
                showDownloadIndicator();
            }
        }
    }
}

//! Cancel all downloads.
void QtMapper::clearDownloadQueue()
{
    if (!downloadQueue.empty()) {
        downloadQueue.clear();
    }

    searchView->stop();
    md->abortDownload();
    downloadInProgress = false;
    hideDownloadIndicator();
}

//! Zoom in the map view.
void QtMapper::zoomIn()
{
    qint32 zoomLevel = currentMapLocation.getZoomLevel();

    if (zoomLevel < maxZoomLevel) {

        zoomLevel++;
        zoom(zoomLevel);

        goToLocation(centerOfScreen.x(), centerOfScreen.y());
    }
}

//! Zoom out the map view.
void QtMapper::zoomOut()
{
    qint32 zoomLevel = currentMapLocation.getZoomLevel();

    if (zoomLevel > minZoomLevel) {

        zoomLevel--;
        zoom(zoomLevel);

        goToLocation(centerOfScreen.x(), centerOfScreen.y());
    }
}

//! Update view center pixels and their coordinates
void QtMapper::setViewCenterCoordinates()
{
    qint32 zoomLevel = currentMapLocation.getZoomLevel();

    viewCenterXPixel = view->horizontalScrollBar()->value() + view->width() / 2;
    viewCenterYPixel = view->verticalScrollBar()->value() + view->height() / 2;

    UnitConverter *converter = new UnitConverter;

    centerOfScreen = converter->coordinatesFromPixels(
                         zoomLevel,
                         upperLeftCornerMapXIndex,
                         upperLeftCornerMapYIndex,
                         viewCenterXPixel,
                         viewCenterYPixel);

    delete converter;
}

//! zoom in and out use this method
void QtMapper::zoom(qint32 zoomLevel)
{
    clearDownloadQueue();

    currentGpsLocation.setZoomLevel(zoomLevel);

    qint32 newGpsX = currentMapLocation
                     .getXTile(currentGpsLocation.getLongitude(), zoomLevel);
    qint32 newGpsY = currentMapLocation
                     .getYTile(currentGpsLocation.getLatitude(), zoomLevel);

    currentGpsLocation.setMapX(newGpsX);
    currentGpsLocation.setMapY(newGpsY);

    currentMapLocation.setZoomLevel(zoomLevel);

    qint32 newMapX = currentMapLocation
                     .getXTile(currentMapLocation.getLongitude(), zoomLevel);
    qint32 newMapY = currentMapLocation
                     .getYTile(currentMapLocation.getLatitude(), zoomLevel);

    currentMapLocation.setMapX(newMapX);
    currentMapLocation.setMapY(newMapY);

    refreshMapView();

}

//! Pan the map view up.
bool QtMapper::panUp()
{
    qint32 y = currentMapLocation.getMapY();

    if (y > 0 && currentMapLocation.getZoomLevel() > 1) {
        clearDownloadQueue();

        currentMapLocation.setMapY(y - 1);
        currentMapLocation
        .setLatitude(currentMapLocation
                     .getLatitude(currentMapLocation.getMapY(),
                                  currentMapLocation.getZoomLevel()));
        refreshMapView();

        return true;
    }
    else {
        return false;
    }
}

//! Pan the map view down.
bool QtMapper::panDown()
{
    qint32 y = currentMapLocation.getMapY();

    if (y < pow(2, currentMapLocation.getZoomLevel()) - 1 &&
            currentMapLocation.getZoomLevel() > 1) {
        clearDownloadQueue();

        currentMapLocation.setMapY(y + 1);
        currentMapLocation
        .setLatitude(currentMapLocation
                     .getLatitude(currentMapLocation.getMapY(),
                                  currentMapLocation.getZoomLevel()));
        refreshMapView();

        return true;
    }
    else {
        return false;
    }
}

//! Pan the map view to the left.
bool QtMapper::panLeft()
{
    qint32 x = currentMapLocation.getMapX();

    if (x > 0 && currentMapLocation.getZoomLevel() > 1) {
        clearDownloadQueue();

        currentMapLocation.setMapX(x - 1);
        currentMapLocation
        .setLongitude(currentMapLocation
                      .getLongitude(currentMapLocation.getMapX(),
                                    currentMapLocation.getZoomLevel()));
        refreshMapView();

        return true;
    }
    else {
        return false;
    }
}

//! Pan the map view to the right.
bool QtMapper::panRight()
{
    qint32 x = currentMapLocation.getMapX();

    if (x < pow(2, currentMapLocation.getZoomLevel()) - 1 &&
            currentMapLocation.getZoomLevel() > 1) {
        clearDownloadQueue();

        currentMapLocation.setMapX(x + 1);
        currentMapLocation
        .setLongitude(currentMapLocation
                      .getLongitude(currentMapLocation.getMapX(),
                                    currentMapLocation.getZoomLevel()));
        refreshMapView();

        return true;
    }
    else {
        return false;
    }
}

//! Centers the view on the current GPS location.
void QtMapper::goToCurrentGpsLocation()
{
    if (currentGpsLocation.getLatitude() != invalidValue) {

        currentGpsLocation.setZoomLevel(currentMapLocation.getZoomLevel());
        currentMapLocation = currentGpsLocation;
        clearDownloadQueue();

        refreshMapView();

        centerOnGpsMarker();
    }
}

//! The latitude received from the GPS device has changed.
void QtMapper::gpsLatitudeChanged(qreal latitude)
{
    currentGpsLocation.setLatitude(latitude);
    currentGpsLocation
    .setMapY(currentGpsLocation
             .getYTile(currentGpsLocation.getLatitude(),
                       currentGpsLocation.getZoomLevel()));

    if (gpsFixQuality > 0) {

        view->removeGpsMarker();
        drawMarker();

        if (centerOnLocationAct->isChecked()) {
            centerOnGpsMarker();
        }
    }
}

//! The longitude received from the GPS device has changed.
void QtMapper::gpsLongitudeChanged(qreal longitude)
{
    currentGpsLocation.setLongitude(longitude);
    currentGpsLocation
    .setMapX(currentGpsLocation
             .getXTile(currentGpsLocation.getLongitude(),
                       currentGpsLocation.getZoomLevel()));

    if (gpsFixQuality > 0) {

        view->removeGpsMarker();
        drawMarker();

        if (centerOnLocationAct->isChecked()) {
            centerOnGpsMarker();
        }
    }
}

//! Some other value (speed etc.) received from the GPS device has changed.
void QtMapper::gpsOtherValueChanged(const QString name, qreal value)
{
    if(name == "speed")
    {
        view->setNewSpeed(value) ;
    }
}

//! Displays the satellite details dialog.
void QtMapper::gpsDetails()
{
    SatelliteDetailsDialog *sd = new SatelliteDetailsDialog(gps->getGpsData(),
            gps->getGpsPoint(),
            this);
    sd->exec();
    delete sd;
}

//! Displays the about dialog
void QtMapper::about()
{
    AboutDialog *about = new AboutDialog(this);
    about->exec();
}

//! Displays the help dialog
void QtMapper::help()
{
    HelpDialog *help = new HelpDialog(this);
    help->exec();
}

//! Displays the preferences dialog
void QtMapper::preferences()
{
    PreferencesDialog *preferences;

    preferences = new PreferencesDialog(this,
                                        sizeLimitForMapImages,
                                        amountOfImagesToRemove,
                                        mapImageDirectory,
                                        repositories,
                                        activeRepository,
                                        view->getRouteLineColor(),
                                        view->getRouteLineThickness(),
                                        routeZoomLevel,
                                        maxZoomLevel,
                                        advancedMode);

    int response = preferences->exec();

    if (response == QDialog::Accepted) {
        //Preferences of POI alarm
        distanceToAlarm = preferences->getDistanceOfPoiAlarm();
        countNearestPois = preferences->getMaxCountOfPois();

        qint64 newSizeLimit = preferences->getMaxImageSpace();

        /* the new size limit for images is set in megabytes in the spinBox,
         * so it must be converted to bytes */
        newSizeLimit = newSizeLimit * 1024 * 1024;

        // check if the limit was changed and set new limit if needed
        if (newSizeLimit != sizeLimitForMapImages) {
            // if new limit is smaller than the old...
            if (newSizeLimit < sizeLimitForMapImages) {
                // ...check the space taken by images and remove some if needed
                limitImageCache(newSizeLimit);
            }

            sizeLimitForMapImages = newSizeLimit;
        }

        amountOfImagesToRemove = preferences->getImageRemoveAmount();

        QString newMapImageDirectory = preferences->getNewDirectory();
        if (newMapImageDirectory != mapImageDirectory) {

            QDir oldDirectory(mapImageDirectory);
            QFileInfoList mapImageFileList = oldDirectory.entryInfoList();

            QFile fileMover(this);
            QString fileName;

            for (int i = 0; i < mapImageFileList.size(); i++) {

                fileName = mapImageFileList.at(i).fileName();

                if ((fileName != ".") && (fileName != "..")) {

                    QString oldFileName(fileName);
                    oldFileName.prepend(mapImageDirectory);

                    QString newFileName(fileName);
                    newFileName.prepend(newMapImageDirectory);

                    if (fileMover.copy(oldFileName, newFileName)) {
                        fileMover.remove(oldFileName);
                    }
                }
            }

            oldDirectory.rmdir(mapImageDirectory);

            mapImageDirectory = newMapImageDirectory;

            if (db->connected()) {
                db->closeConnection();
            }

            delete db;

            db = new MapDatabase(repositories.at(activeRepository),
                                 mapImageDirectory);
            db->createConnection();
            db->changeDatabaseDirectory(mapImageDirectory);
        }

        activeRepository = preferences->getActiveRepository();

        view->setRouteLineColor(preferences->getRouteLineColor());
        view->setRouteLineThickness(preferences->getRouteLineThickness());

        routeZoomLevel = preferences->getRouteZoomlevel();

        advancedMode = preferences->getAdvancedMode();

        if (!repositories.isEmpty()) {
            mapImageDownloadingAllowed = true;
        }
        else {
            mapImageDownloadingAllowed = false;
        }

        saveSettings();
        refreshMapView();
    }

    delete preferences;
}

//! The mouse wheel was moved in the map view area.
void QtMapper::mouseWheelMovedInMapView(QGraphicsSceneWheelEvent *e)
{
    /*
     * If the wheel delta value is negative, the wheel
     * has been moved towards the user. If it's positive,
     * the wheel has been moved away from the user.
     */
    if (e->delta() < 0) {
        zoomOut();
    }
    else {
        zoomIn();
    }
}

//! Opens the "Go to location" dialog.
void QtMapper::openGoToLocationDialog()
{
    const QString dialogTitle("Go to location");

    QDialog *dialog = new QDialog(this);
    dialog->setModal(true);
    dialog->setWindowTitle(dialogTitle);

    QGridLayout *layout = new QGridLayout();
    dialog->setLayout(layout);

    QLabel *latLabel = new QLabel(tr("Latitude"));
    layout->addWidget(latLabel, 0, 0);

    QLineEdit *latEdit = new QLineEdit();
    latEdit->setValidator(new QDoubleValidator(minLatitude, maxLatitude,
                          maxDecimals, latEdit));
    layout->addWidget(latEdit, 0, 1);

    QLabel *lonLabel = new QLabel(tr("Longitude"));
    layout->addWidget(lonLabel, 1, 0);

    QLineEdit *lonEdit = new QLineEdit();
    lonEdit->setValidator(new QDoubleValidator(minLongitude, maxLongitude,
                          maxDecimals, lonEdit));
    layout->addWidget(lonEdit, 1, 1);

    QDialogButtonBox *box = new QDialogButtonBox(QDialogButtonBox::Ok |
            QDialogButtonBox::Cancel);
    connect(box, SIGNAL(accepted()), dialog, SLOT(accept()));
    connect(box, SIGNAL(rejected()), dialog, SLOT(reject()));
    layout->addWidget(box, 2, 0, 1, 2);

    if (dialog->exec() == QDialog::Accepted) {
        qreal lon = lonEdit->text().toDouble();
        qreal lat = latEdit->text().toDouble();

        if (lon < minLongitude) {
            lon = minLongitude;
        }
        else if (lon > maxLongitude) {
            lon = maxLongitude;
        }

        if (lat < minLatitude) {
            lat = minLatitude;
        }
        else if (lat > maxLatitude) {
            lat = maxLatitude;
        }

        goToLocation(lon, lat);
    }
}

//! Centers the view on the location at the specified coordinates.
void QtMapper::goToLocation(qreal longitude, qreal latitude)
{
    qint32 zoom = currentMapLocation.getZoomLevel();

    currentMapLocation.setLatitude(latitude);
    currentMapLocation.setLongitude(longitude);
    currentMapLocation.setMapX(currentMapLocation.getXTile(longitude, zoom));
    currentMapLocation.setMapY(currentMapLocation.getYTile(latitude, zoom));

    clearDownloadQueue();
    refreshMapView();

    view->centerOn(currentMapLocation.getXPixel(longitude, zoom) +
                   mapTileWidth * (int)(mapTileColumns / 2),
                   currentMapLocation.getYPixel(latitude, zoom) +
                   mapTileHeight * (int)(mapTileRows / 2));
}

//! Limits image cache size
void QtMapper::limitImageCache(qint64 byteLimit)
{
    // variable for storing the space taken by the map images
    qint64 totalSize = 0;

    // Make a QDir object from the map image directory
    QDir directory(mapImageDirectory);

    // set the directory sorting so that oldest images come first
    directory.setSorting(QDir::Time | QDir::DirsLast | QDir::Reversed);

    // Make a list of the contents
    QFileInfoList mapImageFiles = directory.entryInfoList();

    // check the current size of images directory
    for (int i = 0; i < mapImageFiles.size(); i++) {
        totalSize = totalSize + mapImageFiles.at(i).size();
    }

    int fileIndexer = 0;

    while (totalSize > byteLimit) {
        // remove the amount of space taken by the oldest file from totalSize
        totalSize = totalSize - mapImageFiles.at(fileIndexer).size();

        // then remove the file
        directory.remove(mapImageFiles.at(fileIndexer).fileName());

        fileIndexer++;
    }
}

//! Saves user settings
void QtMapper::saveSettings()
{
    settings = new QSettings(settingOrganizationName, settingApplicationName);

    // Clear all existing settings before writing the current values.
    settings->clear();

    // Write the settings.
    settings->setValue(settingsizeLimitForMapImages, sizeLimitForMapImages);
    settings->setValue(settingAmountOfImagesToRemove, amountOfImagesToRemove);
    settings->setValue(settingMapImageDirectory, mapImageDirectory);
    settings->setValue(settingCountNearestPois, countNearestPois);
    settings->setValue(settingDistanceToAlarm, distanceToAlarm);

    settings->beginWriteArray(settingMapRepositories);

    for (int i = 0; i < repositories.size(); ++i) {
        settings->setArrayIndex(i);
        settings->setValue(settingRepositoryName, repositories.at(i).getName());
        settings->setValue(settingRepositoryUrl, repositories.at(i).getUrl());
        settings->setValue(settingRepositoryImageNameFormat,
                           repositories.at(i).getImageNameFormat());
        settings->setValue(settingRepositoryDatabaseFileName,
                           repositories.at(i).getDatabase());
        settings->setValue(settingActiveRepository,
                           repositories.at(i).getType());
    }

    settings->endArray();

    settings->setValue(settingActiveRepository, activeRepository);
    settings->setValue(settingRouteLineColor, view->getRouteLineColor());
    settings->setValue(settingRouteLineThickness,
                       view->getRouteLineThickness());

    settings->setValue(settingRouteZoomLevel, routeZoomLevel);

    settings->setValue(settingAdvancedMode, advancedMode);

    settings->beginWriteArray(settingTrackPoints);

    if (trackPoints.count() > 1) {
        for (int i = 0; i < trackPoints.count(); ++i) {
            settings->setArrayIndex(i);
            settings->setValue(settingTrackPointLatitude, trackPoints[i].getLatitude());
            settings->setValue(settingTrackPointLongitude, trackPoints[i].getLongitude());
            settings->setValue(settingTrackPointTime, trackPoints[i].getTime());
        }
        settings->endArray();
    }
    settings->setValue(settingLastLongitudeInView, centerOfScreen.x());
    settings->setValue(settingLastLatitudeInView, centerOfScreen.y());
    settings->setValue(settingLastZoomLevel, currentMapLocation.getZoomLevel());

    delete settings;
}

//! Loads user settings
void QtMapper::loadSettings()
{
    settings = new QSettings(settingOrganizationName, settingApplicationName);

    if (settings->contains(settingsizeLimitForMapImages)) {
        sizeLimitForMapImages = settings
                                ->value(settingsizeLimitForMapImages).toInt();
    }
    else {
        sizeLimitForMapImages = defaultSizeLimitForMapImages;
    }

    if (settings->contains(settingAmountOfImagesToRemove)) {
        amountOfImagesToRemove = settings
                                 ->value(settingAmountOfImagesToRemove).toInt();
    }
    else {
        amountOfImagesToRemove = defaultAmountOfImagesToRemove;
    }

    if (settings->contains(settingMapImageDirectory)) {
        mapImageDirectory = settings
                            ->value(settingMapImageDirectory).toString();
    }
    else {
        mapImageDirectory = defaultMapImageDirectory;
    }

    // check if the image directory exists...
    QDir *imageDirectory = new QDir(mapImageDirectory);

    // ...and if not, create it
    if (!imageDirectory->exists()) {

        delete imageDirectory;

        QDir *imageDirectory = new QDir();
        if (imageDirectory->mkdir(mapImageDirectory)) {
            // Map image directory doesn't exist, creating...
        }

        else {
            /* Could not create image directory,
               setting /tmp/ as image directory */
            mapImageDirectory = settingTemporaryImageDirectory;
        }
    }

    int arraySize = settings->beginReadArray(settingMapRepositories);

    for (int i = 0; i < arraySize; ++i) {
        settings->setArrayIndex(i);

        MapRepository mr;
        mr.setName(settings->value(settingRepositoryName).toString());
        mr.setUrl(settings->value(settingRepositoryUrl).toString());
        mr.setImageNameFormat(settings
                              ->value(settingRepositoryImageNameFormat)
                              .toString());
        mr.setDatabase(settings
                       ->value(settingRepositoryDatabaseFileName).toString());
        mr.setType(RepositoryType(settings
                                  ->value(settingActiveRepository).toInt()));
        repositories.append(mr);
    }

    settings->endArray();

    if (settings->contains(settingActiveRepository)) {
        activeRepository = settings->value(settingActiveRepository).toInt();
    }
    else {
        activeRepository = defaultActiveRepository;
    }

    if (settings->contains(settingRouteLineColor)) {
        view->setRouteLineColor(settings
                                ->value(settingRouteLineColor)
                                .value<QColor> ());
    }
    else {
        view->setRouteLineColor(defaultRouteLineColor);
    }

    if (settings->contains(settingRouteLineThickness)) {
        view->setRouteLineThickness(settings
                                    ->value(settingRouteLineThickness).toInt());
    }
    else {
        view->setRouteLineThickness(defaultRouteLineThickness);
    }

    if (settings->contains(settingRouteZoomLevel)) {
        routeZoomLevel = settings->value(settingRouteZoomLevel).toInt();
    }
    else {
        routeZoomLevel = defaultRouteZoomLevel;
    }

    //! Settings of POI alarm
    if (settings->contains(settingCountNearestPois)) {
        countNearestPois = settings->value(settingCountNearestPois).toInt();
    } 
    else {
        countNearestPois = defaultCountNearestPois;
    }

    if (settings->contains(settingDistanceToAlarm)) {
        distanceToAlarm = settings->value(settingDistanceToAlarm).toDouble();
    }
    else {
        distanceToAlarm = defaultDistanceToAlarm;
    }

    if (settings->contains(settingAdvancedMode)) {
        advancedMode = settings->value(settingAdvancedMode).toBool();
    } 
    else {
        advancedMode = defaultAdvancedMode;
    }

    int trackArraySize = settings->beginReadArray(settingTrackPoints);

    for (int i = 0; i < trackArraySize; ++i) {
        settings->setArrayIndex(i);
        TrackPoint point;
        point.setLatitude(settings->value(settingTrackPointLatitude).toDouble());
        point.setLongitude(settings->value(settingTrackPointLongitude).toDouble());
        QTime time = QTime::fromString(settings->value(settingTrackPointTime).toString());
        point.setTime(time);
        trackPoints.append(point);
    }
    settings->endArray();

    if (trackPoints.count() > 0) {
        showTrackAct->setEnabled(true);
        saveTrackAct->setEnabled(true);
        clearTrackAct->setEnabled(true);
    }

    delete imageDirectory;
    delete settings;
}

//! Sets the mapview to location that was viewed before last shutdown
void QtMapper::goToSavedLocation()
{
    settings = new QSettings(settingOrganizationName, settingApplicationName);

    if (settings->contains(settingLastLongitudeInView) &&
        settings->contains(settingLastLatitudeInView) &&
        settings->contains(settingLastZoomLevel)) {

        zoom(settings->value(settingLastZoomLevel).toInt());
        goToLocation(settings->value(settingLastLongitudeInView).toDouble(),
                     settings->value(settingLastLatitudeInView).toDouble());

        setViewCenterCoordinates();
    }

    delete settings;

}

//! Opens Show POI Categories Dialog
void QtMapper::showPoiCategoriesDialog()
{
    PoiCategoriesDialog *pc = new PoiCategoriesDialog(*this, this);
   
    pc->exec();
}

//! Opens Browse POIs Dialog
void QtMapper::showBrowsePoisDialog()
{
    bool validGpsDataAvailable = currentGpsLocation.isLocationDataValid();

    BrowsePoisDialog *bp;
    bp = new BrowsePoisDialog(*this, currentGpsLocation.getLatitude(),
                              currentGpsLocation.getLongitude(),
                              limitResultOfPois,validGpsDataAvailable, this);
    bp->exec();
}

//! Called when route data has finished downloading.
void QtMapper::routeDownloadComplete()
{
    hideDownloadIndicator();

    // Pass a pointer to the data to the map view.
    view->setRouteData(currentRouteData);

    // Tell the view to start drawing the route stuff.
    view->enableRouteDisplay(true);
    drawRouteLines();

    clearRouteAct->setEnabled(true);

    /*
     * Set the current map location to the route start point
     * with a suitable zoom level.
     */

    zoom(routeZoomLevel);

    goToLocation(currentRouteData->getWaypoints().at(0)->getLongitude(),
                 currentRouteData->getWaypoints().at(0)->getLatitude());

    routeTimer->start(routeCheckInterval);
    refreshMapView();
    setViewCenterCoordinates();

}

//! Called when route data downloading has failed.
void QtMapper::routeDownloadFailed(int errorCode,
                                   const QString &errorDescription)
{
    hideDownloadIndicator();

    QString errorMessage;

    if (errorCode == QNetworkReply::NoError) {
        errorMessage.append((tr("Unable to find directions for the route ")));
        errorMessage.append(routeDownloader->getFromAddress());
        errorMessage.append(tr(" - "));
        errorMessage.append(routeDownloader->getToAddress());
        errorMessage.append(tr(".\n\nPlease check the start and destination "
                               "addresses and try again."));
    }
    else if (errorCode == QNetworkReply::ProtocolUnknownError) {
        return;
    }
    else {
        errorMessage.append(errorDescription);
    }

    QMessageBox::warning(this, tr("Route Download Failed"), errorMessage);
}

//! Imports POIs from XML file
void QtMapper::showImportPoiDialog()
{
    QString fileName = QFileDialog::getOpenFileName(this,
                       tr("Open Gpx File"),
                       QDir::currentPath());

    if (fileName.isEmpty()) {
        return;
    }

    QXmlSimpleReader reader;
    pois = new QVector<Poi>;
    GpxHandler  *handler = new GpxHandler(*pois);
    reader.setContentHandler(handler);
    reader.setErrorHandler(handler);

    QFile file(fileName);

    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, tr("GPX Files"),
                             tr("Cannot read file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
        return;
    }

    QXmlInputSource xmlInputSource(&file);

    if (reader.parse(xmlInputSource)) {
        statusBar()->showMessage(tr("File loaded"), statusBarTimeFileLoaded);
    }
    else {
        statusBar()->showMessage(tr("Wrong file type"),
                                 statusBarTimeFileLoaded);
    }

    file.close();

    if (pois->count() > 0) {
        ImportPoisDialog ip(*pois, *this, this);

        if (ip.exec() == QDialog::Accepted) {
            statusBar()->showMessage(tr("POIs saved"), statusBarTimeFileSaved);
        }
    }

    delete pois;
    delete handler;
}

//! Shows the download progress bar when there are ongoing downloads.
void QtMapper::showDownloadIndicator()
{
    view->showNetworkActivityIndicator(true);
    cancelDownloadsAct->setEnabled(true);
}


//! Hides the download progress bar when all downloads are complete.
void QtMapper::hideDownloadIndicator()
{
    view->showNetworkActivityIndicator(false);
    cancelDownloadsAct->setEnabled(false);
}

//! Called when mapview is scrolled to right edge
void QtMapper::scrolledToRightEdge()
{
    if (panRight()) {
        view->horizontalScrollBar()
        ->setValue(view->horizontalScrollBar()->value() - mapTileWidth);
    }
}

//! Called when mapview is scrolled to left edge
void QtMapper::scrolledToLeftEdge()
{
    if (panLeft()) {
        view->horizontalScrollBar()
        ->setValue(view->horizontalScrollBar()->value() + mapTileWidth);
    }
}

//! Called when mapview is scrolled to top edge
void QtMapper::scrolledToTopEdge()
{
    if (panUp()) {
        view->verticalScrollBar()
        ->setValue(view->verticalScrollBar()->value() + mapTileHeight);
    }
}

//! Called when mapview is scrolled to bottom edge
void QtMapper::scrolledToBottomEdge()
{
    if (panDown()) {
        view->verticalScrollBar()
        ->setValue(view->verticalScrollBar()->value() - mapTileHeight);
    }
}

//! Shows Dialog search location.
void QtMapper::openSearchLocationDialog()
{
    searchLocation = new SearchLocationDialog(this);
    connect(searchLocation->pushButton, SIGNAL(clicked()),
            this, SLOT(goToSearchedLocation()));
    connect(searchLocation, SIGNAL(rejected()),
            this, SLOT(clearButtonState()));
    searchLocation->show();
    goToAddressAct->setEnabled(false);
}

//!called when location download is complete
void QtMapper::locationDownloadComplete()
{
    hideDownloadIndicator();

    if (searchLocation->lineEditStreetName->text() != "") {
        zoom(16);
    }
    else if (searchLocation->lineEditCityName->text() != "") {
        zoom(11);
    }
    else {
        zoom(4);
    }

    if (searchLocationData->getLongitude() != invalidValue ||
        searchLocationData->getLatitude() != invalidValue) {
        goToLocation(searchLocationData->getLongitude(),
                     searchLocationData->getLatitude());

        QString infoText(searchLocationData->getAddress());
        refreshMapView();
        drawInfoMarker(searchLocationData->getLatitude(),
                       searchLocationData->getLongitude(),
                       infoText, infoTypeSearchedLocation);
    }
    
    clearButtonState();
    setViewCenterCoordinates();
}

//! Download Location specified in search location dialog
void QtMapper::goToSearchedLocation()
{
    QString param1(searchLocation->lineEditCountryName->text());
    QString param2(searchLocation->lineEditCityName->text());
    QString param3(searchLocation->lineEditStreetName->text());

    if (searchLocationData != 0) {
        disconnect(searchLocationData, 0, 0, 0);
        delete searchLocationData;
    }

    if (searchLocationDownloader!= 0) {
        disconnect(searchLocationDownloader, 0, 0, 0);
    }

    searchLocationData = new SearchLocationData(this);
    searchLocationDownloader = new SearchLocationDownloader(searchLocationData,
                                                            searchView);
    searchView->initializeLocationSearch(searchLocationDownloader);
    connect(searchLocationDownloader, SIGNAL(completed()),
            this, SLOT(locationDownloadComplete()));
    connect(searchLocationDownloader, SIGNAL(failed(int, const QString &)),
            this, SLOT(locationDownloadFailed(int, const QString &)));

    searchLocationDownloader->downloadData(param1, param2, param3);
    searchLocation->hide();
}

//! show error string if dowload fails
void QtMapper::locationDownloadFailed(int errorCode,
                                      const QString &errorDescription)
{
    hideDownloadIndicator();

    QString errorMessage;

    if (errorCode == QNetworkReply::NoError) {
        errorMessage.append(tr("Unable to find address."));
    }
    else {
        errorMessage.append(errorDescription);
    }

    QMessageBox::information(this, tr("Location Search Failed"),
                             errorMessage, QMessageBox::Ok);

    clearButtonState();
}

//! Handle open search location buttons states
void QtMapper::clearButtonState()
{
    searchLocation->close();
    goToAddressAct->setEnabled(true);
}

//! One of the actions in the tool action group was triggered.
void QtMapper::toolActionsClicked(QAction *act)
{
    if (act == moveToolAct) {
        view->setDragMode(QGraphicsView::ScrollHandDrag);
        mouseZoomInUse = false;
        downloadMapImagesSelectedWithMouse = false;
        showQuickMenuAct->setEnabled(true);

        if (userWantsQuickMenu) {
            showQuickMenuAct->setChecked(true);
        }

    }
    else if (act == zoomToolAct) {
        view->setDragMode(QGraphicsView::RubberBandDrag);
        mouseZoomInUse = true;
        downloadMapImagesSelectedWithMouse = false;
        showQuickMenuAct->setEnabled(false);

        if (userWantsQuickMenu) {
            showQuickMenuAct->setChecked(false);
            userWantsQuickMenu = true;
        }
        else {
            showQuickMenuAct->setChecked(false);
        }

    }
    else if (act == cancelDownloadsAct) {
        clearDownloadQueue();
    }
}


//! Ask the user if he wants to load the default repositories.
void QtMapper::loadDefaultRepositories()
{
    if (repositories.isEmpty()) {
        if (advancedMode) {
            int res = QMessageBox::warning(this, tr("Repository List Empty"),
                                           tr("There are no saved "
                                              "repositories.\nDo you want "
                                              "to load the default "
                                              "repositories?"),
                                           QMessageBox::Yes | QMessageBox::No,
                                           QMessageBox::Yes);

            if (res == QMessageBox::Yes) {
                resetRepositories();
            }
        }
        else {
            resetRepositories();
        }
    }
}


//! Resets the repository list.
void QtMapper::resetRepositories()
{
    repositories.append(openStreetMap);

    repositories.append(googleMapsStreet);

    activeRepository = 0;
    mapImageDownloadingAllowed = true;

    selectTiles();
    centerOnGpsMarker();
}


//! listens to mouse button presses in mapscene
void QtMapper::mouseButtonPressedInMapScene(QGraphicsSceneMouseEvent *e)
{
    if (mouseZoomInUse || downloadMapImagesSelectedWithMouse ||
        coordinateToolAct->isChecked()) {

        mousePressScreenCoordinates.setX( (int)e->scenePos().x() );
        mousePressScreenCoordinates.setY( (int)e->scenePos().y() );


        UnitConverter *conv = new UnitConverter();

        mouseSelectionStartPoint = conv
                                   ->coordinatesFromPixels(currentMapLocation
                                        .getZoomLevel(),
                                        upperLeftCornerMapXIndex,
                                        upperLeftCornerMapYIndex,
                                        mousePressScreenCoordinates.x(),
                                        mousePressScreenCoordinates.y());

        delete conv;
    }

    if (coordinateToolAct->isChecked()) {
        view->setLonLatBoxValues(mouseSelectionStartPoint.x(),
                                 mouseSelectionStartPoint.y());
    }
}


//! listens to mouse button releases in mapscene
void QtMapper::mouseButtonReleasedInMapScene(QGraphicsSceneMouseEvent *e)
{
    if (mouseZoomInUse || downloadMapImagesSelectedWithMouse ) {

        mouseReleaseScreenCoordinates.setX( (int)e->scenePos().x() );
        mouseReleaseScreenCoordinates.setY( (int)e->scenePos().y() );

        UnitConverter *conv = new UnitConverter();

        mouseSelectionEndPoint = conv
                                 ->coordinatesFromPixels(currentMapLocation
                                     .getZoomLevel(),
                                     upperLeftCornerMapXIndex,
                                     upperLeftCornerMapYIndex,
                                     mouseReleaseScreenCoordinates.x(),
                                     mouseReleaseScreenCoordinates.y());

        delete conv;
    }

    if (mouseZoomInUse && !centerOnLocationAct->isChecked()) {
        mouseZoom();
    }
    else if (downloadMapImagesSelectedWithMouse) {
        mapTileDownloadDialog->setFromLatitude(mouseSelectionStartPoint.y());
        mapTileDownloadDialog->setToLatitude(mouseSelectionEndPoint.y());
        mapTileDownloadDialog->setFromLongitude(mouseSelectionStartPoint.x());
        mapTileDownloadDialog->setToLongitude(mouseSelectionEndPoint.x());

        view->setDragMode(QGraphicsView::ScrollHandDrag);
        mouseZoomInUse = false;
        downloadMapImagesSelectedWithMouse = false;

        moveToolAct->setChecked(true);
        showQuickMenuAct->setEnabled(true);
        if (userWantsQuickMenu) {
            showQuickMenuAct->setChecked(true);
        }

        mapTileDownloadDialog->show();

    }
    else if(view->dragMode() == QGraphicsView::ScrollHandDrag) {
        setViewCenterCoordinates();
    }
}


//! slot for mouse zooming feature
void QtMapper::mouseZoom()
{
    // first check the size of area surrounded with rubberband...
    qreal latitudeDifference = mouseSelectionEndPoint.y() -
                               mouseSelectionStartPoint.y();
    qreal longitudeDifference = mouseSelectionEndPoint.x() -
                                mouseSelectionStartPoint.x();

    // if the area isn't big enough, don't zoom
    if (fabs(latitudeDifference) < mouseZoomLatitudeThreshold ||
            fabs(longitudeDifference) < mouseZoomLongitudeThreshold) {
        return;
    }
    if (abs(mouseReleaseScreenCoordinates.x() - mousePressScreenCoordinates.x())
        < mouseZoomPixelThreshold ||
        abs(mouseReleaseScreenCoordinates.y() - mousePressScreenCoordinates.y())
        < mouseZoomPixelThreshold ) {
        return;
    }

    int zoomLevel;

    if (longitudeDifference > latitudeDifference) {
        for (zoomLevel = 0; zoomLevel <= maxZoomLevel; zoomLevel++) {
            qreal longitudesInATile = 360 / pow(2, zoomLevel);
            if (longitudesInATile * mapTileColumns < longitudeDifference) {
                break;
            }
        }
    }

    else {
        for (zoomLevel = 0; zoomLevel <= maxZoomLevel; zoomLevel++) {
            qreal latitudesInATile = 180 / pow(2, zoomLevel);
            if (latitudesInATile * mapTileRows < latitudeDifference) {
                break;
            }
        }
    }

    // reduce one from zoomlevel to fit the whole requested area to mapscene
    zoomLevel--;

    qreal zoomCenterX = mouseSelectionStartPoint.x() + longitudeDifference / 2;
    qreal zoomCenterY = mouseSelectionStartPoint.y() + latitudeDifference / 2;

    currentMapLocation.setLongitude(zoomCenterX);
    currentMapLocation.setLatitude(zoomCenterY);

    zoom(zoomLevel);

    goToLocation(zoomCenterX, zoomCenterY);
}

// Open the route download dialog.
void QtMapper::openRouteDownloadDialog()
{
    centerOnLocationAct->setChecked(false);
    centerViewOnLocationToggled(false);

    DownloadRouteDialog *dialog = new DownloadRouteDialog(this);

    if (routeDownloader && !routeDownloader->getFromAddress().isEmpty()) {
        dialog->setFromAddress(routeDownloader->getFromAddress());
    }

    if (routeDownloader && !routeDownloader->getToAddress().isEmpty()) {
        dialog->setToAddress(routeDownloader->getToAddress());
    }

    if (dialog->exec() == QDialog::Accepted) {
        clearRoute();
        currentRouteData = new RouteData(this);
        routeDownloader = new RouteDownloader(currentRouteData, searchView);

        connect(routeDownloader, SIGNAL(completed()),
                this, SLOT(routeDownloadComplete()));
        connect(routeDownloader, SIGNAL(failed(int, const QString &)),
                this, SLOT(routeDownloadFailed(int, const QString &)));
        searchView->initializeRouteDownload(routeDownloader);
        routeDownloader->downloadData(dialog->getFromAddress(),
                                      dialog->getToAddress(),
                                      locale().name());
    }
}

//Draw markers for POIs to screen
void QtMapper::drawPoiMarkers()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    disconnect(view->getScene(), 
               SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
               view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));
  
    QVector<Poi> pois;
    poiDb->getAllPois(pois);
 
    //draw markers for pois
    for (QVector<Poi>::iterator currentItem = pois.begin();
         currentItem != pois.end(); ++currentItem) {

         QString infoText(currentItem->getLabel());
         infoText.append("\n");
         infoText.append(currentItem->getDescription());
         drawInfoMarker(currentItem->getLatitude(),
         currentItem->getLongitude(),
         infoText, infoTypePoi);
    }

    connect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));

}


//Adds default POI categories if there is no categories in database
void QtMapper::addDefaultPoiCategories()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    QVector<PoiCategory> categories;
    poiDb->getPoiCategories(categories);

    if (categories.count() == 0) {
            poiDb->insertDefaultCategories();
    }
}


//Called when POIs are downloaded from internet
void QtMapper::poiDownloadComplete()
{
    hideDownloadIndicator();
    showDownloadedPoisDialog(poiData->getDownloadedPois());
}

//! Called if the POI downloading failed.
void QtMapper::poiDownloadFailed(int errorCode, const QString &errorDescription)
{
    hideDownloadIndicator();

    QString errorMessage;

    if (errorCode == QNetworkReply::NoError) {
        errorMessage.append(tr("Couldn't find any POIs with the given "
                               "search terms."));
    }
    else {
        errorMessage.append(errorDescription);
    }

    QMessageBox::information(this, tr("POI Download Failed"),
                             errorMessage, QMessageBox::Ok);
}

// Shows download POIs dialog
void QtMapper::showDownloadPoisDialog()
{
    DownloadPoisDialog dialog(this);

    if (dialog.exec() == QDialog::Accepted) {
        if (poiData != 0) {
            disconnect(poiData, 0, 0, 0);
            delete poiData;
        }

        if (poiDownloader!= 0) {
            disconnect(poiDownloader, 0, 0, 0);
        }

        poiData = new PoiData(this);
        poiDownloader = new PoiDownloader(poiData, searchView);
        connect(poiDownloader, SIGNAL(completed()),
                this, SLOT(poiDownloadComplete()));
        connect(poiDownloader, SIGNAL(failed(int, const QString &)),
                this, SLOT(poiDownloadFailed(int, const QString &)));

        searchView->initializePoiDownload(poiDownloader);

        QString keyword(dialog.lineEditKeyWord->text());
        poiDownloader->downloadData(keyword);
    }
}

//! Displays the map image download dialog
void QtMapper::downloadMapImagesFromSelectedAreaDialog()
{
    mapTileDownloadDialog =  new MapTileDownloadDialog(minZoomLevel,
                                                       maxZoomLevel,
                                                       this);

    connect(mapTileDownloadDialog->getSelectAreaButton(), SIGNAL(clicked()),
            this, SLOT(selectAreaButtonClicked()));
    connect(mapTileDownloadDialog, SIGNAL(okButtonClicked()),
            this, SLOT(downloadThemTiles()));

    mapTileDownloadDialog->show();
}

//! Hides "Download map images" dialog, enables area selection with mouse
void QtMapper::selectAreaButtonClicked()
{
    mapTileDownloadDialog->hide();

    moveToolAct->setChecked(false);
    zoomToolAct->setChecked(false);

    showQuickMenuAct->setEnabled(false);
    if (userWantsQuickMenu) {
        showQuickMenuAct->setChecked(false);
        userWantsQuickMenu = true;
    }
    else {
        showQuickMenuAct->setChecked(false);
    }

    view->setDragMode(QGraphicsView::RubberBandDrag);

    mouseZoomInUse = false;
    downloadMapImagesSelectedWithMouse = true;
}

//! Dowloads map images from area set in mapTileDownloadDialog
void QtMapper::downloadThemTiles()
{
    mapTileDownloadDialog->hide();

    int startXIndex;
    int endXIndex;
    int startYIndex;
    int endYIndex;
    int zoomLevel;

    qreal fromLongitude = mapTileDownloadDialog->fromLongitude();
    qreal toLongitude = mapTileDownloadDialog->toLongitude();
    qreal fromLatitude = mapTileDownloadDialog->fromLatitude();
    qreal toLatitude = mapTileDownloadDialog->toLatitude();

    zoomLevel = mapTileDownloadDialog->selectedZoomLevel();

    startXIndex = currentMapLocation.getXTile(fromLongitude, zoomLevel);
    endXIndex = currentMapLocation.getXTile(toLongitude, zoomLevel);
    startYIndex = currentMapLocation.getYTile(fromLatitude, zoomLevel);
    endYIndex = currentMapLocation.getYTile(toLatitude, zoomLevel);

    int temp;

    if (startXIndex > endXIndex) {
        temp = startXIndex;
        startXIndex = endXIndex;
        endXIndex = temp;
    }

    if (startYIndex > endYIndex) {
        temp = startYIndex;
        startYIndex = endYIndex;
        endYIndex = temp;
    }

    int numberOfTilesToDownload = 0;

    if (db != NULL) {
        if (!db->connected()) {
            db->createConnection();
        }

        QString key;
        bool mapTileInDatabase;

        for (int rowIndex = startYIndex; rowIndex <= endYIndex; rowIndex++) {
            for (int columnIndex = startXIndex;
                    columnIndex <= endXIndex;
                    columnIndex++) {
                key = QString("/%1/%2/%3")
                      .arg(zoomLevel)
                      .arg(columnIndex)
                      .arg(rowIndex);

                mapTileInDatabase = db->findMapTile(key);

                if (!mapTileInDatabase) {
                    downloadQueue.push(Location(zoomLevel,
                                                columnIndex,
                                                rowIndex, 0, 0));
                    numberOfTilesToDownload++;
                }
            }
        }
    }

    else {
        QMessageBox::information(this, tr("Error"),
                                 tr("Could not connect to database."),
                                 QMessageBox::Ok);
        delete mapTileDownloadDialog;
        return;
    }

    if (numberOfTilesToDownload > 0) {
        int answer = QMessageBox::warning(this, tr("Confirmation"),
                                          tr("Attempt to download %1 map "
                                             "images?")
                                          .arg(numberOfTilesToDownload),
                                          QMessageBox::Yes | QMessageBox::No);

        if (answer == QMessageBox::Yes) {
            handleDownloadQueue();
        }
        else {
            clearDownloadQueue();
        }
    }

    else {
        QMessageBox::information(this, tr("Nothing to download"),
                                 tr("All maptiles in this area at the "
                                    "requested zoomlevel are already "
                                    "in the database."),
                                 QMessageBox::Ok);
    }
    mapTileDownloadDialog->close();
}

//! Shows downloaded POIs dialog
void QtMapper::showDownloadedPoisDialog(QVector<Poi> downloadedPois)
{
    DownloadedPoisDialog dialog(downloadedPois,
                                currentGpsLocation.getLatitude(),
                                currentGpsLocation.getLongitude(),
                                *this, this);
    dialog.exec();
}

//! Draw lines between the line points along the route.
void QtMapper::drawRouteLines()
{
    QPainterPath path;
    bool firstPoint = true;

    /*
     * Go though the list of the line points and construct the path
     * along which to draw the lines.
     */
    for (int i = 0; i < currentRouteData->getLinePoints().size(); i++) {
        int mapX = currentMapLocation.getMapX();
        int mapY = currentMapLocation.getMapY();
        LinePoint *lp = currentRouteData->getLinePoints().at(i);
        Location l;
        l.setLongitude(lp->getLongitude());
        l.setLatitude(lp->getLatitude());
        l.setZoomLevel(currentMapLocation.getZoomLevel());
        l.setMapX(l.getXTile(l.getLongitude(), l.getZoomLevel()));
        l.setMapY(l.getYTile(l.getLatitude(), l.getZoomLevel()));

        // Check if the current point is in the view.
        for (int row = mapY - mapTileRows / 2;
                row <= mapY + mapTileRows / 2;
                row++) {
            if (row == l.getMapY()) {
                for (int col = mapX - mapTileColumns / 2;
                        col <= mapX + mapTileColumns / 2;
                        col++) {
                    if (col == l.getMapX()) {
                        int x = l.getXPixel(l.getLongitude(), l.getZoomLevel())
                                + (col - mapX + mapTileColumns / 2) * 256;
                        int y = l.getYPixel(l.getLatitude(), l.getZoomLevel())
                                + (row - mapY + mapTileRows / 2) * 256;

                        if (firstPoint) {
                            path.moveTo(x, y);
                            firstPoint = false;
                        }
                        else {
                            path.lineTo(x, y);
                        }
                    }
                }
            }
        }
    }

    // Pass the completed path to the view and draw the route.
    view->drawRouteLine(path);
}

//! Checks if there POI near
void QtMapper::checkIsTherePoiNear()
{
      
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    if (lastLatitude == currentGpsLocation.getLatitude() &&
        lastLongitude == currentGpsLocation.getLongitude()) {
        return;	
    }
    else {
        lastLatitude = currentGpsLocation.getLatitude();
        lastLongitude = currentGpsLocation.getLongitude();

    }
    
    qreal distance;
    bool sound = false;
    bool draw = true;
    QVector<Poi> pois;

    poiDb->getPois(pois,currentGpsLocation.getLatitude(),
              currentGpsLocation.getLongitude(),
              countNearestPois, -1);
   
    for (qint32 i = alarmedPois.count() - 1; i >= 0; i--) {
        distance = distanceBetweenCoordinates(currentGpsLocation.
                                          getLatitude(),
                                          currentGpsLocation.getLongitude(),
                                          alarmedPois.at(i).getLatitude(),
                                          alarmedPois.at(i).getLongitude());
        
        if (distance > distanceToAlarm / 1000) {
            alarmedPois.remove(i);
        }
    }
    
    for (qint32 i = 0; i < pois.count(); i++) {
       distance = distanceBetweenCoordinates(currentGpsLocation.
                                         getLatitude(),
                                         currentGpsLocation.getLongitude(),
                                         pois.at(i).getLatitude(),
                                         pois.at(i).getLongitude());
        if (distance <= distanceToAlarm / 1000) {
            
           for (qint32 j = 0; j < alarmedPois.count(); j++) {
               if (pois.at(i).getLabel() == alarmedPois.at(j).getLabel()) {
                   draw = false;
               }
           }
        
           if (draw == true) {
               
               alarmedPois.append(pois.at(i));
               drawInfoMarker(pois.at(i).getLatitude(), pois.at(i).getLongitude(),
                              pois.at(i).getLabel(), infoTypePoi);
                                           
               disconnect(view->getScene(),
                        SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
                        view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));               
                               
               connect(view->getScene(),
                        SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
                        view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));
               sound = true; 
               
               draw = true;
           }
                          
        }
    }

    if (sound == true) {
        QApplication::beep();
    }
}

//! Change alarm state off or on
void QtMapper::changeAlarmState(bool checked)
{
    if (checked == true) {
        checkPoiTimer->start(checkingInterval);
        drawPoisAct->setEnabled(false);
        centerOnLocationAct->setChecked(true);
        centerViewOnLocationToggled(true);
    }
    else {

        checkPoiTimer->stop();
        alarmedPois.clear();
        lastLatitude = invalidValue;
        drawPoisAct->setEnabled(true);
        centerOnLocationAct->setChecked(false);
        centerViewOnLocationToggled(false);
        refreshMapView();

    }

}

//! Return distance between given parameters
qreal QtMapper::distanceBetweenCoordinates(qreal startLatitude,
                                           qreal startLongitude,
                                           qreal endLatitude,
                                           qreal endLongitude)
{

    /* convert degrees to radians */
    startLatitude = startLatitude * pi / 180;
    startLongitude = startLongitude * pi / 180;
    endLatitude = endLatitude * pi / 180;
    endLongitude = endLongitude * pi / 180;

    qreal latitudeDifference = endLatitude - startLatitude;
    qreal longitudeDifference = endLongitude - startLongitude;

    qreal temp = pow(sin(latitudeDifference / 2), 2) +
                 cos(startLatitude) * cos(endLatitude) *
                 pow(sin(longitudeDifference / 2), 2);

    qreal distance = earthRadiusInKilometres * 2 * atan2(sqrt(temp),
                     sqrt(1 - temp));

    return distance;

}

//! Clears the current route data.
void QtMapper::clearRoute()
{
    view->enableRouteDisplay(false);
    clearRouteAct->setEnabled(false);

    if (currentRouteData != 0) {
        disconnect(currentRouteData, 0, 0, 0);
        delete currentRouteData;
    }

    if (routeDownloader != 0) {
        disconnect(routeDownloader, 0, 0, 0);
    }

    routeTimer->stop();

    refreshMapView();
}

//! Redraws all items in the map view.
void QtMapper::refreshMapView()
{
    view->clear();
    selectTiles();
    
    if (view->trackEnabled() == true) {
        drawTrackLines();
    }
   
    if (drawPoisAct->isChecked() == true) {
        drawPoiMarkers();
    }
}

//! Changes the state of show POIs.
void QtMapper::changeShowPoisState()
{
    if (drawPoisAct->isChecked() == true) {
        alarmOnAct->setEnabled(false);
        refreshMapView();
    }
    else {
        alarmOnAct->setEnabled(true);
        refreshMapView();
    }
}


//! Called when the route destination has been reached.
void QtMapper::destinationReached()
{
    clearRoute();
}

//! Show/hide the quick menu bar.
void QtMapper::showQuickMenu(bool show)
{
    userWantsQuickMenu = show;
    view->enableQuickMenu(show);
    refreshMapView();
}


//! Saves track point to vector.
void QtMapper::storeTrackPoint()
{
    qreal distance = 0.0;
  
    if (trackPoints.count() > 0) {

        distance = distanceBetweenCoordinates(trackPoints.last()
                                              .getLatitude(),
                                              trackPoints.last()
                                              .getLongitude(),
                                              currentGpsLocation
                                              .getLatitude(),
                                              currentGpsLocation
                                              .getLongitude());

        if (distance < minimumTrackLineDistance) {
            return;
        }
    }

    TrackPoint trackPoint;
    QTime time(QTime::currentTime());
    trackPoint.setTime(time);
    trackPoint.setLatitude(currentGpsLocation.getLatitude());
    trackPoint.setLongitude(currentGpsLocation.getLongitude());

    if (trackPoints.count() < maximumTrackSize) {
        
        if (gpsFixQuality != 0) {

            trackPoints.append(trackPoint);
            QSettings settings (settingOrganizationName, settingApplicationName);
            settings.beginWriteArray(settingTrackPoints);
            if (trackPoints.count() > 1) {
                for (int i = 0; i < trackPoints.count(); ++i) {
                    settings.setArrayIndex(i);
                    settings.setValue(settingTrackPointLatitude, trackPoints[i].getLatitude());
                    settings.setValue(settingTrackPointLongitude, trackPoints[i].getLongitude());
                    settings.setValue(settingTrackPointTime, trackPoints[i].getTime());
                }
                settings.endArray();
            }
        }
    }
    else {
        QString info(tr("Track store is full \nSave and clear track\n"
                        "If you still want to use tracking"));
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(tr("Notice"));
        msgBox.setText(info);
        msgBox.exec();
        enableTrackingAct->setChecked(false);
        enableTracking();
    }

    if (showTrackAct->isChecked() == true) {
        showTrack(true);
    }

    if (trackPoints.count() > 1) {
        showTrackAct->setEnabled(true);
        clearTrackAct->setEnabled(true);
        saveTrackAct->setEnabled(true);
    }
}

//! Change conditions of tracking.
void QtMapper::enableTracking()
{
    if (enableTrackingAct->isChecked() == false) {
        openTrackAct->setEnabled(true);
        view->setTrackingEnabled(false);
        trackingTimer->stop();
        refreshMapView();
    } 
    else {
        openTrackAct->setEnabled(false);
        view->setTrackingEnabled(true);
        trackingTimer->start(trackingInterval);
    }
}

//! Check waypoints when drawing route.
void QtMapper::checkWaypoint()
{
    const double tolerance = 0.01;     // 10 meters
    double currLat = currentGpsLocation.getLatitude();
    double currLon = currentGpsLocation.getLongitude();
    double wpLat = currentRouteData->getWaypoints()
                   .at(view->getCurrentWaypoint())->getLatitude();
    double wpLon = currentRouteData->getWaypoints()
                   .at(view->getCurrentWaypoint())->getLongitude();
    double distance = distanceBetweenCoordinates(currLat, currLon,
                      wpLat, wpLon);

    if (distance < tolerance) {
        moveToNextWaypoint();
    }
}

//! Saves track to file.
void QtMapper::saveTrack()
{
    TrackingDataWriter writer(trackPoints);
    writer.save();

}


//! Open track from xml file.
void QtMapper::openTrack()
{
    if (trackPoints.count() > 0) {
        QString info(tr("You have track open.\n"
                        "You loose it if you haven't save it.\n"
                        "Do you want to save it before opening a new one?"));
        QMessageBox msgBox(this);
        msgBox.setWindowTitle(tr("Warning"));
        msgBox.setText(info);
        msgBox.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
        switch (msgBox.exec()) {
        case QMessageBox::Yes:
            saveTrack();
            break;
        case QMessageBox::No:

            break;
        default:

            break;
        }
    }

    trackPoints.clear();

    QString fileName = QFileDialog::getOpenFileName(this,
                       tr("Open Gpx File"),
                       QDir::currentPath());

    if (fileName.isEmpty()) {
        return;
    }

    QFile file(fileName);

    if (!file.open(QFile::ReadOnly | QFile::Text)) {
        QMessageBox::warning(this, tr("GPX Files"),
                             tr("Cannot read file %1:\n%2.")
                             .arg(fileName)
                             .arg(file.errorString()));
        return;
    }

    QXmlInputSource xmlInputSource(&file);

    QXmlSimpleReader trackReader;
    TrackingDataReader  *trackHandler = new TrackingDataReader(trackPoints);
    trackReader.setContentHandler(trackHandler);
    trackReader.setErrorHandler(trackHandler);

    trackReader.parse(xmlInputSource);

    delete trackHandler;
    
    showTrackAct->setChecked(true);
    showTrack(true);

    if (trackPoints.count() > 1) {
        showTrackAct->setEnabled(true);
        clearTrackAct->setEnabled(true);
        saveTrackAct->setEnabled(true);
    }

}

//! Called whenever the QtMapper window is closed
void QtMapper::closeEvent(QCloseEvent *e)
{
    Q_UNUSED(e);
}

//! Updates the gps fix indicator light
void QtMapper::gpsFixChanged(int newFixQuality)
{
    if (newFixQuality != gpsFixQuality) {

        gpsFixQuality = newFixQuality;

        if (gpsFixQuality == 0) {
            gpsFixIndicator->load(gpsFixInvalidIndicatorResource);
            gpsFixLabel->setToolTip(gpsFixInvalidTooltip);
        }
        else {
            gpsFixIndicator->load(gpsFixValidIndicatorResource);
            gpsFixLabel->setToolTip(gpsFixValidTooltip);

            if (!goToGpsLocationAct->isEnabled() &&
                !goToNearestPoiAct->isEnabled()) {

                goToGpsLocationAct->setEnabled(true);
                goToNearestPoiAct->setEnabled(true);
            }
        }

        gpsFixLabel->setPixmap(*gpsFixIndicator);
    }
}


//! Shows or hides track
void QtMapper::showTrack(bool checked)
{
    if (checked == true) {
        drawTrackLines();
        view->setTrackData(trackPoints);//call when track changes
        view->enableTrack(checked);

        if (trackPoints.count() > 0 && centerOnLocationAct->isChecked() == false) {
            goToLocation(trackPoints.first().getLongitude(),
                         trackPoints.first().getLatitude());
            centerOfScreen.setX(trackPoints.first().getLongitude());
            centerOfScreen.setY(trackPoints.first().getLatitude());
        }
    }
    else if (checked == false) {
        view->enableTrack(checked);
        refreshMapView();
    }
}

//! Draw lines between the track points among track.
void QtMapper::drawTrackLines()
{
    QPainterPath path;
    bool firstPoint = true;

    /*
     * Go though the list of the line points and construct the path along
     * which to draw the lines.
     */
    for (int i = 0; i < trackPoints.count(); i++) {
        int mapX = currentMapLocation.getMapX();
        int mapY = currentMapLocation.getMapY();

        Location l;
        l.setLongitude(trackPoints[i].getLongitude());
        l.setLatitude(trackPoints[i].getLatitude());
       
        l.setZoomLevel(currentMapLocation.getZoomLevel());
        l.setMapX(l.getXTile(l.getLongitude(), l.getZoomLevel()));
        l.setMapY(l.getYTile(l.getLatitude(), l.getZoomLevel()));

        // Check if the current point is in the view.
        for (int row = mapY - mapTileRows / 2;
                row <= mapY + mapTileRows / 2;
                row++) {
            if (row == l.getMapY()) {

                for (int col = mapX - mapTileColumns / 2;
                        col <= mapX + mapTileColumns / 2;
                        col++) {
                    if (col == l.getMapX()) {

                        int x = l.getXPixel(l.getLongitude(), l.getZoomLevel())
                                + (col - mapX + mapTileColumns / 2) * 256;
                        int y = l.getYPixel(l.getLatitude(), l.getZoomLevel())
                                + (row - mapY + mapTileRows / 2) * 256;

                        if (firstPoint) {
                            path.moveTo(x, y);
                            firstPoint = false;
                        }
                        else {
                            path.lineTo(x, y);
                        }
                    }
                }
            }
        }
    }

    // Pass the completed path to the view and draw the track.
    view->drawRouteLine(path);
}

//! Clears stored track (track clears also when shutting down the application).
void QtMapper::clearTrack()
{
    QString info(tr("Do you really want to.\n"
                    "clear the latest track?.\n"));
    QMessageBox mb(this);
    mb.setWindowTitle(tr("CONFIRMATION"));
    mb.setText(info);
    mb.setStandardButtons(QMessageBox::Yes | QMessageBox::No);
    QSettings settings(settingOrganizationName, settingApplicationName);

    switch (mb.exec()) {
    case QMessageBox::Yes:
        trackPoints.clear();
        view->setTrackData(trackPoints);
        settings.remove(settingTrackPoints);
        showTrack(false);
        showTrackAct->setChecked(false);
        showTrackAct->setEnabled(false);
        clearTrackAct->setEnabled(false);
        saveTrackAct->setEnabled(false);
        break;
    case QMessageBox::No:

        break;
    default:

        break;
    }
}

//! Add a POI at a location selected with the mouse.
void QtMapper::addPoiFromMapCoordinates(int x, int y)
{
    UnitConverter uc;
    QPointF point = uc.coordinatesFromPixels(currentMapLocation.getZoomLevel(),
                    upperLeftCornerMapXIndex,
                    upperLeftCornerMapYIndex,
                    x, y);

    AddPoiDialog *dialog = new AddPoiDialog(*this, this);
    dialog->setLongitudeContents(point.x());
    dialog->setLatitudeContents(point.y());
    dialog->setCategories();

    if (dialog->exec() == QDialog::Accepted) {
        dialog->addPoiToDatabase();
    }
}

//! Start/stop tracking when the quick menu tracking button is pressed.
void QtMapper::enableTrackingWithQuickMenuButton(bool enable)
{
    enableTrackingAct->setChecked(enable);
    enableTracking();
}

//! Move screen center point to nearest POI.
void QtMapper::goToNearestPoi()
{
    if (!poiDb->isConnected()) {
        poiDb->createConnection();
    }
    
    QVector<Poi> nearestPois;
    poiDb->getPois(nearestPois,currentGpsLocation.getLatitude(),
                   currentGpsLocation.getLongitude(), 1, -1);
   
    if (nearestPois.count() == 0 || currentGpsLocation.isLocationDataValid() == false) {
        return;
    }

    zoom(zoomLevelOfGoToPoi);

    goToLocation(nearestPois.first().getLongitude(),
                 nearestPois.first().getLatitude());
                 
    QString infoText(nearestPois.first().getLabel());
                     infoText.append("\n");
                     infoText.append(nearestPois.first().getDescription());
        
    drawInfoMarker(nearestPois.first().getLatitude(),
                   nearestPois.first().getLongitude(),
                   infoText, infoTypePoi);
                   
    disconnect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));               
                   
    connect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));

    setViewCenterCoordinates();

}

//! Centers the view on gps marker and sets view center coordinates
void QtMapper::centerOnGpsMarker()
{
    QGraphicsPixmapItem *marker = view->getMarker();

    if (marker != NULL) {
        view->centerOn(marker);
        centerOfScreen.setX(currentGpsLocation.getLongitude());
        centerOfScreen.setY(currentGpsLocation.getLatitude());
    }
}

//! Goes to selected POI (from browse POIs) and shows the POI.
void QtMapper::goToSelectedPoi(Poi poi)
{
    if (centerOnLocationAct->isChecked()) {
        centerOnLocationAct->toggle();
    }

    zoom(zoomLevelOfGoToPoi);

    goToLocation(poi.getLongitude(), poi.getLatitude());

    QString infoText(poi.getLabel());
                     infoText.append("\n");
                     infoText.append(poi.getDescription());
        
    drawInfoMarker(poi.getLatitude(),
                   poi.getLongitude(),
                   infoText, infoTypePoi);
    disconnect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));               
                   
    connect(view->getScene(),
            SIGNAL(mouseButtonPressed(QGraphicsSceneMouseEvent*)),
            view, SLOT(mousePressed(QGraphicsSceneMouseEvent*)));

    setViewCenterCoordinates();
    
}

//! Called when the centerOnLocation action is toggled.
void QtMapper::centerViewOnLocationToggled(bool enable)
{
    if (enable) {
        centerOnLocationAct->setIcon(QIcon(lockedIconResource));
        goToCurrentGpsLocation();
        goToAddressAct->setEnabled(false);
        menu_Go_to->setEnabled(false);
        zoomToolAct->setEnabled(false);
        moveToolAct->setEnabled(false);
        menu_Pan->setEnabled(false);

        view->lockView(true);

    }
    else {
        centerOnLocationAct->setIcon(QIcon(unlockedIconResource));
        goToAddressAct->setEnabled(true);
        menu_Go_to->setEnabled(true);
        zoomToolAct->setEnabled(true);
        moveToolAct->setEnabled(true);
        menu_Pan->setEnabled(true);
        view->lockView(false) ;
    }
}

//! Called when coordinateToolAct is triggered
void QtMapper::coordinateToolActTriggered()
{
    if (coordinateToolAct->isChecked()) {
        view->toggleLonLatBox(true);
    }
    else {
        view->toggleLonLatBox(false);
    }
}
