#
# This file is part of Python Terra
# Copyright (C) 2007-2009 Instituto Nokia de Tecnologia
# Contact: Renato Chencarek <renato.chencarek@openbossa.org>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# Additional permission under GNU GPL version 3 section 7
#
# The copyright holders grant you an additional permission under Section 7
# of the GNU General Public License, version 3, exempting you from the
# requirement in Section 6 of the GNU General Public License, version 3, to
# accompany Corresponding Source with Installation Information for the
# Program or any work based on the Program. You are still required to comply
# with all other Section 6 requirements to provide Corresponding Source.
#

import new
import weakref
import logging
import traceback

from terra_object import TerraObject
from notify_list import NotifyList
from time import time


log = logging.getLogger("terra.core.model")


class Model(TerraObject):
    """Base class for all Terra models."""
    terra_type = "Model"

    def __init__(self, name, parent=None):
        TerraObject.__init__(self)
        self.name = name
        self.parent = parent
        if parent is not None:
            parent.append(self)
        self._state_reason = None
        self.callback_state_changed = None

    def _get_state_reason(self):
        return self._state_reason

    def _set_state_reason(self, value):
        if value == self._state_reason:
            return

        self._state_reason = value
        if self.callback_state_changed is not None:
            self.callback_state_changed(self)

    state_reason = property(_get_state_reason, _set_state_reason)

    def _check_valid(self):
        return self._state_reason is None

    is_valid = property(_check_valid)

    def hold(self):
        pass

    def release(self):
        pass

    def __str__(self):
        return "%s(name=%r, parent=%s)" % (self.__class__.__name__, self.name,
                                           bool(self.parent))

    __repr__ = __str__


class WeakMethod(object):
    def __init__(self, method):
        self._ref = weakref.ref(method.im_self)
        self._func = method.im_func
        self._class = method.im_class

    def __call__(self):
        obj = self._ref()
        if obj is None:
            return None

        return new.instancemethod(self._func, obj, self._class)

    def __eq__(self, other):
        try:
            return type(self) is type(other) and \
                self() == other()
        except:
            return False

    def __ne__(self, other):
        return not self.__eq__(other)


class ModelStatus(Model):
    terra_type = "Model/Status"

    def __init__(self, name):
        Model.__init__(self, name)
        self._status = None
        self._callbacks = []

    def _get_status(self):
        return self._status

    def _set_status(self, nstatus):
        if nstatus == self._status:
            return

        self._status = nstatus
        to_remove = []

        for m in self._callbacks:
            cb = m()
            if cb is None:
                to_remove.append(m)
            else:
                cb(self)

        for m in to_remove:
            self._callbacks.remove(m)

    status = property(_get_status, _set_status)

    def add_listener(self, cb):
        self._callbacks.append(WeakMethod(cb))

    def remove_listener(self, cb):
        wm = WeakMethod(cb)
        try:
            self._callbacks.remove(wm)
        except ValueError:
            log.debug_warning("tried to remove listener already removed %r",
                              cb, exc_info=True)


class ModelFolder(Model):
    """Base model class for all Terra folders."""
    terra_type = "Model/Folder"

    def __init__(self, name, parent=None, children=None):
        if children is not None:
            self.children = NotifyList(children)
        else:
            self.children = NotifyList()
        Model.__init__(self, name, parent)
        self.current = None
        self._changed_callbacks = []
        self.callback_loaded = None
        self.is_loading = False
        self._hold_count = 0
        self._load_count = 0

    def changed_callback_add(self, func):
        self._changed_callbacks.append(func)

    def changed_callback_del(self, func):
        self._changed_callbacks.remove(func)

    def _check_loaded(self):
        return self._load_count > 0 and not self.is_loading

    is_loaded = property(_check_loaded)

    def notify_model_changed(self):
        for f in self._changed_callbacks:
            f(self)

    def _next_index_get(self, circular, same_type):
        if self.current is None:
            raise AttributeError("You must set the self.current attribute"
                                 " before using this method")

        index = self._index_inc(self.current, circular)

        if same_type:
            current_type = type(self.children[self.current])
            while current_type != type(self.children[index]):
                index = self._index_inc(index, circular)

        return index

    def _prev_index_get(self, circular, same_type):
        if self.current is None:
            raise AttributeError("You must set the self.current attribute"
                                 " before using this method")

        index = self._index_dec(self.current, circular)

        if same_type:
            current_type = type(self.children[self.current])
            while current_type != type(self.children[index]):
                index = self._index_dec(index, circular)

        return index

    def _index_inc(self, index, circular):
        if not circular:
            if index + 1 >= len(self.children):
                raise IndexError("reached end of list")
            index += 1
        else:
            index = (index + 1) % len(self.children)
        return index

    def _index_dec(self, index, circular):
        if not circular:
            if index - 1 < 0:
                raise IndexError("reached start of list")
            index -= 1
        else:
            index = (index + len(self.children) - 1) % \
                    len(self.children)
        return index

    def append(self, item):
        self.children.append(item)

    def extend(self, items):
        self.children.extend(items)

    def insert(self, index, item):
        self.children.insert(index, item)

    def remove(self, item):
        self.children.remove(item)

    def _do_unload(self):
        log.debug("unloading %r ...", self)
        self.children.callback_changed = None # don't send me updates anymore
        self.do_unload()

    def hold(self):
        assert self.is_loaded or self.is_loading
        assert self._hold_count >= 0
        self._hold_count += 1

    def release(self):
        assert self._load_count >= 0
        assert self._hold_count > 0
        self._hold_count -= 1
        if self._load_count == 0 and \
           self._hold_count == 0:
            log.debug_warning("not held anymore, unloading %r (load count %d, "
                              "hold count %d) ...", self, self._load_count,
                              self._hold_count)
            self._do_unload()

    def inform_loaded(self):
        if self.is_loaded:
            log.debug_warning("model %r inform_loaded() called when model was "
                              "already loaded OR is being called by do_load().",
                              self)
            traceback.print_stack()
            return

        self.is_loading = False
        if self.callback_loaded is not None:
            self.callback_loaded(self)

    def do_load(self):
        """Method called to actually load the folder items.

        @note: this method is supposed to be implemented by subclasses.
        """
        pass

    def load(self):
        if self._load_count < 0:
            log.debug_error("trying to load model folder %r with "
                            "weird load count: %d", self, self._load_count)
            traceback.print_stack()
            self._load_count = 0
            return

        self._load_count += 1
        if self._load_count > 1:
            return

        log.debug("loading %r ...", self)
        self.children.callback_changed = self.notify_model_changed
        t0 = time()
        self.do_load()
        t1 = time()
        dt = (t1 - t0) * 1000
        if dt > 10:
            log.debug_warning("Model's do_load() is too slow: %0.0fms (%s)",
                              dt, self.__class__.__name__)

        # At this point, if the model is not still loading (so won't call
        # inform_loaded() later), we know that we can trigger the
        # callback_loaded. We do this directly to avoid the check in
        # inform_loaded() that doesn't apply to this case.
        if not self.is_loading and self.callback_loaded is not None:
            self.callback_loaded(self)

    def do_unload(self):
        """Method called to unload the folder items.

        @note: a subclass can override this method and implement its own
        unloading but it should remember to call this method or delete the
        C{children} list.
        """
        del self.children[:]

    def unload(self):
        if self._load_count <= 0:
            log.debug_error("trying to unload model folder %r with "
                            "weird load count: %d", self, self._load_count)
            traceback.print_stack()
            self._load_count = 0
            return

        self._load_count -= 1
        if self._load_count > 0:
            return

        if self._hold_count > 0:
            log.debug_warning("still held, not unloading %r (load count %d, "
                              "hold count %d) ...", self, self._load_count,
                              self._hold_count)
            return

        self._do_unload()

    def next(self, circular=False, same_type=True):
        """Method to select the next item in the folder model.

        @param circular: if model folder should be treated as a circular list.
        @param same_type: true if only elements of self.current type should
               be returned.
        @note: the self.current attribute should be set before calling
               this method.
        """
        self.current = self._next_index_get(circular, same_type)
        return self.children[self.current]

    def prev(self, circular=False, same_type=True):
        """Method to select the previous item in the folder model.

        @param circular: if model folder should be treated as a circular list.
        @param same_type: true if only elements of self.current type should
               be returned.
        @note: the self.current attribute should be set before calling
               this method.
        """
        self.current = self._prev_index_get(circular, same_type)
        return self.children[self.current]

    def next_get(self, circular=False, same_type=True):
        """Method that returns the next item in the folder model.

        @param circular: if model folder should be treated as a circular list.
        @param same_type: true if only elements of self.current type should
               be returned.
        @note: the self.current attribute should be set before calling
               this method.
        """
        return self.children[self._next_index_get(circular, same_type)]

    def prev_get(self, circular=False, same_type=True):
        """Method that returns the prev item in the folder model.

        @param circular: if model folder should be treated as a circular list.
        @param same_type: true if only elements of self.current type should
               be returned.
        @note: the self.current attribute should be set before calling
               this method.
        """
        return self.children[self._prev_index_get(circular, same_type)]

    def has_next(self, same_type=True):
        """Method that returns if there is a next item in the folder model.

        @param same_type: true if only elements of self.current type should
               be considered.
        @note: the self.current attribute should be set before calling
               this method.
        """
        try:
            self._next_index_get(False, same_type)
        except IndexError:
            return False
        else:
            return True

    def has_prev(self, same_type=True):
        """Method that returns if there is a previous item in the folder model.

        @param same_type: true if only elements of self.current type should
               be considered.
        @note: the self.current attribute should be set before calling
               this method.
        """
        try:
            self._prev_index_get(False, same_type)
        except IndexError:
            return False
        else:
            return True

    def __str__(self):
        return "%s(name=%r, parent=%s, children=%s)" % \
               (self.__class__.__name__, self.name, bool(self.parent),
                bool(self.children))

    __repr__ = __str__
