import os
import commands
import conic
import gnome.gconf
import sys
import time
import dbus

##Connection item contains all infos about a connection
class ConnectionItem():
	##constants
	CNX_TYPE_OTHER = 0
	CNX_TYPE_WIFI = 1
	CNX_TYPE_DATA = 2
	
	##name of the connection
	name = ''
	##connection type (see contants)
	type = CNX_TYPE_OTHER
	##connection id
	id = ''
	
	##tostring
	def __str__(self):
		return 'ConnectionItem(name=\'%s\', type=\'%d\', id=\'%s\')' % (self.name, self.type, self.id)

##Connection manager exception
class ConnectionManagerException(Exception):
	def __init__(self, message):
		Exception.__init__(self, message)
		
##Connection manager manages all you need about connections
class ConnectionManager():
	##constructor
	##@param onConnectedCallback : define a callback when connection established
	##@param onDisconnectedCallback : define a callback when connection is closed
	def __init__(self, onConnectedCallback=None, onDisconnectedCallback=None):
		#set members
		self.__isConnected = False
		self.__connection = conic.Connection()
		self.__onConnectedCallback = onConnectedCallback
		self.__onConnectedFromConnectToCallback = None
		self.__onDisconnectedCallback = onDisconnectedCallback
		
		#connect to connection event
		self.__connection.connect("connection-event", self.__connectionEvent)
		self.__connection.set_property("automatic-connection-events", True)
		
	##destructor
	def __del__(self):
		if self.__connection!=None:
			self.__connection.set_property("automatic-connection-events", False)
	
	##on connection event
	def __onConnected(self, status, error, iap_id, bearer):
		#change connected status
		self.__isConnected = True
		if error!=conic.CONNECTION_ERROR_NONE:
			self.__isConnected = False
		
		#callback if necessary
		if self.__onConnectedCallback!=None:
			self.__onConnectedCallback(status, error, iap_id, bearer)
		if self.__onConnectedFromConnectToCallback!=None:
			self.__onConnectedFromConnectToCallback(status, error, iap_id, bearer)
			#reset
			self.__onConnectedFromConnectToCallback = None
	
	##on disconnection event
	def __onDisconnected(self, status, error, iap_id, bearer):
		#change connected status
		self.__isConnected = False
		
		#callback if necessary
		if self.__onDisconnectedCallback!=None:
			self.__onDisconnectedCallback(status, error, iap_id, bearer)
			
	##connection event callback
	def __connectionEvent(self, connection, event):
		status = event.get_status()
		error = event.get_error()
		iap_id = event.get_iap_id()
		bearer = event.get_bearer_type()

		if status == conic.STATUS_CONNECTED:
			self.__onConnected(status, error, iap_id, bearer)
		elif status == conic.STATUS_DISCONNECTED:
			self.__onDisconnected(status, error, iap_id, bearer)
	
	##Connect to specified connection id
	##@info http://www.v13.gr/blog/?p=4
	##@param connection : ConnectionItem object
	#@param onConnectedCallback : launch specified callback when connection established
	def connectTo(self, connection, onConnectedCallback):
		if connection!=None and connection.__class__.__name__=='ConnectionItem':
			#check if not connected
			if self.isConnected():
				#connected, disconnect first
				self.disconnect()
				
			#set members
			self.__onConnectedFromConnectToCallback = onConnectedCallback
		
			#now connect
			cnxString = 'dbus-send --system --print-reply --type=method_call ' \
					'--dest=com.nokia.icd /com/nokia/icd ' \
					'com.nokia.icd.connect ' \
					'string:"%s" uint32:0' % connection.id
			status,output = commands.getstatusoutput(cnxString)
			
			#check result
			if status==0:
				#connection successful
				return True
			else:
				#connection failed (already connected?)
				return False
		else:
			raise ConnectionManagerException('need ConnectionItem object in parameter!')
			return False
	
	##Disconnect from current connection
	##@info http://www.v13.gr/blog/?p=4
	def disconnect(self):
		os.system('dbus-send --system --dest=com.nokia.icd ' \
				'/com/nokia/icd_ui com.nokia.icd_ui.disconnect ' \
				'boolean:true')
	
	##return connected status
	##@return True if connected, False otherwise
	def isConnected(self):
		return self.__isConnected
				
	##Return connections
	##@param type : CNX_TYPE_? to get specified connections type or None to get all connections
	##@param name : name of searched connection
	##@return array of connections
	def getConnections(type=None, name=None):
		#init
		connections = []
		client = gnome.gconf.client_get_default()
		
		#list all dirs
		dirs = client.all_dirs('/system/osso/connectivity/IAP')
		for dir in dirs:
			#create new item
			connection = ConnectionItem()
			
			#get connection name
			connection.name = client.get_string(dir+'/name')
			
			#get connection type
			cnxType = client.get_string(dir+'/type')
			if cnxType.startswith('DUN'):
				#data connection
				connection.type = ConnectionItem.CNX_TYPE_DATA
			elif cnxType.startswith('WLAN'):
				#wifi connection
				connection.type = ConnectionItem.CNX_TYPE_WIFI
			else:
				#unknow connection type
				connection.type = ConnectionItem.CNX_TYPE_OTHER
				
			#get connection id
			#format : /system/osso/connectivity/IAP/62d4e38f-257d-42af-bd71-a0f1b60a31ee
			connection.id = dir.split('/')[5]
			
			#store connection item if necessary
			if type==None and name==None:
				#no filter specified, add connection
				connections.append(connection)
			elif type==None and name!=None:
				#filter on name specified
				if connection.name==name:
					#awaited name, add connection
					connections.append(connection)
			elif type!=None and name==None:
				#filter on type
				if connection.type==type:
					#awaited type, add connection
					connections.append(connection)
			else:
				#filter on type and name
				if connection.type==type and connection.name==name:
					connections.append(connection)
	
		#return found connections
		return connections
	getConnections = staticmethod(getConnections)

	##open connection dialog
	def openConnectionDialog(self):
		#TODO
		pass
