import commands
import threading

class LedManager():
	#patterns
	#See /etc/mce/mce.ini for legal pattern names
	__PATTERN_ERROR = 'PatternError'
	__PATTERN_DEVICEON = 'PatternDeviceOn'
	__PATTERN_POWERON = 'PatternPowerOn'
	__PATTERN_POWEROFF = 'PatternPowerOff'
	__PATTERN_COMCALL = 'PatternCommunicationCall'
	__PATTERN_COMCHAT = 'PatternCommunicationChat'
	__PATTERN_COMEMAIL = 'PatternCommunicationEmail'
	__PATTERN_NOTIF = 'PatternCommonNotification'
	__PATTERN_WEBCAMON = 'PatternWebcamActive'
	__PATTERN_BATCHARGING = 'PatternBatteryCharging'
	__PATTERN_BATFULL = 'PatternBatteryFull'
	__PATTERN_PSYCHEDELIA = 'PatternPsychedelia'
	__PATTERN_DEVICESOFTOFF = 'PatternDeviceSoftOff'
	
	#members
	__isPatternOn = False
	__currentPattern = ''
	__timer = None
	__timerStopped = False

	#constructor	
	def __init__(self):
		self.__isPatternOn = False
	
	#destructor
	def __del__(self):
		#turn off led if necessary
		if self.__isPatternOn:
			self.turnAllPatternsOff()
	
	#return all patterns
	def getAllPatterns(self):
		patterns = []
		patterns.append(self.__PATTERN_ERROR)
		patterns.append(self.__PATTERN_DEVICEON)
		patterns.append(self.__PATTERN_POWERON)
		patterns.append(self.__PATTERN_POWEROFF)
		patterns.append(self.__PATTERN_COMCALL)
		patterns.append(self.__PATTERN_COMCHAT)
		patterns.append(self.__PATTERN_COMEMAIL)
		patterns.append(self.__PATTERN_NOTIF)
		patterns.append(self.__PATTERN_WEBCAMON)
		patterns.append(self.__PATTERN_BATCHARGING)
		patterns.append(self.__PATTERN_BATFULL)
		patterns.append(self.__PATTERN_PSYCHEDELIA)
		patterns.append(self.__PATTERN_DEVICESOFTOFF)
		return patterns
	
	#turn led on
	#@param pattern : pattern to apply
	def __turnPatternOn(self, pattern=''):
		#first of all turn off led if already on
		if self.isPatternOn():
			self.turnPatternOff()
			
		#turn pattern on
		string = 'dbus-send --system --type=method_call --dest=com.nokia.mce /com/nokia/mce/request ' \
				'com.nokia.mce.request.req_led_pattern_activate string:"%s"' % pattern
		status,output = commands.getstatusoutput(string)
		
		#update members
		self.__isPatternOn = True
		self.__currentPattern = pattern
		
		return status
	
	#turn pattern off
	#@param blink : set to true stop too blink timer
	def turnPatternOff(self, blink=False):
		if blink:
			self.__turnTimerOff()
	
		if len(self.__currentPattern)==0:
			if self.__isPatternOn:
				#no current pattern and one is on... turn off all patterns
				return self.turnAllPatternsOff()
			else:
				#no pattern on
				pass
		else:
			#turn off current pattern
			string = 'dbus-send --system --type=method_call --dest=com.nokia.mce /com/nokia/mce/request ' \
					'com.nokia.mce.request.req_led_pattern_deactivate string:"%s"' % self.__currentPattern
			status,output = commands.getstatusoutput(string)
			
			#update members
			self.__isPatternOn = False
			self.__currentPattern = ''
			
			return status
	
	#turn led off for all patterns
	def turnAllPatternsOff(self):
		#stop timer
		self.__turnTimerOff()
	
		#get all patterns
		patterns = self.getAllPatterns()
		
		#turn off each pattern
		for pattern in patterns:
			self.__currentPattern = pattern
			self.turnPatternOff()
			
		#update members
		self.__currentPattern = ''
		self.__isPatternOn = False
		
		return True
	
	#return led on status
	def isPatternOn(self):
		return self.__isPatternOn

	#turn on led using specified pattern
	#@param pattern : pattern to use when turn on led (must be known!)
	def turnPatternOn(self, pattern):
		#first of all check if pattern is correct
		patterns = self.getAllPatterns()
		if patterns.count(pattern)==0:
			#pattern doesn't exist
			sys.stderr.write('LedManager : unknown pattern "%s"!\n' % pattern)
			return False
		
		#finally turn on led with specified pattern
		self.__turnPatternOn(pattern)
		return True
		
	#turn off blink timer
	def __turnTimerOff(self):
		self.__timerStopped = True
		if self.__timer!=None:
			self.__timer.cancel()
			self.__timer = None
	
	#turn on blink timer
	#@param interval : interval to repeat timer
	#@param pattern : led pattern to use
	def __turnTimerOn(self, interval, pattern):
		#turn pattern on
		self.turnPatternOn(pattern)
		
		#launch timer again
		if not self.__timerStopped:
			self.__timer = threading.Timer(interval, self.__turnTimerOn, [interval,pattern])
			self.__timer.start()
		else:
			#timer stopped, init timer status for next time
			self.__timerStopped = False
			
	def __blinkLed(self, interval, pattern):
		#turn on green led
		self.turnPatternOn(pattern)
		
		#launch timer
		self.__timer = threading.Timer(interval, self.__turnTimerOn, [interval, pattern])
		self.__timer.start()
	
	#----------------------------------------------------------------------------------------------
	# ALIASES
	#----------------------------------------------------------------------------------------------
	
	#turn led on in red color
	def turnRedLedOn(self):
		self.turnPatternOn('PatternWebcamActive')
	
	#blink green led
	#@note only available when display is off !
	def blinkGreenLed(self):
		self.__blinkLed(1.0, 'PatternDeviceSoftOff')
		
	#blink red led
	def blinkRedLed(self):
		self.__blinkLed(1.0, 'PatternWebcamActive')
		
	#blink blue led
	def blinkBlueLed(self):
		self.turnPatternOn('PatternCommunicationEmail')
		
	#blink white led
	def blinkWhiteLed(self):
		self.__blinkLed(1.0, 'PatternPowerOn')
	
if __name__ == "__main__":
	import gobject
	import sys
	import time
	
	arg = ''
	if len(sys.argv)>1:
		arg = sys.argv[1]
	
	if arg=='all':
		#test all patterns
		def testLed(loop, manager, index):
			#get patterns
			patterns = manager.getAllPatterns()
			if index<len(patterns):
				#choose current pattern
				pattern = patterns[index]
				#change led pattern
				print 'turn pattern "%s" on during 10 seconds...' % pattern
				manager.turnPatternOn(pattern)
				#increase index
				index += 1
				#launch again test
				gobject.timeout_add(10000, testLed, loop, manager, index) 
			else:
				#no more pattern to test, turn off the led
				print 'turn pattern off'
				manager.turnPatternOff()
				loop.quit()
			
		loop = gobject.MainLoop()
		manager = LedManager()
		gobject.timeout_add(500, testLed, loop, manager, 0)
		loop.run()
		
	elif arg=='pattern':
		if len(sys.argv)>2:
			arg = sys.argv[2]
			manager = LedManager()
			print 'turn pattern "%s" on during 10 seconds...' % arg
			manager.turnPatternOn(arg)
			time.sleep(10)
			manager.turnPatternOff()
			del manager
		else:
			print 'no pattern specified'
	
	elif arg=='off':
		manager = LedManager()
		manager.turnAllPatternsOff()
		del manager
		
	elif arg=='blink':
		if len(sys.argv)>2:
			arg = sys.argv[2]
			manager = LedManager()
			print 'turn %s led on during 10 seconds...' % arg
			if arg=='red':
				manager.blinkRedLed()
			elif arg=='green':
				manager.blinkGreenLed()
			elif arg=='blue':
				manager.blinkBlueLed()
			elif arg=='white':
				manager.blinkWhiteLed()
			else:
				print 'unknown color %s' % arg
			time.sleep(10)
			#if you blink led, you must stop blink timer using turnPatternOff(True) or turnAllPatternsOff
			manager.turnPatternOff(True)
			del manager
		else:
			print 'no color specified'
	
	else:
		print 'Unknown argument (available are all, pattern <pattern>, blink <color>, off)'
	