import os
import commands
import conic
import gnome.gconf
import sys
import time

class ConnectionItem():
	#constants
	CNX_TYPE_OTHER = 0
	CNX_TYPE_WIFI = 1
	CNX_TYPE_DATA = 2
	
	#members
	name = ''
	type = CNX_TYPE_OTHER
	id = ''
	
	#tostring
	def __str__(self):
		return 'ConnectionItem(name=\'%s\', type=\'%d\', id=\'%s\')' % (self.name, self.type, self.id)

class ConnectionManager():
	#members
	__connection = None
	__isConnected = False
	__onConnectedCallback = None
	__onDisconnectedCallback = None
	
	#constructor
	def __init__(self, onConnectedCallback=None, onDisconnectedCallback=None):
		#set members
		self.__connection = conic.Connection()
		self.__onConnectedCallback = onConnectedCallback
		self.__onDisconnectedCallback = onDisconnectedCallback
		
		#connect to connection event
		self.__connection.connect("connection-event", self.__connectionEvent)
		self.__connection.set_property("automatic-connection-events", True)
		
	#destructor
	def __del__(self):
		if self.__connection!=None:
			self.__connection.set_property("automatic-connection-events", False)
	
	#on connection event
	def __onConnected(self, status, error, iap_id, bearer):
		#change connected status
		self.__isConnected = True
		if error!=conic.CONNECTION_ERROR_NONE:
			self.__isConnected = False
		
		#callback if necessary
		if self.__onConnectedCallback!=None:
			self.__onConnectedCallback(status, error, iap_id, bearer)
	
	#on disconnection event
	def __onDisconnected(self, status, error, iap_id, bearer):
		#change connected status
		self.__isConnected = False
		
		#callback if necessary
		if self.__onDisconnectedCallback!=None:
			self.__onDisconnectedCallback(status, error, iap_id, bearer)
			
	#connection event callback
	def __connectionEvent(self, connection, event):
		status = event.get_status()
		error = event.get_error()
		iap_id = event.get_iap_id()
		bearer = event.get_bearer_type()

		if status == conic.STATUS_CONNECTED:
			self.__onConnected(status, error, iap_id, bearer)
		elif status == conic.STATUS_DISCONNECTED:
			self.__onDisconnected(status, error, iap_id, bearer)
	
	#Connect to specified connection id
	#@info http://www.v13.gr/blog/?p=4
	#@param connection : ConnectionItem object
	def connectTo(self, connection):
		if connection!=None and connection.__class__.__name__=='ConnectionItem':
			#check if not connected
			if self.isConnected():
				#connected, disconnect first
				self.disconnect()
		
			#now connect
			cnxString = 'dbus-send --system --print-reply --type=method_call ' \
					'--dest=com.nokia.icd /com/nokia/icd ' \
					'com.nokia.icd.connect ' \
					'string:"%s" uint32:0' % connection.id
			status,output = commands.getstatusoutput(cnxString)
			
			#check result
			if status==0:
				#connection successful
				return True
			else:
				#connection failed (already connected?)
				return False
		else:
			sys.stderr.write('ConnectionManager : need ConnectionItem instance as parameter!\n')
			return False
	
	#Disconnect from current connection
	#@info http://www.v13.gr/blog/?p=4
	def disconnect(self):
		os.system('dbus-send --system --dest=com.nokia.icd ' \
				'/com/nokia/icd_ui com.nokia.icd_ui.disconnect ' \
				'boolean:true')
	
	#return connected status
	#@return True if connected, False otherwise
	def isConnected(self):
		return self.__isConnected
				
	#return all available connections
	#@param type : CNX_TYPE_? to get specified connections type or None to get all connections
	#@param name : name of searched connection
	def getAllConnections(self, type=None, name=None):
		#init
		connections = []
		client = gnome.gconf.client_get_default()
		
		#list all dirs
		dirs = client.all_dirs('/system/osso/connectivity/IAP')
		for dir in dirs:
			#create new item
			connection = ConnectionItem()
			
			#get connection name
			connection.name = client.get_string(dir+'/name')
			
			#get connection type
			cnxType = client.get_string(dir+'/type')
			if cnxType.startswith('DUN'):
				#data connection
				connection.type = ConnectionItem.CNX_TYPE_DATA
			elif cnxType.startswith('WLAN'):
				#wifi connection
				connection.type = ConnectionItem.CNX_TYPE_WIFI
			else:
				#unknow connection type
				connection.type = ConnectionItem.CNX_TYPE_OTHER
				
			#get connection id
			#format : /system/osso/connectivity/IAP/62d4e38f-257d-42af-bd71-a0f1b60a31ee
			connection.id = dir.split('/')[5]
			
			#store connection item if necessary
			if type==None or (type!=None and connection.type==type):
				connections.append(connection)
			if name==None or (name!=None and connection.name==name):
				connections.append(connection)
		
		#return found connections
		return connections

if __name__ == "__main__":
	import gobject
	
	arg = ''
	if len(sys.argv)>1:
		arg = sys.argv[1]
		
	if arg=='events':
		#test connection manager events
		def onConnected(id, type, status, error):
			print 'connected'
		def onDisconnected(id, type, status, error):
			print 'disconnected'
		loop = gobject.MainLoop()
		manager = ConnectionManager(onConnected, onDisconnected)
		loop.run()
		del manager

	elif arg=='allcnx':
		#test get all connection
		manager = ConnectionManager()
		connections = manager.getAllConnections()
		for connection in connections:
			print connection
		del manager
	
	elif arg=='discnx':
		#test disconnection
		manager = ConnectionManager()
		print 'Disconnecting...'
		manager.disconnect()
		del manager
		
	elif arg=='iscnx':	
		#test if already connected or not
		def isConnected(manager):
			print 'Connected ? %s' % str(manager.isConnected())
		loop = gobject.MainLoop()
		manager = ConnectionManager()
		gobject.timeout_add(500, isConnected, manager) 
		loop.run()
		del manager
	
	elif arg=='cnx':
		#test connect to first wlan connection
		manager = ConnectionManager()
		connections = manager.getAllConnections(ConnectionItem.CNX_TYPE_DATA)
		print '%d data connections found' % len(connections)
		if len(connections)>0:
			print 'connecting to %s' % str(connections[0])
			manager.connectTo(connections[0])
		del manager
	
	else:
		print 'Unknown argument (try events, allcnx, discnx, cnx, iscnx)'
