#!/usr/bin/python

from decimal import *

import os

class Converter:

	_SI_PREFIXES = (
		( 'deka',	1	),
		( 'hecto',	2	),
		( 'kilo',	3	),
		( 'mega',	6	),
		( 'giga',	9	),
		( 'tera',	12	),
		( 'peta',	15	),
		( 'exa',	18	),
		( 'zetta',	21	),
		( 'yotta',	24	),
		( 'deci',	-1	),
		( 'centi',	-2	),
		( 'milli',	-3	),
		( 'micro',	-6	),
		( 'nano',	-9	),
		( 'pico',	-12	),
		( 'femto',	-15	),
		( 'atto',	-18	),
		( 'zepto',	-21	),
		( 'yocto',	-24	),
	)

	def __init__(self, unit_data='/usr/share/pyconverter'):
		data = {}
		order = { '_TYPES_': [] }

		dosi = False
		type = None
		unit = None
		eqnf = None
		fp = open(os.path.join(unit_data, 'conversions.data'))
		for l in fp.readlines():
			d = l.split()
			if len(d) > 0:
				if d[0] == 'TYPE':
					type = l.split(None, 1)[1].rstrip().lstrip("'").rstrip("'")
					order['_TYPES_'].append(type)
					order[type] = []
					data[type] = {}
				elif type is not None and d[0] == 'UNIT':
					unit = l.split(None, 1)[1].rstrip().lstrip("'").rstrip("'")
					order[type].append(unit)
				elif unit is not None and d[0] == 'BASE':
					if d[1] == 'NORMAL':
						data[type][unit] = ([ 'val' ], [ 'val' ])
					elif d[1] == 'SI_PREFIX':
						data[type][unit] = ([ 'val' ], [ 'val' ])
						self._addSiUnits(data, order, type, unit, data[type][unit])
					else:
						print 'error parsing line "%s"' % l
					unit = None
				elif unit is not None and d[0] == 'DOSI':
					dosi = True
				elif unit is not None and d[0] == 'EQNF':
					eqnf = d[1:]
				elif eqnf is not None and d[0] == 'EQNT':
					data[type][unit] = ( eqnf, d[1:] )
					if dosi:
						self._addSiUnits(data, order, type, unit, data[type][unit])
					eqnf = None
					dosi = False
		fp.close()

		self._conversions = data
		self._conv_orders = order

	def _addSiUnits(self, data, order, type, unit, eqn):
		for (prefix, factor) in self._SI_PREFIXES:
			name = '%s%s' % (prefix, unit)
			name = name.lower().capitalize()
			fact = str(10 ** factor)
			try:
				indx = eqn[0].index('val') + 1
			except ValueError:
				print "Whoops, couldn't find 'val' in equation for %s" % name
				return
			eqnf = eqn[0][:]
			eqnf.insert(indx+0, fact)
			eqnf.insert(indx+1, '*')
			eqnt = eqn[1][:]
			eqnt.insert(indx+0, fact)
			eqnt.insert(indx+1, '/')
			data[type][name] = (eqnf, eqnt)
			order[type].append(name)

	def _rpnEval(self, val, formula):
		rpn = [ ]

		# simple RPN engine
		for op in formula:
			if op == 'val':
				rpn.append(Decimal(val))
			elif op == '+':
				b = rpn.pop()
				a = rpn.pop()
				rpn.append(a + b)
			elif op == '-':
				b = rpn.pop()
				a = rpn.pop()
				rpn.append(a - b)
			elif op == '*':
				b = rpn.pop()
				a = rpn.pop()
				rpn.append(a * b)
			elif op == '/':
				b = rpn.pop()
				a = rpn.pop()
				rpn.append(a / b)
			elif op.startswith('CONVF:'):
				a = rpn.pop()
				rpn.append(self._rpnEval(a, self._conversions[self._current_type][op[6:]][0]))
			elif op.startswith('CONVT:'):
				a = rpn.pop()
				rpn.append(self._rpnEval(a, self._conversions[self._current_type][op[6:]][1]))
			else:
				rpn.append(Decimal(op))

		# return the latest item
		return rpn.pop()

	def getTypeList(self):
		return self._conv_orders['_TYPES_']

	def getUnitList(self):
		return self._conv_orders[self._current_type]

	def setType(self, type):
		if type < len(self._conv_orders['_TYPES_']):
			self._current_type = self._conv_orders['_TYPES_'][type]

	def getType(self):
		return self._current_type

	def doConversion(self, old, new, val):
		# get the user's value
		if val == '':
			return '?'

		# convert the 'from' unit to the base unit
		tmp = self._rpnEval(val, self._conversions[self._current_type][old][0])

		# convert the base unit to the 'to' unit
		tmp = self._rpnEval(tmp, self._conversions[self._current_type][new][1])

		# get rid of exponential notation
		if tmp == tmp.to_integral():
			tmp = tmp.quantize(Decimal(1))
		else:
			tmp = tmp.normalize()

		# return the result as a string
		return str(tmp)
