/*
  PushIt - A simple Sokoban solving game
  Copyright (C) 2006  Tim Teulings

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Sokoban.h"

#include <list>
#include <vector>

#include <assert.h>

Sokoban::Sokoban()
: width(0),height(0)
{
  // no code
}

void Sokoban::SetPackage(size_t x, size_t y)
{
  area[x+y*width]=area[x+y*width] | typePackage;
}
void Sokoban::ClearPackage(size_t x, size_t y)
{
  area[x+y*width]=area[x+y*width] & ~typePackage;
}

void Sokoban::SetSokoban(size_t x, size_t y)
{
  area[x+y*width]=area[x+y*width] | typeSokoban;
}

void Sokoban::ClearSokoban(size_t x, size_t y)
{
  area[x+y*width]=area[x+y*width] & ~typeSokoban;
}

void Sokoban::StartUndo()
{
  undo.push(UndoEntry());
}

void Sokoban::AddCellToUndoEntry(size_t x, size_t y)
{
  UndoCell cell;

  cell.pos=y*width+x;
  cell.type=area[cell.pos];

  undo.top().cells.push_back(cell);
}

void Sokoban::SetAreaSize(size_t width, size_t height)
{
  this->width=width;
  this->height=height;

  area.resize(width*height);

  for (size_t i=0; i<area.size(); ++i) {
    area[i]=typeFloor;
  }
}

void Sokoban::Set(size_t pos, unsigned char type)
{
  area[pos]=type;
}

void Sokoban::Set(size_t x, size_t y, unsigned char type)
{
  area[y*width+x]=type;
}

unsigned char Sokoban::Get(size_t x, size_t y) const
{
  return area[y*width+x];
}

size_t Sokoban::GetWidth() const
{
  return width;
}
size_t Sokoban::GetHeight() const
{
  return height;
}

void Sokoban::GetSokoban(size_t &x, size_t &y) const
{
  size_t pos=0;

  while (!(area[pos] & typeSokoban)) {
    pos++;
  }

  x=pos%width;
  y=pos/width;
}

bool Sokoban::IsIn(size_t x, size_t y) const
{
  return x<width && y << height;
}

bool Sokoban::IsFloor(size_t x, size_t y) const
{
  return area[x+y*width] & typeFloor;
}

bool Sokoban::IsWall(size_t x, size_t y) const
{
  return area[x+y*width] & typeWall;
}

bool Sokoban::IsPackage(size_t x, size_t y) const
{
  return area[x+y*width] & typePackage;
}

bool Sokoban::IsGoal(size_t x, size_t y) const
{
  return area[x+y*width] & typeGoal;
}

bool Sokoban::IsSokoban(size_t x, size_t y) const
{
  return area[x+y*width] & typeSokoban;
}

bool Sokoban::CanMoveLeft() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (x==0) {
    return false;
  }

  if (IsWall(x-1,y)) {
    return false;
  }

  if (IsPackage(x-1,y)) {
    if (x-1==0) {
      return false;
    }

    if (IsWall(x-2,y) || IsPackage(x-2,y)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveRight() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (x>=width-1) {
    return false;
  }

  if (IsWall(x+1,y)) {
    return false;
  }

  if (IsPackage(x+1,y)) {
    if (x+1>=width-1) {
      return false;
    }

    if (IsWall(x+2,y) || IsPackage(x+2,y)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveUp() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (y==0) {
    return false;
  }

  if (IsWall(x,y-1)) {
    return false;
  }

  if (IsPackage(x,y-1)) {
    if (y-1==0) {
      return false;
    }

    if (IsWall(x,y-2) || IsPackage(x,y-2)) {
      return false;
    }
  }

  return true;
}

bool Sokoban::CanMoveDown() const
{
  size_t x,y;

  GetSokoban(x,y);

  if (y>=height-1) {
    return false;
  }

  if (IsWall(x,y+1)) {
    return false;
  }

  if (IsPackage(x,y+1)) {
    if (y+1>=height-1) {
      return false;
    }

    if (IsWall(x,y+2) || IsPackage(x,y+2)) {
      return false;
    }
  }

  return true;
}

void Sokoban::MoveLeft()
{
  assert(CanMoveLeft());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x-1,y);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x-1,y)) {
    AddCellToUndoEntry(x-2,y);
    SetPackage(x-2,y);
    ClearPackage(x-1,y);
  }

  SetSokoban(x-1,y);
  ClearSokoban(x,y);
}

void Sokoban::MoveRight()
{
  assert(CanMoveRight());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x+1,y);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x+1,y)) {
    AddCellToUndoEntry(x+2,y);
    SetPackage(x+2,y);
    ClearPackage(x+1,y);
  }

  SetSokoban(x+1,y);
  ClearSokoban(x,y);
}

void Sokoban::MoveUp()
{
  assert(CanMoveUp());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x,y-1);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x,y-1)) {
    AddCellToUndoEntry(x,y-2);
    SetPackage(x,y-2);
    ClearPackage(x,y-1);
  }

  SetSokoban(x,y-1);
  ClearSokoban(x,y);
}

void Sokoban::MoveDown()
{
  assert(CanMoveDown());

  size_t x,y;

  GetSokoban(x,y);

  StartUndo();

  AddCellToUndoEntry(x,y+1);
  AddCellToUndoEntry(x,y);

  if (IsPackage(x,y+1)) {
    AddCellToUndoEntry(x,y+2);
    SetPackage(x,y+2);
    ClearPackage(x,y+1);
  }

  SetSokoban(x,y+1);
  ClearSokoban(x,y);
}

bool Sokoban::IsFinished() const
{
  for (size_t i=0; i<area.size(); ++i) {
    if (IsGoal(i%width,i/width) && !IsPackage(i%width,i/width)) {
      return false;
    }
  }

  return true;
}

void Sokoban::Undo()
{
  if (undo.empty()) {
    return;
  }

  UndoEntry entry=undo.top();

  undo.pop();

  for (size_t i=0; i<entry.cells.size(); ++i) {
    area[entry.cells[i].pos]=entry.cells[i].type;
  }
}

void Sokoban::SetLineAsText(size_t line, const std::wstring& text)
{
  for (size_t i=0; i<text.length(); ++i) {
    unsigned char type;

    switch (text[i]) {
    case L' ':
      type=Sokoban::typeFloor;
      break;
    case L'#':
      type=Sokoban::typeWall;
      break;
    case L'$':
      type=Sokoban::typeFloor|Sokoban::typePackage;
      break;
    case L'.':
      type=Sokoban::typeGoal;
      break;
    case L'@':
      type=Sokoban::typeFloor|Sokoban::typeSokoban;
      break;
    case L'+':
      type=Sokoban::typeGoal|Sokoban::typeSokoban;
      break;
    case L'*':
      type=Sokoban::typeGoal|Sokoban::typePackage;
      break;
    default:
      assert(false);
      break;
    }
    Set(line*GetWidth()+i,type);
  }
}

void Sokoban::Postprocess()
{
  // We reset every floor information besides
  for (size_t y=0; y<height; y++) {
    for (size_t x=0; x<width; x++) {
      if (IsFloor(x,y) && !(IsWall(x,y) || IsPackage(x,y) || IsGoal(x,y) ||  IsSokoban(x,y))) {
        Set(x,y,Get(x,y) & ~typeFloor);
      }
    }
  }

  /*
    Every neighbour if a floor cell that is not a wall is also a floor cell
    Repeat until no further changes are found.
    This aproach detects floor pieces that are unreachable and marks them as background.
  */

  size_t changes;
  do {
    changes=0;

    for (size_t y=0; y<height; y++) {
      for (size_t x=0; x<width; x++) {
        if (IsFloor(x,y)) {
          // Right
          if (x<width-1 && !IsFloor(x+1,y) && !IsWall(x+1,y)) {
            Set(x+1,y,Get(x+1,y) | typeFloor);
            changes++;
          }
          // Left
          if (x>0 && !IsFloor(x-1,y) && !IsWall(x-1,y)) {
            Set(x-1,y,Get(x-1,y) | typeFloor);
            changes++;
          }
          // Top
          if (y>0 && !IsFloor(x,y-1) && !IsWall(x,y-1)) {
            Set(x,y-1,Get(x,y-1) | typeFloor);
            changes++;
          }
          // Bottom
          if (y<height-1 && !IsFloor(x,y+1) && !IsWall(x,y+1)) {
            Set(x,y+1,Get(x,y+1) | typeFloor);
            changes++;
          }
        }
      }
    }
  } while (changes!=0);
}

