#! /usr/bin/env ruby
#
# $Id$
#
# Pref model visualizer
#

require 'drawing'
require 'model/pref_calc_model'

class PrefCalcViewHelper

  MOUNT_DELIMITER = '⚐'
  POOL_DELIMITER = '⚐'
  WHIST_DELIMITER = '○'

  attr_accessor :selected_round

  def initialize(view)
    @view = view
    @buffer = view.buffer
    @scores = view.scores
    @selected_round = nil
  end

  def share_mount
    return @view.share_mount
  end

  def share_pool
    @view.share_pool
  end

  def share_whists
    @view.share_whists
  end

  #
  # draw the entire board
  #
  # params: cr - context
  #         w, h - the screen size
  #
  def draw_players(cr, w, h)
    w = w/2
    h = h/2
    cr.set_source(0, 0, 0)
    draw_grid(cr, w, h)
    display_names(cr, w, h)

    display_mount(cr, w, h)
    display_pool(cr, w, h)
    display_whists(cr, w, h)
  end

  #
  # concatenate scores into comma-separated list
  #
  def concat_scores(data, hundreds_delimiter)
    rv = ''
    cur_hundred = 0
    data.each_with_index do |elem, idx|
      # skip first element - zero
      if idx != 0
        value = elem.score
        text2display = (value % 100).to_s
        if (value / 100 != cur_hundred)
          text2display = hundreds_delimiter + '.' + text2display
          cur_hundred = value / 100
        end
        if elem.tag == @selected_round
          rv = rv + '<span weight="bold" foreground="red">' + text2display + '</span>'
        else
          rv = rv + text2display
        end
        if idx < (data.length - 1)
          rv = rv + '.'
        end
      end
    end
    return rv
  end

  #
  # display the user name
  #
  # params: cr - context
  #         w, h - half of the screen size
  #         orientation - 0/90/180/270, from 12 o'clock clockwise
  #
  def display_name_in_quarter(cr, name, w, h, orientation)

    pl = @view.create_pango_layout
    pl.ellipsize = Pango::Layout::ELLIPSIZE_END
    pl.wrap = Pango::Layout::WRAP_CHAR
    pl.text = name
    angle = Math::PI * (orientation - 180)/180

    case orientation
      when 0, 180
        # approximately 1/3 of the mount space
        name_offset = ((share_mount * h)/(3.0 * pl.pixel_size[1])).to_i
        name_offset = 1 if name_offset < 1
        if h != 0
          pl.width = Pango::SCALE*2*name_offset*pl.pixel_size[1]*w/h
        end
        base_x = w + ((orientation == 0) ? pl.pixel_size[0]/2 :
                                           -pl.pixel_size[0]/2)
        base_y = h + ((orientation == 0) ? -pl.pixel_size[1]*(name_offset + 1) : 
                                            pl.pixel_size[1]*name_offset)
      when 90, 270
        # approximately 1/3 of the mount space
        name_offset = ((share_mount * w)/(3.0 * pl.pixel_size[1])).to_i
        name_offset = 1 if name_offset < 1
        if w != 0
          pl.width = Pango::SCALE*2*name_offset*pl.pixel_size[1]*h/w
        end
        base_x = w + ((orientation == 270) ? -pl.pixel_size[1]*(name_offset + 1) : 
                                              pl.pixel_size[1]*name_offset)
        base_y = h + ((orientation == 270) ? -pl.pixel_size[0]/2 :
                                             pl.pixel_size[0]/2)
    end

    cr.move_to(base_x, base_y)
    cr.rotate(angle)
    cr.update_pango_layout(pl)
    cr.show_pango_layout(pl);       
    cr.rotate(-angle)
  end

  #
  # calculate the position for the text output
  #
  # params: pl - pango layout
  #         screen_dimensions - halved dimensions of the screen, 
  #                            [x, y] or [y, x] depending on the swapped parameter
  #         share - the ratio to be used as the basis for the text 
  #                 (0-bottom of the sector, 1-center of the screen)
  #         part - which "horizontal" part of the sector is used, for whists only
  #                (1-total_parts)
  #         total_parts - total number of parts 
  #                       (for pool/mount 1, for whists 2 or 3)
  #         orientation - 0/90/180/270, from 12 o'clock clockwise
  #         swapped - whether screen_dimensions are swapped or not
  #
  # returns: position
  #
  def calc_position(pl, screen_dimensions, share, orientation, part, total_parts, swapped)
    w = screen_dimensions[0]
    h = screen_dimensions[1]
    if h != 0
      # text height
      th = pl.pixel_size[1]
      # gap (because of diagonal line)
      dx = 1.0 * w * th/h
      # sector limits
      x1 = w * share
      x2 = 2 * w - x1
      # part limits
      x1p = x1 + (part - 1) * (x2 - x1)/total_parts
      x2p = x1 + part * (x2 - x1)/total_parts
      # text width
      tw = x2p - x1p
      tw -= dx if part == 1
      tw -= dx if part == total_parts
      pl.width = Pango::SCALE * tw
      y = h * share + th
      # possible drawing corners
      x1l = x1p
      x2l = x2p
      case orientation 
        when 0
          # North
	  case @scores.game.num_players
	    when 4
             x = x2l
             x -= dx if part == total_parts
	    when 3
	     case total_parts
	      when 2
	       x = x2l
              when 1
	       x = 2*w
             end
	  end
        when 360
          # West
          x = x1l
          x += dx if part == 1
        when 180
          y = 2 * h - y
          if swapped
            # East
            x = x2l
            x -= dx if part == total_parts
          else
            # South
            x = x1l
            x += dx if part == 1
          end
      end
    end
    return [x, y]
  end

  #
  # display scores
  #
  # params: scores - the scores (list of TaggedScore objects)
  #         delimiter - unicode char used to delimit hundreds
  #         cr - context
  #         w, h - half-size of the screen
  #         orientation - 0/90/180/270, from 12 o'clock clockwise
  #         part - which "horizontal" part of the sector is used, for whists only
  #                (1-total_parts)
  #         total_parts - total number of parts 
  #                       (for pool/mount 1, for whists 2 or 3)
  #
  def display_scores(scores, hundreds_delimiter, cr, w, h, share, orientation, part=1, total_parts=1)
    pl = @view.create_pango_layout
    pl.ellipsize = Pango::Layout::ELLIPSIZE_START
    pl.wrap = Pango::Layout::WRAP_CHAR
    text2display = concat_scores(scores, hundreds_delimiter)
    pl.markup = text2display
    case orientation
      when 0,180
        coords = calc_position(pl, [w, h], share, orientation, part, total_parts, false)
        x = coords[0]
        y = coords[1]
      when 90,270
        coords = calc_position(pl, [h, w], share, orientation + 90, part, total_parts, true)
        x = coords[1]
        y = coords[0]
    end
    if not (x.nil? or y.nil?)
      cr.move_to(x, y)
      angle = Math::PI * (orientation - 180)/180
      cr.rotate(angle)
      cr.update_pango_layout(pl)
      cr.show_pango_layout(pl);       
      cr.rotate(-angle)
    end
  end

end

class PrefCalcViewHelper3 < PrefCalcViewHelper

  def initialize(view)
    super
  end

  #
  # draw the grid - all lines
  #
  def draw_grid(cr, w, h)
    cr.antialias = Cairo::ANTIALIAS_SUBPIXEL
    cr.line_cap = Cairo::LineCap::ROUND
    cr.line_join = Cairo::LineJoin::ROUND
    cr.set_dash(nil)

    # 3 sectors
    cr.move_to(0, 0)
    cr.line_to(w, h)
    cr.move_to(0, 2*h)
    cr.line_to(w, h)
    cr.move_to(w, h)
    cr.line_to(w*2, h)

    # m/p/w
    # [0], horizontal
    cr.move_to(share_whists*w, share_whists*h)
    cr.line_to(2*w, share_whists*h)
    cr.move_to((share_whists + share_pool)*w, (share_whists + share_pool)*h)
    cr.line_to(2*w, (share_whists + share_pool)*h)

    # [1], horizontal
    cr.move_to(share_whists*w, (2 - share_whists)*h)
    cr.line_to(2*w, (2 - share_whists)*h)
    cr.move_to((share_whists + share_pool)*w, (2 - share_whists - share_pool)*h)
    cr.line_to(2*w, (2 - share_whists - share_pool)*h)

    # [2], vertical
    cr.move_to(share_whists*w, share_whists*h)
    cr.line_to(share_whists*w, (2 - share_whists)*h)

    cr.move_to((share_whists + share_pool)*w, (share_whists + share_pool)*h)
    cr.line_to((share_whists + share_pool)*w, (2 - share_whists - share_pool)*h)
    cr.stroke

    # separate whists
    cr.set_dash(10)
    cr.move_to(w, 0)
    cr.line_to(w, share_whists*h)
    cr.move_to(w, (2 - share_whists)*h)
    cr.line_to(w, 2*h)
    cr.move_to(0, h)
    cr.line_to(share_whists*w, h)
    cr.stroke
  end

  def display_names(cr, w, h)
    # [0], horizontal
    display_name_in_quarter(cr, @scores.game.accounts[0].name, w, h, 0)

    # [1], horizontal
    display_name_in_quarter(cr, @scores.game.accounts[1].name, w, h, 180)

    # [2]: vertical
    display_name_in_quarter(cr, @scores.game.accounts[2].name, w, h, 270)
  end

  def display_mount(cr, w, h)
    display_scores(@scores.player_scores[0].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 0)
    display_scores(@scores.player_scores[1].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 180)
    display_scores(@scores.player_scores[2].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 270)
  end

  def display_pool(cr, w, h)
    display_scores(@scores.player_scores[0].pool, POOL_DELIMITER, cr, w, h, share_whists, 0)
    display_scores(@scores.player_scores[1].pool, POOL_DELIMITER, cr, w, h, share_whists, 180)
    display_scores(@scores.player_scores[2].pool, POOL_DELIMITER, cr, w, h, share_whists, 270)
  end

  def display_whists(cr, w, h)
    display_scores(@scores.player_scores[0].whists[2], WHIST_DELIMITER, cr, w, h, 0, 0, 1, 2)
    display_scores(@scores.player_scores[0].whists[1], WHIST_DELIMITER, cr, w, h, 0, 0, 2, 2)

    display_scores(@scores.player_scores[1].whists[2], WHIST_DELIMITER, cr, w, h, 0, 180, 1, 2)
    display_scores(@scores.player_scores[1].whists[0], WHIST_DELIMITER, cr, w, h, 0, 180, 2, 2)

    display_scores(@scores.player_scores[2].whists[0], WHIST_DELIMITER, cr, w, h, 0, 270, 1, 2)
    display_scores(@scores.player_scores[2].whists[1], WHIST_DELIMITER, cr, w, h, 0, 270, 2, 2)
  end
end

class PrefCalcViewHelper4 < PrefCalcViewHelper

  def initialize(view)
    super
  end

  #
  # draw the grid - all lines
  #
  def draw_grid(cr, w, h)
    cr.antialias = Cairo::ANTIALIAS_SUBPIXEL
    cr.line_cap = Cairo::LineCap::ROUND
    cr.line_join = Cairo::LineJoin::ROUND
    cr.line_width = 0.5
    cr.set_dash(nil)

    # 4 sectors
    cr.move_to(0, 0); cr.line_to(2*w, 2*h)
    cr.move_to(0, 2*h); cr.line_to(2*w, 0)

    # m/p/w
    cr.move_to(share_whists*w, share_whists*h)
    cr.line_to((2 - share_whists)*w, share_whists*h)
    cr.move_to((share_whists + share_pool)*w, (share_whists + share_pool)*h)
    cr.line_to((2 - share_whists - share_pool)*w, (share_whists + share_pool)*h)

    cr.move_to(share_whists*w, (2 - share_whists)*h)
    cr.line_to((2 - share_whists)*w, (2 - share_whists)*h)
    cr.move_to((share_whists + share_pool)*w, (2 - share_whists - share_pool)*h)
    cr.line_to((2 - share_whists - share_pool)*w, (2 - share_whists - share_pool)*h)

    cr.move_to(share_whists*w, share_whists*h)
    cr.line_to(share_whists*w, (2 - share_whists)*h)

    cr.move_to((share_whists + share_pool)*w, (share_whists + share_pool)*h)
    cr.line_to((share_whists + share_pool)*w, (2 - share_whists - share_pool)*h)

    cr.move_to((2-share_whists)*w, share_whists*h)
    cr.line_to((2-share_whists)*w, (2 - share_whists)*h)

    cr.move_to((2 - share_whists - share_pool)*w, (share_whists + share_pool)*h)
    cr.line_to((2 - share_whists - share_pool)*w, (2 - share_whists - share_pool)*h)
    cr.stroke

    # separate whists
    cr.set_dash(3)
    cr.move_to(2*w/3, 0)
    cr.line_to(2*w/3, share_whists*h)
    cr.move_to(4*w/3, 0)
    cr.line_to(4*w/3, share_whists*h)

    cr.move_to(2*w/3, (2 - share_whists)*h)
    cr.line_to(2*w/3, 2*h)
    cr.move_to(4*w/3, (2 - share_whists)*h)
    cr.line_to(4*w/3, 2*h)

    cr.move_to(0, 2*h/3)
    cr.line_to(share_whists*w, 2*h/3)
    cr.move_to(0, 4*h/3)
    cr.line_to(share_whists*w, 4*h/3)

    cr.move_to((2 - share_whists)*w, 2*h/3)
    cr.line_to(2*w, 2*h/3)
    cr.move_to((2 - share_whists)*w, 4*h/3)
    cr.line_to(2*w, 4*h/3)
    cr.stroke
  end

  #
  # display all users' names
  #
  def display_names(cr, w, h)
    display_name_in_quarter(cr, @scores.game.accounts[0].name, w, h, 0)
    display_name_in_quarter(cr, @scores.game.accounts[1].name, w, h, 90)
    display_name_in_quarter(cr, @scores.game.accounts[2].name, w, h, 180)
    display_name_in_quarter(cr, @scores.game.accounts[3].name, w, h, 270)
  end

  #
  # display all mount scores
  #
  def display_mount(cr, w, h)
    display_scores(@scores.player_scores[0].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 0)
    display_scores(@scores.player_scores[1].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 90)
    display_scores(@scores.player_scores[2].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 180)
    display_scores(@scores.player_scores[3].mount, MOUNT_DELIMITER, cr, w, h, share_whists + share_pool, 270)
  end

  #
  # display all pool scores
  #
  def display_pool(cr, w, h)
    display_scores(@scores.player_scores[0].pool, POOL_DELIMITER, cr, w, h, share_whists, 0)
    display_scores(@scores.player_scores[1].pool, POOL_DELIMITER, cr, w, h, share_whists, 90)
    display_scores(@scores.player_scores[2].pool, POOL_DELIMITER, cr, w, h, share_whists, 180)
    display_scores(@scores.player_scores[3].pool, POOL_DELIMITER, cr, w, h, share_whists, 270)
  end

  #
  # display all whists
  #
  def display_whists(cr, w, h)
    display_scores(@scores.player_scores[0].whists[3], WHIST_DELIMITER, cr, w, h, 0, 0, 1, 3)
    display_scores(@scores.player_scores[0].whists[2], WHIST_DELIMITER, cr, w, h, 0, 0, 2, 3)
    display_scores(@scores.player_scores[0].whists[1], WHIST_DELIMITER, cr, w, h, 0, 0, 3, 3)

    display_scores(@scores.player_scores[1].whists[0], WHIST_DELIMITER, cr, w, h, 0, 90, 1, 3)
    display_scores(@scores.player_scores[1].whists[3], WHIST_DELIMITER, cr, w, h, 0, 90, 2, 3)
    display_scores(@scores.player_scores[1].whists[2], WHIST_DELIMITER, cr, w, h, 0, 90, 3, 3)

    display_scores(@scores.player_scores[2].whists[3], WHIST_DELIMITER, cr, w, h, 0, 180, 1, 3)
    display_scores(@scores.player_scores[2].whists[0], WHIST_DELIMITER, cr, w, h, 0, 180, 2, 3)
    display_scores(@scores.player_scores[2].whists[1], WHIST_DELIMITER, cr, w, h, 0, 180, 3, 3)

    display_scores(@scores.player_scores[3].whists[0], WHIST_DELIMITER, cr, w, h, 0, 270, 1, 3)
    display_scores(@scores.player_scores[3].whists[1], WHIST_DELIMITER, cr, w, h, 0, 270, 2, 3)
    display_scores(@scores.player_scores[3].whists[2], WHIST_DELIMITER, cr, w, h, 0, 270, 3, 3)
  end
end

class PrefCalcView < Canvas

  attr_reader :buffer, :scores
  attr_reader :share_mount, :share_pool, :share_whists

  def initialize
    super()

    # sum = 1
    @share_mount = 0.5
    @share_pool = 0.3
    @share_whists = 0.2
  end

  def scores=(scores)
    @scores = scores
    @scores.game.add_listener(self) if @scores
    game_updated(nil)
  end

  def game_updated(hand_results)
    self.select_round_by_index(-1)
  end

  def configure_event(w, e)
    super

    return if @buffer.nil?
    return if @scores.nil?

    prepare_buffer(-1)
  end

  def prepare_buffer(selected_round_index)
    return if @scores.nil?
    g = self.window.geometry
    return if !(g[2] > 0 && g[3] > 0)

    cr = Cairo::Context.new(@buffer)

    w = g[2]
    h = g[3]

    background_pixbuf = PrefUtil.background_pixbuf
    sx = w*1.0/background_pixbuf.width
    sy = h*1.0/background_pixbuf.height
    cr.scale(sx, sy)

    bgis = Cairo::ImageSurface.from_png(PrefUtil.background_png_path)
    cr.set_source(bgis, 0, 0)
    cr.paint
    cr.scale(1/sx, 1/sy)

    case @scores.game.num_players 
      when 3:
        @helper = PrefCalcViewHelper3.new(self)
      when 4:
        @helper = PrefCalcViewHelper4.new(self)
    end
    if (selected_round_index == -1)
      @helper.selected_round = @scores.game.rounds.last
    else
      @helper.selected_round = @scores.game.rounds[selected_round_index]
    end
    # We want to highlight that game
    @helper.draw_players(cr, w, h)
  end

  def select_round_by_index(idx)
    return if !self.window

    g = self.window.geometry
    return if !(g[2] > 0 && g[3] > 0)

    clear
    prepare_buffer(idx)
    queue_draw_area(0, 0, g[2], g[3])
  end
end
