#!/usr/bin/env ruby

#
# $Id$
#
# Pref scores model
#

require 'model/pref_calc_model'

class TaggedScore

  attr_reader :score, :tag

  def initialize(s, t)
    @score = s
    @tag = t
  end

  def inspect
    score.to_s + "(" + tag.inspect + ")"
  end
end

#
# Array with [a1, a2, ..] inspect
# 
class List < Array

  def inspect
    s = "["
    self.each_index do |idx| 
      s += ", " if idx != 0
      s += "#{self[idx]}"
    end
    return s + "]"
  end
end


#
# Accounting per-player
#
class PlayerScores

  attr_reader :pool, :mount, :whists

  def initialize(num_players)
    @pool = List.[]( TaggedScore.new(0, nil) )
    @mount = List.[]( TaggedScore.new(0, nil) )
    @whists = List.new(num_players) { List.[]( TaggedScore.new(0, nil) ) }
  end

  def add_pool(value, tag)
    if value != 0
      @pool << TaggedScore.new(pool_value + value, tag)
    end
  end

  def add_mount(value, tag)
    if value != 0
      @mount << TaggedScore.new(mount_value + value, tag)
    end
  end

  def add_whists(player_idx, value, tag)
    if value != 0
      @whists[player_idx] << TaggedScore.new(whists_value(player_idx) + value, tag)
    end
  end

  def inspect
    s = "    Pool: #{@pool.inspect}\n"
    s += "    Mount: #{@mount.inspect}\n    Whists: "
    @whists.each_index do |idx|
      s += ", " if idx != 0
      s += "#{idx} -> #{@whists[idx].inspect}"
    end
    return s + "\n"
  end

  def pool_value
    @pool.last.score
  end

  def mount_value
    @mount.last.score
  end

  def whists_value(player_idx)
    @whists[player_idx].last.score
  end
end

#
# All game information
#
class GameScores

  attr_reader :player_scores, :model, :variant, :game
  attr_accessor :pass_out_counter

  def initialize(game)
    @game = game

    @player_scores = Array.new()
    num_players = @game.accounts.length
    @game.accounts.each do |p| 
      @player_scores.push(PlayerScores.new(num_players))
    end

    @game.add_listener(self)
  end

  def add_pool(player_idx, value, tag)
    @player_scores[player_idx].add_pool(value, tag)
  end

  def add_mount(player_idx, value, tag)
    @player_scores[player_idx].add_mount(value, tag)
  end

  def add_whists(player_idx, player_idx2, value, tag)
    @player_scores[player_idx].add_whists(player_idx2, value, tag)
  end

  def inspect
    s = "Total players: #{@player_scores.length}\n"
    @player_scores.each_index do |idx| 
      s += "  Player #{idx}:\n" + @player_scores[idx].inspect
    end
    return s
  end

  def append(hand_results)

    variant = @game.variant

    case hand_results.type
      when GameModel::Type::POSITIVE
        total_whists = hand_results.tricks_by_whister1 + hand_results.tricks_by_whister2
        untaken_tricks_by_bid_winner = hand_results.declared_game - 
                                       10 + total_whists

        # game coefficient: 1, 2, 3, 4
        game_coeff = (hand_results.declared_game - 5)
        whisters_tricks_reqd = [-1, -1, -1, -1, -1, -1,  4, 2, 1, 1, 0]
        each_whister_tricks_reqd = [-1, -1, -1, -1, -1, -1, 2, 1, 1, 1, 0]

        whister_idxs = @game.get_whister_idxs(hand_results.dealer, 
                                              hand_results.bid_winner)

        if (untaken_tricks_by_bid_winner <= 0)
          # game played exactly or even better
          variant.reset_pass_out(@game)
          v1 = hand_results.whister1_defending ? 
               (hand_results.whister2_defending ? 
                  hand_results.tricks_by_whister1 :
                  total_whists) * 
                 variant.trick_price_as_whist * 
                 game_coeff : 
               0;
          v2 = hand_results.whister2_defending ? 
               (hand_results.whister1_defending ? 
                  hand_results.tricks_by_whister2 :
                  total_whists) * 
                 variant.trick_price_as_whist * 
                 game_coeff : 
               0;
          add_pool(hand_results.bid_winner, 
                   variant.base_game_price * game_coeff,
                   hand_results)
          untaken_tricks_by_whisters = whisters_tricks_reqd[hand_results.declared_game] -
                                         hand_results.tricks_by_whister1 -
                                         hand_results.tricks_by_whister2 
          # if whisters did not take enough tricks
          if (untaken_tricks_by_whisters > 0)
            if (hand_results.whister1_defending && 
                hand_results.whister2_defending)            
              # if two whisters were on their own - add mount to the ones which did not take
              untaken_tricks_by_whister1 = each_whister_tricks_reqd[hand_results.declared_game] -
                                           hand_results.tricks_by_whister1 
              untaken_tricks_by_whister2 = each_whister_tricks_reqd[hand_results.declared_game] -
                                           hand_results.tricks_by_whister2 
              add_mount(whister_idxs[0], 
                        variant.base_untaken_trick_price_on_whist * 
                          untaken_tricks_by_whister1 * 
                          game_coeff,
                        hand_results)
              add_mount(whister_idxs[1], 
                        variant.base_untaken_trick_price_on_whist * 
                          untaken_tricks_by_whister2 * 
                          game_coeff,
                        hand_results)
            else
              if hand_results.whister1_defending
                add_mount(whister_idxs[0], 
                          variant.base_untaken_trick_price_on_whist * 
                            untaken_tricks_by_whisters * 
                            game_coeff,
                          hand_results)
              end
              if hand_results.whister2_defending
                add_mount(whister_idxs[1], 
                          variant.base_untaken_trick_price_on_whist * 
                            untaken_tricks_by_whisters * 
                            game_coeff,
                          hand_results)
              end
              # add half-whist, if necessary
              if !hand_results.half_whist_to.nil?
                add_whists(hand_results.half_whist_to, hand_results.bid_winner, 
                           whisters_tricks_reqd[hand_results.declared_game]/2, hand_results)
              end
            end
          end
        else
          # player failed
          add_mount(hand_results.bid_winner, 
                    variant.base_untaken_trick_price_on_game * 
                      untaken_tricks_by_bid_winner * 
                      game_coeff,
                    hand_results)
          if (hand_results.whister1_defending && hand_results.whister2_defending)
            v1 = (hand_results.tricks_by_whister1 + untaken_tricks_by_bid_winner) * 
                   variant.trick_price_as_whist * game_coeff;
            v2 = (hand_results.tricks_by_whister2 + untaken_tricks_by_bid_winner) * 
                   variant.trick_price_as_whist * game_coeff;
          else
            if variant.is_whist_gentlemanish
              # Gentleman's whists - share the victory!
              v1 = ((total_whists + untaken_tricks_by_bid_winner) * 
                     variant.trick_price_as_whist * game_coeff)/2;
              v2 = v1;
            else
              # Non-gentleman's whists - non-defendant whists gets only diff
              v1 = (total_whists * variant.trick_price_as_whist * game_coeff);
              v2 = (untaken_tricks_by_bid_winner * 
                     variant.trick_price_as_whist * game_coeff);
              # Swap if 2nd whister was defending
              if (hand_results.whister2_defending)
                v = v1; v1 = v2; v2 = v
              end
            end
          end
        end
        add_whists(whister_idxs[0], hand_results.bid_winner, v1, hand_results)
        add_whists(whister_idxs[1], hand_results.bid_winner, v2, hand_results)
        if !@game.model.dealer_plays?
          # Can only be 0 or 1 - for returned star
          add_mount(hand_results.dealer, 
                    variant.base_untaken_trick_price_on_game * 
                      hand_results.mount_tricks_for_dealer *
                      game_coeff,
                    hand_results)
          # For every untaken trick on positive hame - add whists for the dealer
          add_whists(hand_results.dealer, hand_results.bid_winner,
                     hand_results.tricks_on_bid_winner_for_dealer *
                       variant.trick_price_as_whist * game_coeff,
                     hand_results);
        end
      when GameModel::Type::MIZERE
        if (hand_results.tricks_by_mizere_player == 0)
          add_pool(hand_results.mizere_player, 10, hand_results)
        else
          add_mount(hand_results.mizere_player, 10*hand_results.tricks_by_mizere_player, hand_results)
        end
        if !@game.model.dealer_plays?
          # Can only be 0 or 1 - for returned star
          add_mount(hand_results.dealer, 
                    10 * hand_results.mount_tricks_for_dealer,
                    hand_results) 
          # For every taken trick on mizere - add whists for the dealer
          add_whists(hand_results.dealer, hand_results.mizere_player,
                     10 * hand_results.tricks_on_bid_winner_for_dealer,
                     hand_results)
        end
      when GameModel::Type::PASS_OUT
        pass_out_idxs = @game.get_pass_out_idxs(hand_results.dealer)
        if @game.model.dealer_plays?
          min_tricks = [hand_results.tricks_by_player1,
                        hand_results.tricks_by_player2,
                        hand_results.tricks_by_player3].min
        else
          min_tricks = [hand_results.tricks_by_player1,
                        hand_results.tricks_by_player2,
                        hand_results.tricks_by_player3,
                        hand_results.mount_tricks_for_dealer].min
        end
        price = variant.taken_trick_price_on_pass_out(@game)
        add_mount(pass_out_idxs[0], 
                  price * 
                      (hand_results.tricks_by_player1 - min_tricks),
                  hand_results)
        add_mount(pass_out_idxs[1], 
                  price * 
                      (hand_results.tricks_by_player2 - min_tricks),
                  hand_results)
        add_mount(pass_out_idxs[2], 
                  price * 
                      (hand_results.tricks_by_player3 - min_tricks),
                  hand_results)

        add_mount(hand_results.dealer, 
                  price * 
                      (hand_results.mount_tricks_for_dealer - min_tricks),
                  hand_results) if !@game.model.dealer_plays?

        [hand_results.tricks_by_player1,
         hand_results.tricks_by_player2,
         hand_results.tricks_by_player3].each_with_index {|nt, idx|
          # TODO: parametrize?
          # put the game into pool for the lucky one, taken 0 tricks
          if (nt == 0)
            add_pool(pass_out_idxs[idx],
                     variant.base_game_price * (@game.pass_out_counter - 5),
                     hand_results)
          end
        }
        variant.another_pass_out(@game)
    end
  end

  #
  # Calculate final results of the game
  #
  def scores

    # Is 20, 10 for Leningrad version only?
    base_scores = @player_scores.collect do |acc| 
       (20 * acc.pool.last.score) - (10 * acc.mount.last.score)
    end

    # Guarantee that total sum is 0
    delta = base_scores.inject { |sum, s| sum + s } / @player_scores.length.to_f

    # First, try to round down
    idelta = delta.floor.to_i
    final_scores = base_scores.collect { |val| val - idelta }
    if final_scores.inject { |sum, s| sum + s } != 0
      max = final_scores.max
      final_scores.collect! do |val|
        val==max ? val - 1 : val
      end
    end
    if final_scores.inject { |sum, s| sum + s } != 0
      # If down is not good - try rounding up
      idelta = delta.ceil.to_i
      final_scores = base_scores.collect { |val| val - idelta }
      if final_scores.inject { |sum, s| sum + s } != 0
        max = final_scores.max
        final_scores.collect! do |val|
          val==max ? val + 1 : val
        end
      end
    end

    # +- Whists
    @player_scores.each_with_index do |acc, idx1| 
      acc.whists.each_with_index do |wh, idx2|
        final_scores[idx1] += wh.last.score
        final_scores[idx2] -= wh.last.score
      end
    end

    return final_scores
  end

  #
  # Listener method - notified by the game
  #
  def game_updated(hand_results)
    append(hand_results)
  end

end


