#!/usr/bin/env ruby

#
# $Id$
#
# Pref game model
#

#
# Player as such
#
class PlayerAccount

  attr_reader :name

  def initialize(name)
    @name = name
  end
end

#
# All game information
#
class Game

  attr_reader :accounts, :model, :variant, :rounds, :whist_price
  attr_accessor :pass_out_counter

  def initialize(variant, player_names, whist_price)
    @variant = variant
    @accounts = Array.new()
    player_names.each do |p| 
      @accounts.push(PlayerAccount.new(p))
    end
    case player_names.length
      when 3
        @model = GameModel::GM3.new
      when 4
        @model = GameModel::GM4.new
    end
    @variant.reset_pass_out(self)

    @listeners = Array.new()
    @rounds = Array.new()
    @dealer_idx = 0
    @whist_price = whist_price
  end

  def add_listener(l)
    @listeners.push(l)
  end

  def inspect
    s = "Total players: #{@accounts.length}\n"
    @accounts.each_index do |idx| 
      s += "  Player #{idx}:\n" + @accounts[idx].inspect
    end
    return s
  end

  # Array of all player names
  def names
    @accounts.map { |acc| acc.name }
  end

  # Total number of players in the game
  def num_players
    @accounts.length
  end

  # Find the player by name
  # Ugly implementation
  def idx_by_name(name)
    @accounts.each_index do |idx| 
      if @accounts[idx].name == name
        return idx
      end
    end
    return nil
  end

  def append(hand_results)
    @rounds.push(hand_results)

    @dealer_idx = (@dealer_idx + 1) % @accounts.length

    # Notify interested objects
    @listeners.each { |l| l.game_updated(hand_results) }
  end

  # Get the list of names of whisters, in proper order
  def get_whister_names(dealer_name, bid_winner_name)
    dnames = names
    dnames.delete_if { |name| name == dealer_name } if not @model.dealer_plays?
    dnames = dnames * 2
    dnames = dnames.slice(dnames.index(bid_winner_name) + 1, 2)
    dnames
  end

  # Get the list of ids of whisters, in proper order
  def get_whister_idxs(dealer_idx, bid_winner_idx)
    idxs = (0..@accounts.length-1).to_a
    idxs.delete_if { |idx| idx == dealer_idx } if not @model.dealer_plays?
    idxs = idxs * 2
    idxs = idxs.slice(idxs.index(bid_winner_idx) + 1, 2)
    idxs
  end

  def get_pass_out_names(dealer_name)
    dnames = names * 2
    dnames.delete_if { |name| name == dealer_name } if not @model.dealer_plays?
    idx = dnames.index(dealer_name)
    dnames = dnames.slice(idx.nil? ? 0 : idx + 1, 3)
    dnames
  end

  def get_pass_out_idxs(dealer_idx)
    idxs = (0..@accounts.length-1).to_a
    first_idx = idxs.index(dealer_idx)
    if @model.dealer_plays?
      first_idx += 1
    else
      idxs.delete_if { |idx| idx == dealer_idx } 
    end
    idxs = idxs * 2
    idxs = idxs.slice(first_idx, 3)
    idxs
  end

  def dealer
    @accounts[@dealer_idx]
  end
end

#
# Game model
#
class GameModel
  include GetText

  # What kind of game
  class Type
    POSITIVE = 0
    MIZERE = 1
    PASS_OUT = 2
  end

  TYPE_NAME = ["Positive", "Mizere", "Pass Out"]

  # Whether dealer is allowed to play
  def dealer_plays?
    nil
  end

  # Check the rules whether combination of tricks, defence flags etc is valid
  def validate_positive_game(bid, whister1_tricks, whister2_tricks,
                             defending_whister1, defending_whister2)
    if (!(defending_whister1 || defending_whister2) &&
         (whister1_tricks > 0 || whister2_tricks > 0))
      return Hash[HandResult::VALIDATION_RESULT=>false,
                  HandResult::VALIDATION_MESSAGE=>_('Some tricks are given to whisters without defence')]
    end
    if (whister1_tricks + whister2_tricks > 10)
      return Hash[HandResult::VALIDATION_RESULT=>false,
                  HandResult::VALIDATION_MESSAGE=>_('Total number of whists taken cannot exceed 10')]
    end
    return true
  end

  # Checks whether the tricks compose a full pass-out game
  def validate_pass_out(dealer_tricks, player1_tricks, player2_tricks, player3_tricks)
    result = ((dealer_tricks + player1_tricks + player2_tricks + player3_tricks) == 10)
    return result ? true :
                    Hash[HandResult::VALIDATION_RESULT=>false,
                         HandResult::VALIDATION_MESSAGE=>_('Total number of whists taken must be equal 10')]
  end

  #
  # Data structure for keeping the game results
  #
  class HandResult

    VALIDATION_RESULT = 'if_valid'
    VALIDATION_MESSAGE = 'validation_message'

    attr_reader :game
    attr_accessor :type, :dealer, :mount_tricks_for_dealer, :tricks_on_bid_winner_for_dealer

    def initialize(game, t, d, mtfd, tobwfd)
      @game = game
      @type = t
      @dealer = d
      @mount_tricks_for_dealer = mtfd
      @tricks_on_bid_winner_for_dealer = tobwfd
    end

    def inspect
      mt = @mount_tricks_for_dealer == 0 ? "" : format("mount tricks %s", @mount_tricks_for_dealer)
      tobw = @tricks_on_bid_winner_for_dealer == 0 ? "" : format("tricks on bid winner %s", @tricks_on_bid_winner_for_dealer)
      if (mt == "")
        if (tobw == "")
          dt = ""
        else
          dt = format("(%s)", tobw)
        end
      else
        if (tobw == "")
          dt = format("(%s)", mt)
        else
          dt = format("(%s, %s)", mt, tobw)
        end
      end
      return format('Type: %s, dealer: "%s"%s',
                    GameModel::TYPE_NAME[@type],
                    @game.accounts[@dealer].name,
                    dt)
    end

    class Positive < HandResult
      attr_accessor :declared_game, :bid_winner
      attr_accessor :tricks_by_whister1, :tricks_by_whister2
      attr_accessor :whister1_defending, :whister2_defending
      attr_accessor :half_whist_to

      def initialize(game, d, mtfd, tobwfd, dg, bw, tbw1, tbw2, w1d, w2d, hw2)
        super(game, Type::POSITIVE, d, mtfd, tobwfd)
        @declared_game = dg
        @bid_winner = bw
        @tricks_by_whister1 = tbw1
        @tricks_by_whister2 = tbw2
        @whister1_defending = w1d
        @whister2_defending = w2d
        @half_whist_to = hw2
      end

      def inspect
        widxs = @game.get_whister_idxs(@dealer, @bid_winner)
        wn1 = @game.accounts[widxs[0]].name
        wn2 = @game.accounts[widxs[1]].name
        hw2 = @half_whist_to.nil? ? 'nobody' : @game.accounts[@half_whist_to].name
        return super + format(', declared game: %s, bid winner: "%s", ' +
                               'tricks by "%s": %s%s, tricks by "%s": %s%s, ' +
                               'half whist to %s',
                              @declared_game, 
                              @game.accounts[@bid_winner].name,
                              wn1, @tricks_by_whister1, @whister1_defending ? " (defending)" : "",
                              wn2, @tricks_by_whister2, @whister2_defending ? " (defending)" : "",
                              hw2)
      end
    end

    class Mizere < HandResult
      attr_accessor :mizere_player
      attr_accessor :tricks_by_mizere_player

      def initialize(game, d, mtfd, tobwfd, mp, tbmp)
        super(game, Type::MIZERE, d, mtfd, tobwfd)
        @mizere_player = mp
        @tricks_by_mizere_player = tbmp
      end

      def inspect
        return super + format(', mizere player: "%s" (%s tricks)', 
                              @game.accounts[@mizere_player].name, @tricks_by_mizere_player)
      end
    end

    class PassOut < HandResult
      attr_accessor :tricks_by_player1
      attr_accessor :tricks_by_player2
      attr_accessor :tricks_by_player3

      def initialize(game, d, mtfd, tbp1, tbp2, tbp3)
        super(game, Type::PASS_OUT, d, mtfd, 0)
        @tricks_by_player1 = tbp1
        @tricks_by_player2 = tbp2
        @tricks_by_player3 = tbp3
      end

      def inspect
        idxs = @game.get_pass_out_idxs(@dealer)
        return super + format(', tricks by "%s": %s, tricks by "%s": %s, tricks by "%s": %s', 
                              @game.accounts[idxs[0]].name, @tricks_by_player1,
                              @game.accounts[idxs[1]].name, @tricks_by_player2,
                              @game.accounts[idxs[2]].name, @tricks_by_player3)
      end
    end

  end

  class GM3 < GameModel
    def dealer_plays?
      true
    end
  end

  class GM4 < GameModel
    def dealer_plays?
      false
    end
  end
end

#
# Variations of some rules
# Abstract class
#
class GameVariant
  # How much one trick gives into whists
  def trick_price_as_whist
    raise 'Not implemented'
  end

  # How much goes to the pool for a game (6)
  def base_game_price
    raise 'Not implemented'
  end

  # How much goes to mount for untaken trick (6)
  def base_untaken_trick_price_on_game
    raise 'Not implemented'
  end

  def base_untaken_trick_price_on_whist
    raise 'Not implemented'
  end

  # How much goes to mount for one trick on pass-out
  # (depending on the game state)
  def taken_trick_price_on_pass_out(game)
    raise 'Not implemented'
  end

  # Reset the game state (for pass-outs)
  def reset_pass_out(game)
    raise 'Not implemented'
  end

  # Change the game state (for pass-outs)
  def another_pass_out(game)
    raise 'Not implemented'
  end

  # If true, on failed game whists are divided equally
  # otherwise, only defending whister gets real whists (passer gets only "difference")
  def is_whist_gentlemanish
    raise 'Not implemented'
  end
end

