#!/usr/bin/env ruby

#
# $Id$
#
# Collecting the results of one hand
#

require 'model/pref_calc_model'
require 'pref_util'

class HandResultsDialog
  include GetText

  DIALOG_NAME = "hand_results"

  attr_reader :game, :results

  def clear_text_combo_box(combo)
    cnt = 0
    combo.model.each do |item| cnt += 1 end
    (1..cnt).each do |idx| combo.remove_text(0) end
  end

  #
  # Initialise the dialog
  #
  def initialize(game)
    @glade = PrefUtil::load_glade(DIALOG_NAME, self)
    @game = game

    # Populate combos
    clear_text_combo_box(self['cmb_bid_winner'])
    clear_text_combo_box(self['cmb_mizere_player'])
    @game.names.each do |name|
      if game.model.dealer_plays? || name != game.dealer.name
        self['cmb_bid_winner'].append_text(name)
        self['cmb_mizere_player'].append_text(name)
      end
    end
    self['lbl_dealer'].text = game.dealer.name
    
    # In the glade file, tabs are enabled - for editing purposes
    self['game_type_notebook'].show_tabs = false
    self['frm_extra_tricks_for_dealer'].visible = !@game.model.dealer_plays?

    reset_tricks(1..3)

    # Update labels for pass-out
    dnames = @game.get_pass_out_names(game.dealer.name)

    (0..2).each do |idx|
      self["lbl_num_tricks_by_player#{idx+1}"].text = dnames[idx]
      self["sp_num_tricks_by_player#{idx+1}"].sensitive = true 
    end

    allow_proceed(false)
  end

  #
  # Updated OK button and message label, depending on the validation result
  #
  # Parameters:
  #   validation_result - either Hash (GameModel::HandResult::VALIDATION_RESULT/VALIDATION_MESSAGE)
  #                       or boolean
  #
  def allow_proceed(validation_result)
    if (validation_result.is_a? Hash)
      self['btn_hand_ok'].sensitive = validation_result[GameModel::HandResult::VALIDATION_RESULT]
      vmsg = validation_result[GameModel::HandResult::VALIDATION_MESSAGE]
      self['lbl_message'].text = vmsg.nil? ? '' : vmsg
    else
      if (validation_result.is_a? TrueClass or 
          validation_result.is_a? FalseClass)
        self['btn_hand_ok'].sensitive = validation_result
        self['lbl_message'].text = ''
      end
    end
  end

  def widget
    return @glade[DIALOG_NAME]
  end

  def [](widget_name)
    @glade[widget_name]
  end

  def validate_positive_game
    if (self['cmb_bid_winner'].active_text.nil?)
      allow_proceed(false)
    else
      allow_proceed(@game.model.validate_positive_game(
        self['sp_declared_game'].text.to_i,
        self['sp_num_tricks_by_whister1'].text.to_i,
        self['sp_num_tricks_by_whister2'].text.to_i,
        self['chk_defending_whister1'].active?,
        self['chk_defending_whister2'].active?))
    end
  end

  def validate_mizere
    allow_proceed(!self['cmb_mizere_player'].active_text.nil?)
  end

  def validate_pass_out
    allow_proceed(@game.model.validate_pass_out(
      self['sp_mount_tricks_for_dealer'].text.to_i,
      self['sp_num_tricks_by_player1'].text.to_i,
      self['sp_num_tricks_by_player2'].text.to_i,
      self['sp_num_tricks_by_player3'].text.to_i))
  end
  #
  # Set the widgets in the tab()s to the initial state
  #
  # Parameters:
  #   tabs: the set of integers 1..3
  #
  def reset_tricks(tabs)
    if tabs.include?(1)
      populate_whisters
    end
    if tabs.include?(2)
      self['lbl_num_tricks_by_mizere_player'].text = ''
      self['sp_num_tricks_by_mizere_player'].sensitive = false
    end
    if tabs.include?(3)
      self['lbl_num_tricks_by_player1'].text = ''
      self['sp_num_tricks_by_player1'].sensitive = false
      self['lbl_num_tricks_by_player2'].text = ''
      self['sp_num_tricks_by_player2'].sensitive = false
      self['lbl_num_tricks_by_player3'].text = ''
      self['sp_num_tricks_by_player3'].sensitive = false
    end
  end

  #
  # More detailed setup of the range/value for
  # setting tricks on bid winner (for the dealer)
  #
  def configure_tricks_on_bid_winner_for_dealer
    if not @game.model.dealer_plays?
      case self['cmb_game_type'].active
        when GameModel::Type::POSITIVE
          # For the positive game, the limit of tricks is "undertaken" - "untaken + 3" (for star)
          min_tobwfd = self['sp_declared_game'].text.to_i - 
                       10 +
                       self['sp_num_tricks_by_whister1'].text.to_i +
                       self['sp_num_tricks_by_whister2'].text.to_i
          min_tobwfd = 0 if min_tobwfd < 0
          self['sp_tricks_on_bid_winner_for_dealer'].set_range(min_tobwfd, min_tobwfd + 3)
        when GameModel::Type::MIZERE
          # For mizere, the exact value is number of tricks taken by the player
          self['sp_tricks_on_bid_winner_for_dealer'].text=self['sp_num_tricks_by_mizere_player'].text
        when GameModel::Type::PASS_OUT
          # For pass-out, there cannot be tricks
          self['sp_tricks_on_bid_winner_for_dealer'].text='0'
      end
    end
  end

  #
  # Populate whister names based on the bid winner
  #
  def populate_whisters
    bid_winner = self['cmb_bid_winner'].active_text
    dealer = self['lbl_dealer'].text
    clear_text_combo_box(self['cmb_half_whist'])
    
    if bid_winner.nil?
      (0..1).each do |idx|
        self["lbl_whister#{idx+1}"].text = ''
        self["sp_num_tricks_by_whister#{idx+1}"].sensitive = false
      end
    else
      dnames = @game.get_whister_names(dealer, bid_winner)

      (0..1).each do |idx|
        self["lbl_whister#{idx+1}"].text = dnames[idx]
        self["sp_num_tricks_by_whister#{idx+1}"].sensitive = true
      end

      # Populate half-whisters
      self['cmb_half_whist'].append_text('')
      # First whister can never net half-whist!
      self['cmb_half_whist'].append_text(dnames[1])
      self['cmb_half_whist'].append_text(game.dealer.name) if !@game.model.dealer_plays?
    end
    self['cmb_half_whist'].active = 0
  end

  #
  # Enable/disable half-whist combo depending on the circumstances
  #
  def sense_half_whist_combo
     self['cmb_half_whist'].sensitive = (self['sp_declared_game'].text.to_i < 8) &&
       !self['chk_defending_whister1'].active? &&
       !self['chk_defending_whister2'].active? &&
       self['sp_num_tricks_by_whister1'].text == '0' &&
       self['sp_num_tricks_by_whister2'].text == '0'
  end

  #
  # Various widget event handlers ...
  #
  def on_cmb_game_type_changed(widget)
    self['game_type_notebook'].page = widget.active + 1
    self['sp_mount_tricks_for_dealer'].text='0'
    self['sp_mount_tricks_for_dealer'].sensitive=!game.model.dealer_plays?
    self['sp_tricks_on_bid_winner_for_dealer'].text='0'
    case self['cmb_game_type'].active
      when GameModel::Type::POSITIVE
        self['sp_mount_tricks_for_dealer'].set_range(0, 1)
        self['lbl_tricks_on_bid_winner_for_dealer'].sensitive=true
        self['sp_tricks_on_bid_winner_for_dealer'].sensitive=true
        validate_positive_game
      when GameModel::Type::MIZERE
        self['sp_mount_tricks_for_dealer'].set_range(0, 1)
        self['lbl_tricks_on_bid_winner_for_dealer'].sensitive=false
        self['sp_tricks_on_bid_winner_for_dealer'].sensitive=false
        validate_mizere
      when GameModel::Type::PASS_OUT
        self['sp_mount_tricks_for_dealer'].set_range(0, 2)
        self['lbl_tricks_on_bid_winner_for_dealer'].sensitive=false
        self['sp_tricks_on_bid_winner_for_dealer'].sensitive=false
        validate_pass_out
    end
    configure_tricks_on_bid_winner_for_dealer
  end

  def on_cmb_bid_winner_changed(widget)
    if widget.active < 0
      reset_tricks([1])
      return
    end

    populate_whisters
    validate_positive_game
  end

  def on_cmb_mizere_player_changed(widget)
    if widget.active < 0
      reset_tricks([2])
      return
    end
    self['lbl_num_tricks_by_mizere_player'].text = _("Tricks taken by %s") % widget.active_text
    self['sp_num_tricks_by_mizere_player'].sensitive = true
    validate_mizere
  end

  def on_hand_results_response(widget, response)
    if response == Gtk::Dialog::RESPONSE_OK.to_i
      case self['cmb_game_type'].active
        when -1
          # do nothing
          return
        when GameModel::Type::POSITIVE
          @results = GameModel::HandResult::Positive.new(@game,
                       @game.idx_by_name(self['lbl_dealer'].text),
                       self['sp_mount_tricks_for_dealer'].text.to_i,
                       self['sp_tricks_on_bid_winner_for_dealer'].text.to_i,
                       self['sp_declared_game'].text.to_i,
                       @game.idx_by_name(self['cmb_bid_winner'].active_text),
                       self['sp_num_tricks_by_whister1'].text.to_i,
                       self['sp_num_tricks_by_whister2'].text.to_i,
                       self['chk_defending_whister1'].active?,
                       self['chk_defending_whister2'].active?,
                       @game.idx_by_name(self['cmb_half_whist'].active_text))
        when GameModel::Type::MIZERE
          @results = GameModel::HandResult::Mizere.new(@game,
                       @game.idx_by_name(self['lbl_dealer'].text),
                       self['sp_mount_tricks_for_dealer'].text.to_i,
                       self['sp_tricks_on_bid_winner_for_dealer'].text.to_i,
                       @game.idx_by_name(self['cmb_mizere_player'].active_text),
                       self['sp_num_tricks_by_mizere_player'].text.to_i)
        when GameModel::Type::PASS_OUT
          @results = GameModel::HandResult::PassOut.new(@game,
                       @game.idx_by_name(self['lbl_dealer'].text),
                       self['sp_mount_tricks_for_dealer'].text.to_i,
                       self["sp_num_tricks_by_player1"].text.to_i,
                       self["sp_num_tricks_by_player2"].text.to_i,
                       self["sp_num_tricks_by_player3"].text.to_i)
      end
    end
    widget.destroy
  end
  
  def on_sp_mount_tricks_for_dealer_value_changed
    if self['cmb_game_type'].active == GameModel::Type::PASS_OUT
      validate_pass_out
    end
  end

  def on_sp_declared_game_value_changed
    sense_half_whist_combo
    validate_positive_game
    configure_tricks_on_bid_winner_for_dealer
  end

  def on_sp_num_tricks_by_whister1_value_changed
    sense_half_whist_combo
    validate_positive_game
  end

  def on_sp_num_tricks_by_whister2_value_changed
    sense_half_whist_combo
    validate_positive_game
  end

  def on_chk_defending_whister1_toggled
    sense_half_whist_combo
    validate_positive_game
  end

  def on_chk_defending_whister2_toggled
    sense_half_whist_combo
    validate_positive_game
  end

  def on_cmb_half_whist_changed
    if self['cmb_half_whist'].active_text != ''
      self['sp_num_tricks_by_whister1'].sensitive = false
      self['sp_num_tricks_by_whister2'].sensitive = false
      self['chk_defending_whister1'].sensitive = false
      self['chk_defending_whister2'].sensitive = false

      self['sp_num_tricks_by_whister1'].text = '0'
      self['sp_num_tricks_by_whister2'].text = '0'
      self['chk_defending_whister1'].active = false
      self['chk_defending_whister2'].active = false
    else
      self['sp_num_tricks_by_whister1'].sensitive = true
      self['sp_num_tricks_by_whister2'].sensitive = true
      self['chk_defending_whister1'].sensitive = true
      self['chk_defending_whister2'].sensitive = true
    end
    validate_positive_game
  end

  def on_sp_num_tricks_by_mizere_player_value_changed
    configure_tricks_on_bid_winner_for_dealer
  end

  def on_sp_num_tricks_by_player1_value_changed
    validate_pass_out
  end

  def on_sp_num_tricks_by_player2_value_changed
    validate_pass_out
  end

  def on_sp_num_tricks_by_player3_value_changed
    validate_pass_out
  end

end

# Main program
if __FILE__ == $0
  # Set values as your own application. 
  game = Game.new(["Ali", "Bill", "Ciaran", "Denis"])
  #game = Game.new(["Ali", "Bill", "Ciaran"])
  dlg = HandResultsDialog.new(game, "prefcalc")
  dlg.widget.show_all 
  Gtk.main
end
