// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <QDebug>
#include <QMessageBox>
#include <QMouseEvent>
#include <timeshop.hpp>
#include "plansplant/widgets.hpp"
#include "plansplant/tasks_changes.hpp"
#include "plansplant/utility.hpp"

namespace PlansPlant
{
  const QColor SpecItemsColors[] = { QColor( 176, 176, 176 ), QColor( 192, 192, 192 ), QColor( 208, 208, 208 ) };
  // TasksModel::Item
  TasksModel::Item::~Item()
  {
    while( !SubItems.empty() )
    {
      delete SubItems.back();
      SubItems.pop_back();
    }
  } // ~Item()
  QVariant TasksModel::Item::data( int /*Column*/, int Role ) const
  {
    QVariant Result;
#ifdef PLANSPLANT_HANDHELD
    if( Role == Qt::SizeHintRole && !MainWindow::desktop_view() )
      Result = QSize( 300, 42 );
#endif
    return Result;
  } // data( int, int ) const
  bool TasksModel::Item::move_subitem( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubItems.size() && To >= 0 && To < SubItems.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	  SubItems.swap( Index, Index+Inc );
	Result = true;
      }
    }
    return Result;
  } // move_subitem( int, int )
  void TasksModel::Item::task_added( Task& NewTask, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->task_added( NewTask, Model );
  } // task_added( Task&, TasksModel& )
  void TasksModel::Item::task_removed( Task::ID TaskID, TasksModel& Model )
  {
    for( int I = SubItems.size()-1; I >= 0; --I )
      if( Item* It = subitem( I ) )
      {
	if( It->task() && It->task()->id() == TaskID )
	  remove_subitem( I, Model );
	else
	  It->task_removed( TaskID, Model );
      }
  } // task_removed( Task::ID, Model )
  void TasksModel::Item::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    if( Field == Task::Change::SuperTask )
    {
      for( int I = 0; I < SubItems.size(); ++I )
	if( Item* It = subitem( I ) )
	{
	  if( It->relation() == Item::Subtask && It->task() == &ChangedTask /*&& ChangedTask.supertask() != task()*/ )
	    remove_subitem( I--, Model );
	  else
	    It->task_changed( ChangedTask, Field, Model );
	}
    }
    else
      foreach( Item* It, SubItems )
	if( It )
	  It->task_changed( ChangedTask, Field, Model );
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksModel::Item::task_moved( const Task& MovedTask, int OldIndex, int NewIndex, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->task_moved( MovedTask, OldIndex, NewIndex, Model );
  } // task_moved( const Task&, int, int, TasksModel& )
  void TasksModel::Item::blocker_added( const Task& Dependant, Task& NewBlocker, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->blocker_added( Dependant, NewBlocker, Model );
  } // blocker_added( const Task&, Task&, TasksModel& )
  void TasksModel::Item::blocker_removed( const Task& Dependant, const Task& OldBlocker, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->blocker_removed( Dependant, OldBlocker, Model );
  } // blocker_removed( const Task&, const Task&, TasksModel& )
  void TasksModel::Item::blocker_moved( const Task& Dependant, const Task& Blocker, int OldIndex, int NewIndex, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->blocker_moved( Dependant, Blocker, OldIndex, NewIndex, Model );
  } // blocker_moved( const Task&, const Task&, int, int, TasksModel& )
  void TasksModel::Item::blockers_replaced( const Task& Dependant, TasksModel& Model )
  {
    foreach( Item* It, SubItems )
      if( It )
	It->blockers_replaced( Dependant, Model );
  } // blockers_replaced( const Task&, TasksModel& )
  // SubItems manipulations.
  void TasksModel::Item::add_subitem( Item& NewSubItem, TasksModel& Model, int Place )
  {
    int Row = Place < 0 ? SubItems.size() : Place;
    Model.beginInsertRows( Model.index( *this ), Row, Row );
    if( Place < 0 )
      SubItems.push_back( &NewSubItem );
    else
      SubItems.insert( Place, &NewSubItem );
    Model.endInsertRows();
  } // add_subitem( Item& NewSubItem, TasksModel& Model )
  void TasksModel::Item::remove_subitem( int Index, TasksModel& Model )
  {
    Model.beginRemoveRows( Model.index( *this ), Index, Index );
    delete SubItems.takeAt( Index );
    Model.endRemoveRows();
  } // remove_subitem( int, TasksModel& )
  void TasksModel::Item::move_subitem( int From, int To, TasksModel& Model )
  {
    QModelIndex Index = Model.index( *this );
#if QT_VERSION >= 0x040600
    int FixedTo = To > From ? To+1 : To;
    if( Model.beginMoveRows( Index, From, From, Index, FixedTo ) )
    {
      SubItems.move( From, To );
      Model.endMoveRows();
    }
#else
    Model.beginRemoveRows( Index, From, From );
    Item* It = SubItems.takeAt( From );
    Model.endRemoveRows();
    Model.beginInsertRows( Index, To, To );
    SubItems.insert( To, It );
    Model.endInsertRows();
#endif
  } // move_subitem( int, int, TasksModel& )

  // TasksModel::Items::TaskItem
  QVariant TasksModel::Items::TaskItem::data( int Column, int Role ) const
  {
    QVariant Result = Item::data( Column, Role );
    switch( Role )
    {
    case::Qt::DisplayRole:
      switch( Column )
      {
      case NameCol: Result = Object.name(); break;
      case CompletedCol: Result = QString::number( Object.completed() * 100 ) + '%'; break;
      case EstimationCol:
	if( Object.estimation() > 0 )
	  Result = QString::number( Object.estimation() ) + ' ' + Task::units_short_name( Object.estimation_units() );
	break;
      case PlanStartCol:
	if( Object.plan_start().isValid() )
	  Result = Object.plan_start().date();
	break;
      case PlanFinishCol:
	if( Object.plan_finish().isValid() )
	  Result = Object.plan_finish().date();
	break;
      case ElapsedCol:
	{
	  int Secs = Object.secs( Object.active() ? QDateTime::currentDateTime() : QDateTime() );
	  if( Secs > 0 )
	    Result = Timeshop::format_time( Secs*1000 );
	}
	break;
      default: break;
      }
      break;
    case Qt::DecorationRole:
      if( Column == NameCol )
	Result = plansplant_icon( "task" );
      else if( Column == PriorityIconCol )
	switch( Object.priority() )
	{
	case 2: Result = plansplant_icon( "priority-highest" ); break;
	case 1: Result = plansplant_icon( "priority-high" ); break;
	case -1: Result = plansplant_icon( "priority-low" ); break;
	case -2: Result = plansplant_icon( "priority-lowest" ); break;
	}
      break;
    case Qt::CheckStateRole:
      if( Column == CompletedMarkCol )
      {
	if( Object.completed() >= 1 )
	  Result = Qt::Checked;
	else
	{ if( !Object.blocked() ) Result = Qt::Unchecked; }
      }
      break;
    case Qt::ForegroundRole:
      if( Column == ElapsedCol && Object.active() )
	Result = QColor( 0, 0, 160 );
      break;
#ifndef PLANSPLANT_HANDHELD
    case Qt::TextAlignmentRole:
      switch( Column )
      {
      case CompletedCol:
      case PlanStartCol:
      case PlanFinishCol:
      case EstimationCol:
      case ElapsedCol:
	Result = Qt::AlignRight;
	break;
      default: break;
      }
      break;
#endif // PLANSPLANT_HANDHELD
    default: break;
    }
    return Result;
  } // data( int, int ) const
  TasksModel::Item::Pointer TasksModel::Items::SubtaskItem::parent() const { return &Parent; }
  bool TasksModel::Items::TaskItem::is_leaf() const { return Object.subtasks().empty() && Object.blockers().empty(); }
  void TasksModel::Items::TaskItem::populate()
  {
    if( !populated() )
    {
      foreach( Task* T, Object.subtasks() )
	SubItems.push_back( new SubtaskItem( *this, *T ) );
      foreach( Task* T, Object.blockers() )
	SubItems.push_back( new BlockerItem( *this, *T ) );
    }
  } // populate()
  void TasksModel::Items::TaskItem::task_added( Task& NewTask, TasksModel& Model )
  {
    if( populated() )
    {
      Item::task_added( NewTask, Model );
      if( NewTask.supertask() == &Object )
	add_subitem( *new Items::SubtaskItem( *this, NewTask ), Model, Object.subtasks().size()-1 );
    }
  } // task_added( Task&, TasksModel& )
  void TasksModel::Items::TaskItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    Item::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::SuperTask && ChangedTask.supertask() == &Object && populated() )
      add_subitem( *new Items::SubtaskItem( *this, ChangedTask ), Model, Object.subtasks().size()-1 );
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksModel::Items::TaskItem::task_moved( const Task& MovedTask, int OldIndex, int NewIndex, TasksModel& Model )
  {
    Item::task_moved( MovedTask, OldIndex, NewIndex, Model );
    if( MovedTask.supertask() == &Object && populated() && SubItems[ OldIndex ]->task() == &MovedTask )
      move_subitem( OldIndex, NewIndex, Model );
  } // task_moved( const Task&, int, int, TasksModel& )
  void TasksModel::Items::TaskItem::blocker_added( const Task& Dependant, Task& NewBlocker, TasksModel& Model )
  {
    if( populated() )
    {
      Item::blocker_added( Dependant, NewBlocker, Model );
      if( &Dependant == &Object )
	add_subitem( *new BlockerItem( *this, NewBlocker ), Model );
    }
  } // blocker_added( const Task&, Task&, TasksModel& )
  void TasksModel::Items::TaskItem::blocker_removed( const Task& Dependant, const Task& OldBlocker, TasksModel& Model )
  {
    if( populated() )
    {
      if( &Dependant == &Object )
      {
	for( int I = 0; I < SubItems.size(); I++ )
	  if( Item* It = SubItems[ I ] )
	  {
	    if( It->relation() == Item::Blocker && It->task() == &OldBlocker  )
	      remove_subitem( I--, Model );
	    else
	      Item::blocker_removed( Dependant, OldBlocker, Model );
	  }
      }
      else
	Item::blocker_removed( Dependant, OldBlocker, Model );
    }
  } // blocker_removed( const Task&, const Task&, TasksModel& )
  void TasksModel::Items::TaskItem::blocker_moved( const Task& Dependant, const Task& Blocker, int OldIndex, int NewIndex, TasksModel& Model )
  {
    if( populated() )
    {
      Item::blocker_moved( Dependant, Blocker, OldIndex, NewIndex, Model );
      if( &Dependant == &Object && SubItems[ OldIndex ] && SubItems[ OldIndex ]->relation() == Item::Blocker && SubItems[ OldIndex ]->task() == &Blocker )
	move_subitem( OldIndex, NewIndex, Model );
    }
  } // blocker_moved( const Task&, const Task&, int, int, TasksModel& )
  void TasksModel::Items::TaskItem::blockers_replaced( const Task& Dependant, TasksModel& Model )
  {
    if( ( &Dependant == &Object ) )
    {
      if( populated() )
      {
	QModelIndex Index = Model.index( *this );
	Model.beginRemoveRows( Index, Object.subtasks().size(), SubItems.size() );
	while( SubItems.size() > Object.subtasks().size() )
	  delete SubItems.takeLast();
	Model.endRemoveRows();
	Item::blockers_replaced( Dependant, Model );
	Model.beginInsertRows( Index, SubItems.size(), SubItems.size()+Object.blockers().size()-1 );
	foreach( Task* Blocker, Object.blockers() )
	  SubItems.push_back( new Items::BlockerItem( *this, *Blocker ) );
	Model.endInsertRows();
      }
    }
    else
      Item::blockers_replaced( Dependant, Model );
  } // blockers_replaced( const Task&, TasksModel& )
  // TasksModel::Items::BlokerItem
  QVariant TasksModel::Items::BlockerItem::data( int Column, int Role ) const
  {
    QVariant Result = TaskItem::data( Column, Role );
    if( Role == Qt::DecorationRole && Column == NameCol )
      Result = plansplant_icon( "dependency" );
    return Result;
  } // data( int, int ) const
  // TasksModel::Items::SpecialItem
  QVariant TasksModel::Items::SpecialItem::data( int Column, int Role ) const
  {
    QVariant Result = Item::data( Column, Role );
    if( Column == 0 )
      switch( Role )
      {
      case Qt::DisplayRole:
	Result = Name;
	break;
      case Qt::ForegroundRole:
	Result = Color;
	break;
      case Qt::FontRole:
	{
	  QFont Font = QApplication::font();
	  Font.setBold( true );
	  Result = Font;
	}
	break;
      case Qt::BackgroundRole:
	Result = QBrush( SpecItemsColors[ 0 ] );
	break;
      default: break;
      }
    return Result;
  } // data( int, int ) const
  TasksModel::Items::TaskItem::Pointer TasksModel::Items::SpecialItem::task( int Index ) const
  { return ( Index < 0 || Index >= SubItems.size() ) ? 0 : TaskItem::downcast( SubItems[ Index ] ); } // task( int Index ) const
  void TasksModel::Items::SpecialItem::add( Task& Tsk ) { SubItems.push_back( new SpecialTaskItem( *this, Tsk ) ); } // add( Task& )
  TasksModel::Item::Pointer TasksModel::Items::SpecialTaskItem::parent() const { return &Parent; }
  // TasksModel
  TasksModel::TasksModel( TasksFile& File0, QObject* Parent ) : QAbstractItemModel( Parent ), Root( 0 ), File( File0 )
  {
    add_informer( File );
  } // TasksModel( TasksFile*, QObject* )
  TasksModel::~TasksModel() { if( Root ) delete Root; }
  TasksModel::Item* TasksModel::item_from_index( const QModelIndex& Index ) const
  { return Index.isValid() ? static_cast<Item*>( Index.internalPointer() ) : Root; } // item_from_index( const QModelIndex& ) const
  Task* TasksModel::task_from_index( const QModelIndex& Index ) const
  {
    Task* Result = 0;
    if( Index.isValid() )
      if( Item* It = item_from_index( Index ) )
	Result = It->task();
    return Result;
  } // task_from_index( const QModelIndex& ) 
  QModelIndex TasksModel::index( TasksModel::Item& ForItem, int Column ) const
  {
    QModelIndex Result;
    if( &ForItem != Root )
      if( Item* Parent = ForItem.parent() )
      {
	int Row = Parent->subitems().indexOf( &ForItem );
	if( Row >= 0 )
	  Result = createIndex( Row, Column, &ForItem );
      }
    return Result;
  } // index( TasksModel::Item& ) const
  bool TasksModel::is_subtask( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Item* It = item_from_index( Index ) )
      Result = ( It->relation() == Item::Subtask );
    return Result;
  } // is_subtask( const QModelIndex& ) const
  bool TasksModel::is_blocker( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( Item* It = item_from_index( Index ) )
      Result = ( It->relation() == Item::Blocker );
    return Result;
  } // is_blocker( const QModelIndex& ) const
  
  bool TasksModel::hasChildren( const QModelIndex& Parent ) const
  {
    bool Result = false;
    if( Item* It = item_from_index( Parent ) )
      Result = !It->is_leaf();
    return Result;
  } // hasChildren( const QModelIndex& ) const
  int TasksModel::rowCount( const QModelIndex& Parent ) const
  {
    int Result = 0;
    if( Item* It = item_from_index( Parent ) )
      Result = It->subitems().size();
    return Result;
  } // rowCount( const QModelIndex& ) const
  int TasksModel::columnCount( const QModelIndex& /*Parent*/ ) const { return TotalCols; }
  bool TasksModel::canFetchMore( const QModelIndex& Parent ) const
  {
    bool Result = false;
    if( Item* It = item_from_index( Parent ) )
      Result = !It->populated();
    return Result;
  } // canFetchMore( const QModelIndex& ) const
  void TasksModel::fetchMore( const QModelIndex& Parent )
  {
    if( Item* It = item_from_index( Parent ) )
      It->populate();
  } // fetchMore( const QModelIndex& )
  QModelIndex TasksModel::index( int Row, int Column, const QModelIndex& Parent ) const
  {
    QModelIndex Result;
    if( Item* It = item_from_index( Parent ) )
    {
      It->populate(); //!< \todo Think: may be it's better to return invalid index for unpopulated parents?
      if( Row >= 0 && Row < It->subitems().size() )
	Result = createIndex( Row, Column, It->subitem( Row ) );
    }
    return Result;
  } // index( int, int, const QModelIndex& ) const
  QModelIndex TasksModel::parent( const QModelIndex& Index ) const
  {
    QModelIndex Result;
    if( Item* It = item_from_index( Index ) )
      if( It != Root )
	if( Item* Parent = It->parent() )
	  Result = index( *Parent );
    return Result;
  } // parent( const QModelIndex& ) const
  QVariant TasksModel::data( const QModelIndex& Index, int Role ) const
  {
    QVariant Result;
    if( Item* It = item_from_index( Index ) )
      Result = It->data( Index.column(), Role );
    return Result;
  } // data( const QModelIndex&, int ) const
  QVariant TasksModel::headerData( int Section, Qt::Orientation Orient, int Role ) const
  {
    QVariant Result;
    if( Orient == Qt::Horizontal && Role == Qt::DisplayRole )
    {
      switch( Section )
      {
      case NameCol: Result = tr( "Name" ); break;
      case CompletedCol: Result = tr( "Completed" ); break;
      case PlanStartCol: Result = tr( "Start" ); break;
      case PlanFinishCol: Result = tr( "Finish" ); break;
      case EstimationCol: Result = tr( "Estimation" ); break;
      case ElapsedCol: Result = tr( "Elapsed" ); break;
      default: break;
      }
    }
    return( Result );
  } // headerData( int, Qt::Orientation, int )
  
  void TasksModel::resize_columns( QTreeView* Tree ) const
  {
    Tree->setAlternatingRowColors( true );
    Tree->setColumnWidth( TasksTreeModel::NameCol, 400 );
    if( MainWindow::desktop_view() )
    {
      Tree->setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
      Tree->setColumnWidth( TasksTreeModel::PriorityIconCol, 24 );
      Tree->setColumnWidth( TasksTreeModel::CompletedCol, 43 );
    }
    else
    {
      Tree->setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
      Tree->setColumnWidth( TasksTreeModel::PriorityIconCol, 32 );
      Tree->setColumnWidth( TasksTreeModel::CompletedCol, 70 );
      Tree->setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
      Tree->setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
    }
    if( QHeaderView* Header = Tree->header() )
    {
      // if( MainWindow::desktop_view() ) // We can move this on handheld too because now we have priority column between branch and complete mark.
      {
	if( Header->visualIndex( CompletedMarkCol ) > 0 )
	{
	  Header->moveSection( CompletedMarkCol, 0 );
	  Header->moveSection( PriorityIconCol, 1 );
	}
#if 0
	else if( Header->visualIndex( PriorityIconCol ) > 0 )
	  Header->moveSection( PriorityIconCol, 0 );
#endif
      }
    }
    else qDebug() << "No header in the tree";
  } // resize_columns( QTreeView* ) const
  void TasksModel::add_task( const QModelIndex& Parent, QWidget* Widget )
  {
    TasksTreeModel Tasks( File, this );
    TaskDialog Dlg( Tasks, 0, Widget );
    if( Task* Sel = task_from_index( Parent ) )
      Dlg.supertask( Sel );
    Dlg.exec();
  } // add_task()
  void TasksModel::open_task( const QModelIndex& Index, QWidget* Widget )
  {
    if( Task* Sel = task_from_index( Index ) )
    {
      TasksTreeModel Tasks( File, this );
      TaskDialog Dlg( Tasks, Sel, Widget );
      Dlg.exec();
    }
  } // open_task()
  void TasksModel::delete_task( const QModelIndex& Index, QWidget* Widget )
  {
    if( Task* Sel = task_from_index( Index ) )
      if( QMessageBox::question( Widget, tr( "PlansPlant" ), tr( "Are you shure that you want to delete task\n\"" ) + Sel->name() + tr( "\"?\nThere's NO WAY to restore it." ),
				 QMessageBox::Yes | QMessageBox::No ) == QMessageBox::Yes )
	file().delete_task( *Sel ); //! \todo else Error message, here and above
  } // delete_task()
  void TasksModel::add_blocker( const QModelIndex& Index, QWidget* Widget )
  {
    if( Task* Object = task_from_index( Index ) )
    {
      BlockerSelectDialog Dlg( *this, Widget, Object, Object->blockers().empty() ? Object : Object->blockers().back() );
      if( Dlg.exec() )
      {
	if( Dlg.current_task() )
	{
	  file().add_blocker( *Object, *Dlg.current_task() );
	  qDebug() << "Add blocker" << Dlg.current_task()->name() << "to" << Object->name();
	}
	else
	  qDebug() << "No task selected. Don't add anything.";
      }
    }
  } // add_blocker( const QModelIndex&, QWidget* )
  void TasksModel::edit_blockers( const QModelIndex& Index, QWidget* Widget )
  {
    if( Task* Object = task_from_index( Index ) )
    {
      TasksTreeModel Tasks( File, this );
      BlockersEditorDialog Dlg( Tasks, Object, Widget );
      Dlg.exec();
    }
  } // edit_blockers( const QModelIndex&, QWidget* )
  void TasksModel::remove_blocker( const QModelIndex& Index, QWidget* /*Widget*/ )
  {
    if( Item* Current = item_from_index( Index ) )
      if( Current->relation() == Item::Blocker )
	if( Task* Block = Current->task() )
	  if( Item* Parent = Current->parent() )
	    if( Task* Dep = Parent->task() )
	      file().remove_blocker( *Dep, *Block );
	    else qDebug() << "Blocker item\'s parent have no task!";
	  else qDebug() << "Can't remove blocker: it hasn't a parent!";
	else qDebug() << "Blocker item has no task.";
      else qDebug() << "Can't remove blocker: it's not a blocker.";
    else qDebug() << "Can't remove blocker: nothing selected.";
  } // remove_blocker()
  void TasksModel::move_item( const QModelIndex& Index, int Distance, QWidget* /*Widget*/ )
  {
    if( Item* It = item_from_index( Index ) )
      if( Task* Tsk = It->task() )
	switch( It->relation() )
	{
	case Item::Subtask:
	  file().move_task( *Tsk, Index.row(), Index.row()+Distance );
	  break;
	case Item::Blocker:
	  if( Item* Parent = It->parent() )
	    if( Task* ParentTask = Parent->task() )
	    {
	      int BlockI = Index.row()-ParentTask->subtasks().size();
	      file().move_blocker( *ParentTask, *Tsk, BlockI, BlockI+Distance );
	    }
	  // else error message
	  break;
	default:
	  break;
	}
  } // move_item()
  void TasksModel::item_clicked( const QModelIndex& Index, QWidget* Widget )
  {
    if( Index.column() == TasksModel::CompletedMarkCol )
      if( Task* Cur = task_from_index( Index ) )
	if( !Cur->blocked() && Cur->completed() < 1
	    && QMessageBox::question( Widget, tr( "Plans Plant" ),
				      tr( "Complete task\n\"" ) + Cur->name() + tr( "\"?" ),
				      QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes ) == QMessageBox::Yes )
	  file().change_task( *Cur, new Task::Changes::Completed( 1 ) );
  } // item_clicked( const QModelIndex& )

  void TasksModel::file_closed( TasksFile& ClosedFile )
  {
    qDebug() << "File is closed. Destroy later.";
    Watcher::file_closed( ClosedFile );
    if( &File == &ClosedFile )
      deleteLater();
  } // file_closed( TasksFile& )
  void TasksModel::task_added( Task& NewTask )
  {
    qDebug() << "Added task" << NewTask.name() << "id:" << NewTask.id().str()
	     << ( NewTask.supertask() ? ", subtask of \"" + NewTask.supertask()->name() + "\", id: " + NewTask.supertask()->id().str() : QString( "root task" ) );
    if( Root ) Root->task_added( NewTask, *this );
    Watcher::task_added( NewTask );
  } // task_added( Task& )
  void TasksModel::task_removed( Task::ID TaskID )
  {
    qDebug() << "Removed task" << TaskID.str();
    if( Root ) Root->task_removed( TaskID, *this );
    Watcher::task_removed( TaskID );
  } // task_removed( Task::ID )
  void TasksModel::task_changed( Task& Object, Task::Change::FieldID Field )
  {
    if( Root ) Root->task_changed( Object, Field, *this );
    Watcher::task_changed( Object, Field );
  } // task_changed( Task& Object, Task::Change::FieldID Field )
  void TasksModel::task_moved( const Task& Object, int OldIndex, int NewIndex )
  {
    qDebug() << "Task" << Object.name() << "moved from" << OldIndex << "to" << NewIndex;
    if( Root ) Root->task_moved( Object, OldIndex, NewIndex, *this );
    Watcher::task_moved( Object, OldIndex, NewIndex );
  } // task_moved( const Task&, int, int )
  void TasksModel::blocker_added( const Task& Object, Task& NewBlocker )
  {
    qDebug() << "Blocker" << NewBlocker.name() << "added to" << Object.name();
    if( Root ) Root->blocker_added( Object, NewBlocker, *this );
    Watcher::blocker_added( Object, NewBlocker );
  } // blocker_added( const Task&, const Task& )
  void TasksModel::blocker_removed( const Task& Object, const Task& OldBlocker )
  {
    qDebug() << "Blocker" << OldBlocker.name() << "removed from" << Object.name();
    if( Root ) Root->blocker_removed( Object, OldBlocker, *this );
    Watcher::blocker_removed( Object, OldBlocker );
  } // blocker_removed( const Task&, const Task& )
  void TasksModel::blocker_moved( const Task& Object, const Task& Blocker, int OldIndex, int NewIndex )
  {
    qDebug() << Object.name() << "'s blocker" << Blocker.name() << "moved from" << OldIndex << "to" << NewIndex;
    int OldBlockerIndex = OldIndex + Object.subtasks().size(); //! \todo Find why this is here and why we pass corrected indexes to superclass' method.
    int NewBlockerIndex = NewIndex + Object.subtasks().size();
    if( Root ) Root->blocker_moved( Object, Blocker, OldBlockerIndex, NewBlockerIndex, *this );
    Watcher::blocker_moved( Object, Blocker, OldBlockerIndex, NewBlockerIndex );
  } // blocker_moved( const Task&, const Task&, int, int )
  void TasksModel::blockers_replaced( const Task& Object )
  {
    qDebug() << "Blockers for" << Object.name() << "has been replaced.";
    if( Root ) Root->blockers_replaced( Object, *this );
  } // blockers_replaced( const Task& )

  bool TasksTreeModel::Items::TasksRootItem::is_leaf() const { return File.roots().empty(); }
  void TasksTreeModel::Items::TasksRootItem::populate()
  {
    if( !populated() )
      foreach( Task* T, File.roots() )
	if( T )
	  SubItems.push_back( new Items::RootTaskItem( *this, *T ) );
  } // populate()
  void TasksTreeModel::Items::TasksRootItem::task_added( Task& NewTask, TasksModel& Model )
  {
    if( populated() )
    {
      RootItem::task_added( NewTask, Model );
      if( !NewTask.supertask() )
	add_subitem( *new Items::RootTaskItem( *this, NewTask ), Model );
    }
  } // task_added( Task&, TasksModel& )
  void TasksTreeModel::Items::TasksRootItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    RootItem::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::SuperTask && !ChangedTask.supertask() && populated() )
      add_subitem( *new Items::RootTaskItem( *this, ChangedTask ), Model );
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTreeModel::Items::TasksRootItem::task_moved( const Task& MovedTask, int OldIndex, int NewIndex, TasksModel& Model )
  {
    RootItem::task_moved( MovedTask, OldIndex, NewIndex, Model );
    if( !MovedTask.supertask() && populated() && SubItems[ OldIndex ]->task() == &MovedTask )
      move_subitem( OldIndex, NewIndex, Model );
  } // task_moved( const Task&, int, int, TasksModel& )
  TasksModel::Item::Pointer TasksTreeModel::Items::RootTaskItem::parent() const { return &Parent; }

  TasksTreeModel::TasksTreeModel( TasksFile& File0, QObject* Parent ) : TasksModel( File0, Parent ) { Root = new Items::TasksRootItem( File ); }

  // TasksTimeListModel::Items
  void TasksTimeListModel::Items::NoTimeItem::task_added( Task& NewTask, TasksModel& Model )
  {
    Item::task_added( NewTask, Model );
    if( NewTask.completed() < 1 && NewTask.subtasks().empty() && !NewTask.plan_start().isValid() && !NewTask.plan_finish().isValid() )
    {
      int I = 0;
      Item* It = 0;
      Task* T = 0;
      while( I < SubItems.size() && ( It = SubItems[ I ] ) && ( T = It->task() ) && T->priority() >= NewTask.priority() ) ++I;
      add_subitem( *new SpecialTaskItem( *this, NewTask ), Model, I );
    }
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::NoTimeItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    SpecialItem::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::Completed || Field == Task::Change::PlanStart || Field == Task::Change::PlanFinish || Field == Task::Change::Priority )
    {
      int ActPlace = -1;
      int RightPlace = -1;
      int I = 0; //! \todo Use filter and compare to avoid similar methods in multiple Items.
      if( ChangedTask.completed() < 1 && ChangedTask.subtasks().empty() && !( ChangedTask.plan_start().isValid() || ChangedTask.plan_finish().isValid() ) )
      {
	for( ; I < SubItems.size() && RightPlace < 0; ++I )
	{
	  if( Item* It = SubItems[ I ] )
	    if( Task* T = It->task() )
	    {
	      if( ChangedTask.priority() > T->priority() ) RightPlace = I;
	      if( T == &ChangedTask )
	      {
		ActPlace = I;
		if( ChangedTask.priority() == T->priority() ) RightPlace = ActPlace;
	      }
	    }
	}
	if( RightPlace < 0 ) RightPlace = SubItems.size();
      }
      for( ; ActPlace < 0 && I < SubItems.size(); ++I )
	if( Item* It = SubItems[ I ] )
	  if( Task* T = It->task() )
	    if( T == &ChangedTask )
	      ActPlace = I;
      if( RightPlace < 0 )
      {
	if( ActPlace >= 0 )
	  remove_subitem( ActPlace, Model );
      }
      else if( ActPlace < 0 )
	add_subitem( *new Items::SpecialTaskItem( *this, ChangedTask ), Model, RightPlace );
      else
      {
	if( ActPlace < RightPlace ) --RightPlace;
	if( RightPlace != ActPlace ) move_subitem( ActPlace, RightPlace, Model );
      }
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTimeListModel::Items::NoTimeItem::add( Task& Tsk )
  {
    if( Tsk.completed() < 1 && Tsk.subtasks().empty() && !Tsk.plan_start().isValid() && !Tsk.plan_finish().isValid() )
    {
      int I = 0;
      Item* It = 0;
      Task* T = 0;
      while( I < SubItems.size() && ( It = SubItems[ I ] ) && ( T = It->task() ) && T != &Tsk && T->priority() >= Tsk.priority() ) ++I;
      if( T != &Tsk ) SubItems.insert( I, new SpecialTaskItem( *this, Tsk ) );
    }
  } // add( Task& )

  TasksTimeListModel::Items::OverdueItem::OverdueItem( Item& Parent0 ) : SpecialItem( Parent0, tr( "Overdue" ), QColor( 180, 0, 0 ) ), TimeStamp( QDateTime::currentDateTime() ) {}
  void TasksTimeListModel::Items::OverdueItem::add( Task& Tsk )
  {
    if( Tsk.completed() < 1 && Tsk.plan_finish().isValid() && Tsk.plan_finish() < TimeStamp )
    {
      Item::List::iterator It = SubItems.begin();
      Task* T = 0;
      while( It != SubItems.end() && ( T = (*It)->task() ) && T != &Tsk // Be careful: this is an assignment for optimization.
	     && ( T->plan_finish() < Tsk.plan_finish() || ( T->plan_finish() == Tsk.plan_finish() && T->priority() > Tsk.priority() ) ) )
	++It;
      if( T != &Tsk ) SubItems.insert( It, new SpecialTaskItem( *this, Tsk ) );
    }
  } // add( Task& )
  void TasksTimeListModel::Items::OverdueItem::task_added( Task& NewTask, TasksModel& Model )
  {
    Item::task_added( NewTask, Model );
    if( NewTask.completed() < 1 && NewTask.plan_finish().isValid() && NewTask.plan_finish() < TimeStamp )
    {
      int I = 0;
      Item* It = 0;
      Task* T = 0;
      while( I < SubItems.size() && ( It = SubItems[ I ] ) && ( T = It->task() )
	     && ( T->plan_finish() < NewTask.plan_finish() || ( T->plan_finish() == NewTask.plan_finish() && T->priority() > NewTask.priority() ) ) )
	++I;
      add_subitem( *new SpecialTaskItem( *this, NewTask ), Model, I );
    }
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::OverdueItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    SpecialItem::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::Completed || Field == Task::Change::PlanFinish || Field == Task::Change::Priority )
    {
      int ActPlace = -1;
      int RightPlace = -1;
      int I = 0; //! \todo Use filter and compare to avoid similar methods in multiple Items.
      if( ChangedTask.completed() < 1 && ChangedTask.plan_finish().isValid() && ChangedTask.plan_finish() < TimeStamp )
      {
	for( ; I < SubItems.size() && RightPlace < 0; ++I )
	{
	  if( Item* It = SubItems[ I ] )
	    if( Task* T = It->task() )
	    {
	      if( ChangedTask.plan_finish() < T->plan_finish() || ( ChangedTask.plan_finish() == T->plan_finish() && ChangedTask.priority() > T->priority() ) ) RightPlace = I;
	      if( T == &ChangedTask )
	      {
		ActPlace = I;
		if( ChangedTask.priority() == T->priority() ) RightPlace = ActPlace;
	      }
	    }
	}
	if( RightPlace < 0 ) RightPlace = SubItems.size();
      }
      for( ; ActPlace < 0 && I < SubItems.size(); ++I )
	if( Item* It = SubItems[ I ] )
	  if( Task* T = It->task() )
	    if( T == &ChangedTask )
	      ActPlace = I;
      if( RightPlace < 0 )
      {
	if( ActPlace >= 0 )
	  remove_subitem( ActPlace, Model );
      }
      else if( ActPlace < 0 )
	add_subitem( *new Items::SpecialTaskItem( *this, ChangedTask ), Model, RightPlace );
      else
      {
	if( ActPlace < RightPlace ) --RightPlace;
	if( RightPlace != ActPlace ) move_subitem( ActPlace, RightPlace, Model );
      }
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )

  QVariant TasksTimeListModel::Items::DayTaskItem::data( int Column, int Role ) const
  {
    QVariant Result = TaskItem::data( Column, Role );
    switch( Role )
    {
    case Qt::DisplayRole:
      switch( Column )
      {
      case PlanStartCol:
	if( Object.plan_start().isValid() )
	{
	  if( Object.plan_start().date() == day().date() )
	    Result = Object.plan_start().time();
	  else
	    Result = Object.plan_start().date();
	}
	break;
      case PlanFinishCol:
	if( Object.plan_finish().isValid() )
	{
	  if( Object.plan_finish().date() == day().date() )
	    Result = Object.plan_finish().time();
	  else
	    Result = Object.plan_finish().date();
	}
	break;
      default: break;
      }
      break;
    case Qt::ForegroundRole:
      switch( Column )
      {
      case PlanStartCol:
	if( Object.plan_start().isValid() && Object.plan_start().date() != day().date() )
	  Result = QApplication::palette().brush( QPalette::Disabled, QPalette::WindowText );
	break;
      case PlanFinishCol:
	if( Object.plan_finish().isValid() && Object.plan_finish().date() != day().date() )
	  Result = QApplication::palette().brush( QPalette::Disabled, QPalette::WindowText );
	break;
      default: break;
      }
      break;
    default: break;
    }
    return Result;
  } // data( int, int ) const
  TasksModel::Item* TasksTimeListModel::Items::DayTaskItem::parent() const { return &Day; }
  TasksTimeListModel::Items::DayItem& TasksTimeListModel::Items::DayTaskItem::day() const { return Day; }

  QVariant TasksTimeListModel::Items::DayItem::data( int Column, int Role ) const
  {
    QVariant Result = Item::data( Column, Role );
    if( Column == 0 )
      switch( Role )
      {
      case Qt::DisplayRole:
	Result = Day;
	break;
      case Qt::ForegroundRole:
	if( date().dayOfWeek() >= Qt::Saturday )
	  Result = QColor( Qt::blue );
	break;
      case Qt::BackgroundRole:
	Result = QBrush( SpecItemsColors[ 2 ] );
	break;
      default: break;
      }
    return Result;
  } // data( int, int ) const
  TasksTimeListModel::Item::Pointer TasksTimeListModel::Items::DayItem::parent() const { return &Month; }
  TasksTimeListModel::Items::DayTaskItem::Pointer TasksTimeListModel::Items::DayItem::task( int Index ) const
  { return ( Index < 0 || Index >= SubItems.size() ) ? 0 : DayTaskItem::downcast( SubItems[ Index ] ); } // task( int Index ) const
  QDate TasksTimeListModel::Items::DayItem::date() const { return QDate( Month.year().year(), Month.month(), Day ); }
  void TasksTimeListModel::Items::DayItem::task_added( Task& NewTask, TasksModel& Model )
  {
    Item::task_added( NewTask, Model );
    add( NewTask, &Model );
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::DayItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    Item::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::PlanStart || Field == Task::Change::PlanFinish || Field == Task::Change::Priority )
    {
      int ActPlace = -1;
      QDate Date = date();
      int RightPlace = -1;
      int I = 0;
      if( ChangedTask.plan_start().date() == Date || ChangedTask.plan_finish().date() == Date )
      {
	for( ; I < SubItems.size() && RightPlace < 0; ++I )
	{
	  if( Item* It = SubItems[ I ] )
	    if( Task* T = It->task() )
	    {
	      if( compare( ChangedTask, *T, Date ) < 0 ) RightPlace = I;
	      if( T == &ChangedTask )
	      {
		ActPlace = I;
		if( compare( ChangedTask, *T, Date ) == 0 ) RightPlace = ActPlace;
	      }
	    }
	}
	if( RightPlace < 0 ) RightPlace = SubItems.size();
      }
      for( ; ActPlace < 0 && I < SubItems.size(); ++I )
	if( Item* It = SubItems[ I ] )
	  if( Task* T = It->task() )
	    if( T == &ChangedTask )
	      ActPlace = I;
      if( RightPlace < 0 )
      {
	if( ActPlace >= 0 )
	  remove_subitem( ActPlace, Model );
      }
      else if( ActPlace < 0 )
	add_subitem( *new Items::DayTaskItem( *this, ChangedTask ), Model, RightPlace );
      else
      {
	if( ActPlace < RightPlace ) --RightPlace;
	if( RightPlace != ActPlace ) move_subitem( ActPlace, RightPlace, Model );
      }
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTimeListModel::Items::DayItem::add( Task& Tsk, TasksModel* Model )
  {
    QDate Date = date();
    if( Tsk.plan_start().date() == Date || Tsk.plan_finish().date() == Date )
    {
      int I = 0;
      Task* T = 0;
      while( I < SubItems.size() && ( T = task( I )->task() ) && T != &Tsk && compare( Tsk, *T, Date ) >= 0 ) I++;
      if( T != &Tsk )
      {
	if( Model ) add_subitem( *new DayTaskItem( *this, Tsk ), *Model, I );
	else	SubItems.insert( I, new DayTaskItem( *this, Tsk ) );
      }
    }
  } // add( Task&, TasksModel* )

  QVariant TasksTimeListModel::Items::MonthItem::data( int Column, int Role ) const
  {
    QVariant Result = Item::data( Column, Role );
    switch( Role )
    {
    case Qt::DisplayRole:
      if( Column == 0 )
	Result = QDate::longMonthName( Month, QDate::StandaloneFormat );
      break;
    case Qt::FontRole:
      {
	QFont Font = QApplication::font();
	Font.setItalic( true );
	Result = Font;
      }
      break;
    case Qt::BackgroundRole:
      Result = QBrush( SpecItemsColors[ 1 ] );
      break;
    default: break;
    }
    return Result;
  } // data( int, int ) const
  TasksTimeListModel::Item::Pointer TasksTimeListModel::Items::MonthItem::parent() const { return &Year; }
  TasksTimeListModel::Items::DayItem::Pointer TasksTimeListModel::Items::MonthItem::day( int Index ) const
  { return ( Index < 0 || Index >= SubItems.size() ) ? 0 : DayItem::downcast( SubItems[ Index ] ); } // day( int Index ) const
  void TasksTimeListModel::Items::MonthItem::task_added( Task& NewTask, TasksModel& Model )
  {
    Item::task_added( NewTask, Model );
    add( NewTask, &Model );
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::MonthItem::task_removed( Task::ID TaskID, TasksModel& Model )
  {
    Item::task_removed( TaskID, Model );
    for( int I = subitems().size()-1; I >= 0; --I )
      if( Item* It = subitem( I ) )
	if( It->is_leaf() )
	  remove_subitem( I, Model );
  } // task_removed( Task::ID, Model )
  void TasksTimeListModel::Items::MonthItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    Item::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::PlanStart || Field == Task::Change::PlanFinish )
    {
      int Day = -1;
      QDate NewDate = Field == Task::Change::PlanStart ? ChangedTask.plan_start().date() : ChangedTask.plan_finish().date();
      if( NewDate.isValid() && NewDate.year() == year().year() && NewDate.month() == Month )
	Day = NewDate.day();
      for( int I = 0; I < SubItems.size(); I++ )
	if( DayItem* It = day( I ) )
	{
	  if( Day > 0 )
	  {
	    if( It->day() > Day )
	    {
	      add_day_item( ChangedTask, Day, I, Model );
	      Day = -1;
	    }
	    else if( It->day() == Day )
	      Day = -1;
	  }
	  if( It->is_leaf() )
	    remove_subitem( I--, Model );
	}
      if( Day > 0 )
	add_day_item( ChangedTask, Day, SubItems.size(), Model );
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTimeListModel::Items::MonthItem::add_day_item( Task& DayTask, int Day, int Place, TasksModel& Model )
  {
    DayItem* NewDay = new Items::DayItem( *this, Day );
    add_subitem( *NewDay, Model, Place );
    NewDay->add( DayTask, &Model );
  } // add_day_item( Task&, int, int, TasksModel& )
  void TasksTimeListModel::Items::MonthItem::add( Task& Tsk, TasksModel* Model )
  {
    if( Tsk.plan_start().isValid() || Tsk.plan_finish().isValid() )
    {
      int Ds[ 2 ];
      int NumDays = 0;
      if( Tsk.plan_start().isValid() && Tsk.plan_start().date().year() == year().year() && Tsk.plan_start().date().month() == Month )
	Ds[ NumDays++ ] = Tsk.plan_start().date().day();
      if( Tsk.plan_finish().isValid() && Tsk.plan_finish().date().year() == year().year() && Tsk.plan_finish().date().month() == Month )
	Ds[ NumDays++ ] = Tsk.plan_finish().date().day();
      if( NumDays > 1 && Ds[ 1 ] == Ds[ 0 ] ) NumDays--;
      int D = 0;
      for( int I = 0; D < NumDays && I < SubItems.size(); I++ )
      {
	if( DayItem* Day = day( I ) )
	{
	  if( Day->day() > Ds[ D ] )
	  {
	    Day = new Items::DayItem( *this, Ds[ D ] );
	    if( Model ) add_subitem( *Day, *Model, I );
	    else	SubItems.insert( I, Day );
	  }
	  if( Day->day() == Ds[ D ] )
	  {
	    Day->add( Tsk, Model );
	    D++;
	  }
	}
      }
      while( D < NumDays )
      {
	Items::DayItem* NewDay = new Items::DayItem( *this, Ds[ D ] );
	if( Model ) add_subitem( *NewDay, *Model );
	else	    SubItems.push_back( NewDay );
	NewDay->add( Tsk, Model );
	D++;
      }
    }
  } // add( Task&, TasksModel* )

  QVariant TasksTimeListModel::Items::YearItem::data( int Column, int Role ) const
  {
    QVariant Result = Item::data( Column, Role );
    switch( Role )
    {
    case Qt::DisplayRole:
      if( Column == 0 )
	Result = Year;
      break;
    case Qt::FontRole:
      {
	QFont Font = QApplication::font();
	Font.setBold( true );
	Result = Font;
      }
      break;
    case Qt::BackgroundRole:
      Result = QBrush( SpecItemsColors[ 0 ] );
      break;
    default: break;
    }
    return Result;
  } // data( int, int ) const
  TasksTimeListModel::Items::MonthItem::Pointer TasksTimeListModel::Items::YearItem::month( int Index ) const
  { return ( Index < 0 || Index >= SubItems.size() ) ? 0 : MonthItem::downcast( SubItems[ Index ] ); } // month( int Index ) const
  TasksModel::Item::Pointer TasksTimeListModel::Items::YearItem::parent() const { return &Parent; }
  void TasksTimeListModel::Items::YearItem::task_added( Task& NewTask, TasksModel& Model )
  {
    Item::task_added( NewTask, Model );
    add( NewTask, &Model );
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::YearItem::task_removed( Task::ID TaskID, TasksModel& Model )
  {
    Item::task_removed( TaskID, Model );
    for( int I = subitems().size()-1; I >= 0; --I )
      if( Item* It = subitem( I ) )
	if( It->is_leaf() )
	  remove_subitem( I, Model );
  } // task_removed( Task::ID, Model )
  void TasksTimeListModel::Items::YearItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    Item::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::PlanStart || Field == Task::Change::PlanFinish )
    {
      int Month = -1;
      QDate NewDate = Field == Task::Change::PlanStart ? ChangedTask.plan_start().date() : ChangedTask.plan_finish().date();
      if( NewDate.isValid() && NewDate.year() == Year )
	Month = NewDate.month();
      for( int I = 0; I < SubItems.size(); I++ )
	if( MonthItem* It = month( I ) )
	{
	  if( Month > 0 )
	  {
	    if( It->month() > Month )
	    {
	      add_month_item( ChangedTask, Month, I, Model );
	      Month = -1;
	    }
	    else if( It->month() == Month )
	      Month = -1;
	  }
	  if( It->is_leaf() )
	    remove_subitem( I--, Model );
	}
      if( Month > 0 )
	add_month_item( ChangedTask, Month, SubItems.size(), Model );
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTimeListModel::Items::YearItem::add_month_item( Task& MonthTask, int Month, int Place, TasksModel& Model )
  {
    MonthItem* NewMonth = new Items::MonthItem( *this, Month );
    add_subitem( *NewMonth, Model, Place );
    NewMonth->add( MonthTask, &Model );
  } // add_month_item( Task&, int, int, TasksModel& )
  void TasksTimeListModel::Items::YearItem::add( Task& Tsk, TasksModel* Model )
  {
    if( Tsk.plan_start().isValid() || Tsk.plan_finish().isValid() )
    {
      int Mons[ 2 ];
      int NumMonths = 0;
      if( Tsk.plan_start().isValid() && Tsk.plan_start().date().year() == Year ) Mons[ NumMonths++ ] = Tsk.plan_start().date().month();
      if( Tsk.plan_finish().isValid() && Tsk.plan_finish().date().year() == Year ) Mons[ NumMonths++ ] = Tsk.plan_finish().date().month();
      if( NumMonths > 1 && Mons[ 1 ] == Mons[ 0 ] ) NumMonths--;
      int M = 0;
      for( int I = 0; M < NumMonths && I < SubItems.size(); I++ )
      {
	if( MonthItem* Month = month( I ) )
	{
	  if( Month->month() > Mons[ M ] )
	  {
	    Month = new Items::MonthItem( *this, Mons[ M ] );
	    if( Model ) add_subitem( *Month, *Model, I );
	    else	SubItems.insert( I, Month );
	  }
	  if( Month->month() == Mons[ M ] )
	  {
	    Month->add( Tsk, Model );
	    M++;
	  }
	}
      }
      while( M < NumMonths )
      {
	Items::MonthItem* NewMonth = new Items::MonthItem( *this, Mons[ M ] );
	if( Model ) add_subitem( *NewMonth, *Model );
	else 	    SubItems.push_back( NewMonth );
	NewMonth->add( Tsk, Model );
	M++;
      }
    }
  } // add( Task&, TasksModel* )

  bool TasksTimeListModel::Items::TimesRootItem::is_leaf() const { return false; }
  void TasksTimeListModel::Items::TimesRootItem::populate()
  {
    if( !populated() )
    {
      Specials.push_back( new Items::NoTimeItem( *this ) );
      Specials.push_back( new Items::OverdueItem( *this ) );
      foreach( Task* T, File.roots() )
	add_branch( *T );
      foreach( Items::SpecialItem::Pointer It, Specials )
	SubItems.push_back( It );
    }
  } // populate()
  void TasksTimeListModel::Items::TimesRootItem::task_added( Task& NewTask, TasksModel& Model )
  {
    RootItem::task_added( NewTask, Model );
    add( NewTask, &Model );
  } // task_added( Task&, TasksModel& )
  void TasksTimeListModel::Items::TimesRootItem::task_removed( Task::ID TaskID, TasksModel& Model )
  {
    RootItem::task_removed( TaskID, Model );
    for( int I = Years.size()-1; I >= 0; --I )
      if( Item* It = Years[ I ] )
	if( It->is_leaf() )
	{
	  Years.removeAt( I );
	  remove_subitem( I, Model );
	}
  } // task_removed( Task::ID, Model )
  void TasksTimeListModel::Items::TimesRootItem::task_changed( Task& ChangedTask, Task::Change::FieldID Field, TasksModel& Model )
  {
    RootItem::task_changed( ChangedTask, Field, Model );
    if( Field == Task::Change::PlanStart || Field == Task::Change::PlanFinish )
    {
      int Year = -1;
      QDate NewDate = Field == Task::Change::PlanStart ? ChangedTask.plan_start().date() : ChangedTask.plan_finish().date();
      if( NewDate.isValid() )
	Year = NewDate.year();
      for( int I = 0; I < Years.size(); I++ )
	if( YearItem* It = Years[ I ] )
	{
	  if( Year > 0 )
	  {
	    if( It->year() > Year )
	    {
	      add_year_item( ChangedTask, Year, I, Model );
	      Year = -1;
	    }
	    else if( It->year() == Year )
	      Year = -1;
	  }
	  if( It->is_leaf() )
	  {
	    Years.removeAt( I );
	    remove_subitem( I, Model );
	    --I;
	  }
	}
      if( Year > 0 )
	add_year_item( ChangedTask, Year, Years.size(), Model );
    }
  } // task_changed( Task&, Task::Change::FieldID, TasksModel& )
  void TasksTimeListModel::Items::TimesRootItem::add_year_item( Task& YearTask, int Year, int Place, TasksModel& Model )
  {
    YearItem* NewYear = new Items::YearItem( *this, Year );
    Years.insert( Place, NewYear );
    add_subitem( *NewYear, Model, Place );
    NewYear->add( YearTask, &Model );
  } // add_year_item( Task&, int, int, TasksModel& )
  void TasksTimeListModel::Items::TimesRootItem::add( Task& Tsk, TasksModel* Model )
  {
    if( Tsk.plan_start().isValid() || Tsk.plan_finish().isValid() )
    {
      int TaskYears[ 2 ];
      int NumYears = 0;
      if( Tsk.plan_start().isValid() ) TaskYears[ NumYears++ ] = Tsk.plan_start().date().year();
      if( Tsk.plan_finish().isValid() && ( !Tsk.plan_start().isValid() || Tsk.plan_finish().date().year() != Tsk.plan_start().date().year() ) )
	TaskYears[ NumYears++ ] = Tsk.plan_finish().date().year();
      if( NumYears > 1 && TaskYears[ 0 ] > TaskYears[ 1 ] )
      {
	int Tmp = TaskYears[ 0 ];
	TaskYears[ 0 ] = TaskYears[ 1 ];
	TaskYears[ 1 ] = Tmp;
      }
      int Y = 0;
      for( int I = 0; Y < NumYears && I < Years.size(); I++ )
      {
	if( Years[ I ]->year() > TaskYears[ Y ] )
	{
	  Years.insert( I, new Items::YearItem( *this, TaskYears[ Y ] ) );
	  if( Model ) add_subitem( *Years[ I ], *Model, I );
	  else	      SubItems.insert( I, Years[ I ] );
	}
	if( Years[ I ]->year() == TaskYears[ Y ] )
	{
	  Years[ I ]->add( Tsk, Model );
	  Y++;
	}
      }
      while( Y < NumYears )
      {
	Items::YearItem* NewYear = new Items::YearItem( *this, TaskYears[ Y ] );
	Years.push_back( NewYear );
	if( Model ) add_subitem( *NewYear, *Model, Years.size()-1 );
	else	    SubItems.insert( Years.size()-1, NewYear );
	NewYear->add( Tsk, Model );
	Y++;
      }
    }
    foreach( Items::SpecialItem* It, Specials )
      It->add( Tsk );
  } // add( Task& )
  void TasksTimeListModel::Items::TimesRootItem::add_branch( Task& Root )
  {
    add( Root );
    foreach( Task* T, Root.subtasks() )
      add_branch( *T );
  } // add_branch( Task& )

  class TreeIterator
  {
  public:
    TreeIterator( TasksModel& Model0 ) : Model( Model0 ), Current( Model0.index( 0, 0 ) ) {}
    TreeIterator( TasksModel& Model0, const QModelIndex& Start ) : Model( Model0 ), Current( Start ) {}
    TreeIterator& operator++();
    TreeIterator& operator--();
    const QModelIndex& index() const { return Current; }
    const QModelIndex& operator*() const { return Current; }
    operator bool() const { return Current.isValid(); }
  protected:
    TasksModel& Model;
    QModelIndex Current;
  }; // TreeIterator
  TreeIterator& TreeIterator::operator++()
  {
    QModelIndex Next = Current.child( 0, 0 ); // Search the model in-depth.
    if( !Next.isValid() )
    {
      Next = Current.sibling( Current.row()+1, 0 );
      while( !Next.isValid() && Current.isValid() ) // Try to find next sibling on any level.
      {
	Current = Current.parent();
	if( Current.isValid() )
	  Next = Current.sibling( Current.row()+1, 0 );
      }
    }
    Current = Next;
    return *this;
  } // operator++()
  TreeIterator& TreeIterator::operator--()
  {
    QModelIndex Prev;
    if( Current.row() > 0 ) Prev = Current.sibling( Current.row()-1, 0 ); // Try to step backward.
    if( !Prev.isValid() ) Prev = Current.parent(); // Try to step up.
    Current = Prev;
    return *this;
  } // operator--()
  TasksTreeWidget::TasksTreeWidget( QWidget* Parent, TasksModel* Model0 ) : QTreeView( Parent )
  {
    // setItemDelegate( new HeaderItemDelegate( this ) );
    //    setIndentation( 0 );
    setUniformRowHeights( true );
#ifdef PLANSPLANT_HANDHELD
    if( MainWindow::desktop_view() )
    {
#ifdef Q_WS_MAEMO_5
      disable_kinetic_scroller( this );
#endif // Q_WS_MAEMO_5
      setIconSize( QSize( 14, 14 ) );
    }
#else // !PLANSPLANT_HANDHELD
    setContextMenuPolicy( Qt::ActionsContextMenu );
#endif // PLANSPLANT_HANDHELD
    TaskAddAction = new QAction( plansplant_icon( "task-new" ), tr( "&Add task..." ), this );
    TaskAddAction->setShortcut( Qt::CTRL | Qt::Key_T );
    connect( TaskAddAction, SIGNAL( triggered() ), SLOT( add_task() ) );
    addAction( TaskAddAction );
    TaskEditAction = new QAction( plansplant_icon( "task-edit" ), tr( "&Open task..." ), this );
    TaskEditAction->setShortcut( Qt::CTRL | Qt::Key_E );
    connect( TaskEditAction, SIGNAL( triggered() ), SLOT( open_task() ) );
    addAction( TaskEditAction );
    TaskDeleteAction = new QAction( tr( "&Delete task" ), this );
    connect( TaskDeleteAction, SIGNAL( triggered() ), SLOT( delete_task() ) );
    addAction( TaskDeleteAction );
    QAction* Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    DependencyAddAction = new QAction( plansplant_icon( "dependency-new" ), tr( "Add de&pendency..." ), this );
    DependencyAddAction->setShortcut( Qt::CTRL | Qt::Key_D );
    connect( DependencyAddAction, SIGNAL( triggered() ), SLOT( add_blocker() ) );
    addAction( DependencyAddAction );
    DependenciesEditAction = new QAction( plansplant_icon( "dependency-edit" ), tr( "&Edit dependencies..." ), this );
    connect( DependenciesEditAction, SIGNAL( triggered() ), SLOT( edit_blockers() ) );
    addAction( DependenciesEditAction );
    DependencyRemoveAction = new QAction( tr( "Remo&ve dependency" ), this );
    connect( DependencyRemoveAction, SIGNAL( triggered() ), SLOT( remove_blocker() ) );
    addAction( DependencyRemoveAction );
    Act = new QAction( this );
    Act->setSeparator( true );
    addAction( Act );
    MoveUpAction = new QAction( load_icon( "go-up" ), tr( "Move &up" ), this );
    MoveUpAction->setShortcut( Qt::CTRL | Qt::Key_Up );
    connect( MoveUpAction, SIGNAL( triggered() ), SLOT( move_up() ) );
    addAction( MoveUpAction );
    MoveDownAction = new QAction( load_icon( "go-down" ), tr( "Move do&wn" ), this );
    MoveDownAction->setShortcut( Qt::CTRL | Qt::Key_Down );
    connect( MoveDownAction, SIGNAL( triggered() ), SLOT( move_down() ) );
    addAction( MoveDownAction );
    connect( this, SIGNAL( clicked( const QModelIndex& ) ), SLOT( item_clicked( const QModelIndex& ) ) );
    if( Model0 ) tasks( Model0 );
    else selection_changed( QModelIndex() );
  } // TasksTreeWidget( TasksModel&, QWidget* )
  QSize TasksTreeWidget::sizeHint() const { return QSize( 800, 0 ); }
  void TasksTreeWidget::tasks( TasksModel* Tasks )
  {
    setModel( Tasks );
#ifndef PLANSPLANT_NOT_RESIZE_TREE_IN_MODEL
    Tasks->resize_columns( this );
#else // PLANSPLANT_NOT_RESIZE_TREE_IN_MODEL
    setColumnWidth( TasksTreeModel::NameCol, 400 );
    if( MainWindow::desktop_view() )
    {
      setColumnWidth( TasksTreeModel::CompletedMarkCol, 24 );
      setColumnWidth( TasksTreeModel::PriorityIconCol, 24 );
      setColumnWidth( TasksTreeModel::CompletedCol, 43 );
      setAlternatingRowColors( true );
    }
    else
    {
      setColumnWidth( TasksTreeModel::CompletedMarkCol, 55 );
      setColumnWidth( TasksTreeModel::PriorityIconCol, 55 );
      setColumnWidth( TasksTreeModel::CompletedCol, 70 );
      setColumnWidth( TasksTreeModel::PlanStartCol, 128 );
      setColumnWidth( TasksTreeModel::PlanFinishCol, 128 );
    }
    if( QHeaderView* Header = header() )
    {
      if( Header->visualIndex( TasksTreeModel::CompletedMarkCol ) > 0 )
      {
	Header->moveSection( TasksTreeModel::CompletedMarkCol, 0 );
	Header->moveSection( TasksTreeModel::PriorityIconCol, 1 );
      }
      else if( Header->visualIndex( TasksTreeModel::PriorityIconCol ) > 0 )
	Header->moveSection( TasksTreeModel::PriorityIconCol, 0 );
    }
    else qDebug() << "No header in the tree";
#endif // PLANSPLANT_NOT_RESIZE_TREE_IN_MODEL
    connect( selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ), SLOT( selection_changed( const QItemSelection& ) ) );
    selection_changed( QModelIndex() );
  } // tasks( TasksTreeModel* )
  QModelIndex TasksTreeWidget::selected_index() const
  {
    QModelIndex Result;
    QModelIndexList Selected = selectedIndexes();
    if( !Selected.isEmpty() )
      Result = Selected.front();
    return Result;
  } // selected_index() const
  bool TasksTreeWidget::select_task( const Task* NewCurrent )
  {
    bool Result = false;
    QModelIndex Index;
    if( NewCurrent )
    {
      if( TasksModel* Tasks = tasks() )
	for( TreeIterator It( *Tasks ); It && !Result; ++It )
	  if( Tasks->is_subtask( *It ) && Tasks->task_from_index( *It ) == NewCurrent )
	  {
	    Index = *It;
	    Result = true;
	  }
    }
    else
      Result = true;
    setCurrentIndex( Index );
    return Result;
  } // select_task( const Task* )
  void TasksTreeWidget::add_task()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->add_task( selected_index(), this );
  } // add_task()
  void TasksTreeWidget::open_task()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->open_task( selected_index(), this );
  } // open_task()
  void TasksTreeWidget::delete_task()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->delete_task( selected_index(), this );
  } // delete_task()
  void TasksTreeWidget::add_blocker()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->add_blocker( selected_index(), this );
  } // add_blocker()
  void TasksTreeWidget::edit_blockers()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->edit_blockers( selected_index(), this );
  } // add_blocker()
  void TasksTreeWidget::remove_blocker()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->remove_blocker( selected_index(), this );
  } // remove_blocker()
  void TasksTreeWidget::move_up()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->move_item( selected_index(), -1, this );
    selection_changed( selected_index() );
  } // move_up()
  void TasksTreeWidget::move_down()
  {
    if( TasksModel* Tasks = tasks() )
      Tasks->move_item( selected_index(), 1, this );
    selection_changed( selected_index() );
  } // move_down()
  void TasksTreeWidget::item_clicked( const QModelIndex& Index )
  {
    if( Index.column() == TasksTreeModel::CompletedMarkCol )
      if( TasksModel* Tasks = tasks() )
	if( Task* Cur = Tasks->task_from_index( Index ) )
	  if( !Cur->blocked() && Cur->completed() < 1
	      && QMessageBox::question( this, tr( "Plans Plant" ),
					tr( "Complete task\n\"" ) + Cur->name() + tr( "\"?" ),
					QMessageBox::Yes | QMessageBox::No, QMessageBox::Yes ) == QMessageBox::Yes )
	    Tasks->file().change_task( *Cur, new Task::Changes::Completed( 1 ) );
  } // item_clicked( const QModelIndex& )
  void TasksTreeWidget::selection_changed( const QItemSelection& New )
  {
    QModelIndex NewIndex;
    if( !New.indexes().isEmpty() )
      NewIndex = New.indexes().front();
    selection_changed( NewIndex );
  } // selection_changed( const QItemSelection& )
  void TasksTreeWidget::selection_changed( const QModelIndex& NewIndex )
  {
    bool TasksActions = false;
    bool CanMoveUp = false;
    bool CanMoveDown = false;
    if( TasksModel* Tasks = tasks() )
    {
      TaskAddAction->setEnabled( true );
      if( Task* CurTask = Tasks->task_from_index( NewIndex ) )
      {
	qDebug() << "Selected \"" << CurTask->name() << "\".";
	TasksActions = true;
	QModelIndex Parent = Tasks->parent( NewIndex );
	if( Tasks->is_blocker( NewIndex ) )
	{
	  DependencyRemoveAction->setEnabled( true );
	  if( Task* Dependent = Tasks->task_from_index( Parent ) )
	  {
	    CanMoveUp = CurTask != Dependent->blockers().front();
	    CanMoveDown = CurTask != Dependent->blockers().back();
	  }
	}
	else if( Tasks->is_subtask( NewIndex ) )
	{
	  DependencyRemoveAction->setEnabled( false );
	  if( Task* Supertask = Tasks->task_from_index( Parent ) )
	  {
	    CanMoveUp = Supertask->subtasks().front() != CurTask;
	    CanMoveDown = Supertask->subtasks().back() != CurTask;
	  }
	  else
	  { //! \todo Here and above: make it filter-friendly (second task can become first if real first is filtered out).
	    CanMoveUp = Tasks->file().roots().front() != CurTask;
	    CanMoveDown = Tasks->file().roots().back() != CurTask;
	  }
	}
      }
      else qDebug() << "No task selected.";
    }
    else // No model - maybe issue an error message
      TaskAddAction->setEnabled( true );
    TaskEditAction->setEnabled( TasksActions );
    TaskDeleteAction->setEnabled( TasksActions );
    DependencyAddAction->setEnabled( TasksActions );
    if( !TasksActions ) DependencyRemoveAction->setEnabled( false ); // Don't always enable if we have tasks actions.
    MoveUpAction->setEnabled( CanMoveUp );
    MoveDownAction->setEnabled( CanMoveDown );
  } // selection_changed( const QModelIndex& )

  TasksTimeListModel::TasksTimeListModel( TasksFile& File0, QObject* Parent ) : TasksModel( File0, Parent )
  {
    Root = new Items::TimesRootItem( File );
  } // TasksTimeListModel( TasksFile&, QObject* )
  void TasksTimeListModel::min_max_for_date( const Task& T, const QDate& Date, QDateTime& Min, QDateTime& Max )
  {
    if( T.plan_start().date() == Date )
    {
      Min = T.plan_start();
      if( T.plan_finish().date() == Date )
	Max = T.plan_finish();
      else
	Max = Min;
    }
    else if( T.plan_finish().date() == Date )
    {
      Min = T.plan_finish();
      Max = Min;
    }
  } // min_max_for_date( const Task&, const QDate&, QDateTime&, QDateTime& )
  int TasksTimeListModel::compare( const Task& Task1, const Task& Task2, const QDate& Date )
  { //! \todo Work out better comparision criteria.
    int Result = 0;
    if( &Task1 != &Task2 )
    {
      QDateTime Min1;
      QDateTime Max1;
      min_max_for_date( Task1, Date, Min1, Max1 );
      QDateTime Min2;
      QDateTime Max2;
      min_max_for_date( Task2, Date, Min2, Max2 );
      if( Min1 > Max2 )
	Result = 1;
      else if( Min2 > Max1 )
	Result = -1;
      else if( Task1.priority() < Task2.priority() )
	Result = 1;
      else if( Task2.priority() < Task1.priority() )
	Result = -1;
    }
    return Result;
  } // compare( const Task&, const Task& )

#ifdef PLANSPLANT_TIME_LIST_NO_INDENT
  bool TasksTimeListWidget::span( const QModelIndex& Index ) const
  {
    bool Result = false;
    if( TasksTimeListModel::Item* It = static_cast<TasksTimeListModel*>( model() )->item_from_index( Index ) )
      Result = !It->indent();
    return Result;
  } // span( const QModelIndex& ) const
  void TasksTimeListWidget::drawRow( QPainter* Painter, const QStyleOptionViewItem& Option, const QModelIndex& Index ) const
  {
    int X = -horizontalOffset();
    QStyleOptionViewItemV4 Op( Option );
    if( selectionModel()->isSelected( Index ) )
      Op.state |= QStyle::State_Selected;
    if( span( Index ) )
    {
      QVariant VBr = model()->data( Index, Qt::BackgroundRole );
      if( VBr.isValid() )
	Op.backgroundBrush = VBr.value<QBrush>();
      Op.rect = QRect( X, Option.rect.top(), indentation(), Option.rect.height() );
      style()->drawPrimitive( QStyle::PE_PanelItemViewItem, &Op, Painter, this ); // Draw correct background under the branch indicator
      drawBranches( Painter, Op.rect, Index );
      X += indentation();
      Op.rect = QRect( X, Option.rect.top(), Option.rect.right()-X+1, Option.rect.height() );
      if( QAbstractItemDelegate* Delegate = itemDelegate( Index ) )
	Delegate->paint( Painter, Op, Index );
    }
    else
      if( QHeaderView* Header = header() )
      { // uniformRowsHeight must be set to true for this to work.
	if( ( ( verticalOffset()+Option.rect.top() ) / Option.rect.height() ) & 1 )
	  Op.features |= QStyleOptionViewItemV2::Alternate;
	else
	  Op.features &= ~QStyleOptionViewItemV2::Alternate;
	style()->drawPrimitive( QStyle::PE_PanelItemViewRow, &Op, Painter, this );
	for( int I = 0; I < Header->count(); ++I )
	{
	  int Logical = Header->logicalIndex( I );
	  int Right = X + Header->sectionSize( Logical );
	  if( Logical == 0 )
	  {
	    int Width = indent( Index );
	    drawBranches( Painter, QRect( X + Width, Option.rect.top(), indentation(), Option.rect.height() ), Index );
	    X += Width + indentation();
	  }
	  Op.rect = QRect( X, Option.rect.top(), Right-X, Option.rect.height() );
	  QModelIndex Ind = model()->index( Index.row(), Logical, model()->parent( Index ) );
	  QAbstractItemDelegate* Delegate = itemDelegate( Ind );
	  Delegate->paint( Painter, Op, Ind );
	  X = Right;
	}
    }
  } // drawRow( QPainter*, const QStyleOptionViewItem&, const QModelIndex& ) const
  int TasksTimeListWidget::indent( const QModelIndex& Ind ) const
  {
    int Result = 0;
    if( TasksTimeListModel::Item* It = static_cast<TasksTimeListModel*>( model() )->item_from_index( Ind ) )
      for( TasksTimeListModel::Item* Parent = It->parent(); Parent; Parent = Parent->parent() )
	if( Parent->indent() )
	  Result += indentation();
    return Result;
  } // indent( const QModelIndex& )
  void TasksTimeListWidget::mousePressEvent( QMouseEvent* Event )
  {
    if( !( Event->buttons() == Qt::LeftButton && expand_collapse( Event->pos() ) ) )
      QAbstractItemView::mousePressEvent( Event );
  } // mousePressEvent( QMouseEvent* )
  bool TasksTimeListWidget::expand_collapse( const QPoint& Pos )
  {
    bool Result = false;
    QModelIndex Ind = indexAt( Pos );
    if( span( Ind ) ) Ind = model()->index( Ind.row(), 0, Ind.parent() );
    if( Ind.isValid() && Ind.column() == 0 && model()->hasChildren( Ind ) )
    {
      int X = horizontalOffset();
      if( !span( Ind ) ) X = columnViewportPosition( 0 ) + indent( Ind );
      if( Pos.x() >= X && Pos.x() < X + indentation() )
      {
	if( isExpanded( Ind ) ) collapse( Ind );
	else expand( Ind );
	Result = true;
      }
    }
    return Result;
  } // expand_collapse( const QPoint& )
#endif //  PLANSPLANT_TIME_LIST_NO_INDENT
} // PlansPlant
