// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <timeshop.hpp>
#include <plansplant/events.hpp>
#include <plansplant/syncserver.hpp>
#include <QMessageBox>
#include <QDebug>
#include <cctype>
using std::isspace;

namespace PlansPlant
{
  inline void wait( const QString& Text = QString() )
  {
#if 0
    QMessageBox::information( 0, "Wait...", Text );
#endif
  } // wait( const QString& )
  Synchronizer::QueuedEvent::QueuedEvent( const Event& Ev ) : Ident( Ev.id() )
  {
    QXmlStreamWriter Stream( &Data );
    Ev.write( Stream );
  } // QueuedEvent( const Event& )
  Synchronizer::QueuedEvent::QueuedEvent( Event::ID Ident0, const QByteArray& Data0 ) : Ident( Ident0 ), Data( Data0 ) {}

  Synchronizer::Synchronizer( TasksFile& Tasks0 ) : Tasks( Tasks0 ) { add_informer( Tasks ); }
  Synchronizer::~Synchronizer() {} // Here will be Log clearing (no need anymore).
  void Synchronizer::start() {}
  bool Synchronizer::load( QXmlStreamReader& Stream )
  {
    while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      load_field( Stream, Stream.name() );
    return !Stream.hasError();
  } // load( QXmlStreamReader& )
  bool Synchronizer::load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown )
  {
    bool Result = true;
    if( Tag == "queued_event" )
    {
      Event::ID EventID = Stream.attributes().value( "id" ).toString();
      Log.push_back( QueuedEvent( EventID, Stream.readElementText().toUtf8() ) );
    }
    else
    {
      Result = false;
      if( SkipUnknown ) Timeshop::Persistent::Loader::skip( Stream );
    }
    return Result;
  } // load_field( QXmlStreamReader&, const QStringRef& )
  Task::Watcher::Event* Synchronizer::read_event( QXmlStreamReader& Stream )
  {
    Event* Result = 0;
    Event::ID EventID = Stream.attributes().value( "id" ).toString();
    if( EventID && EventID.origin() != Tasks.id() )
    {
      int Type = Task::Watcher::Event::None;
      if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "type", Type ) )
      {
	switch( Type )
	{
	case Task::Watcher::Event::TaskAdded: Result = Task::Watcher::Events::TaskAdded::load( Stream, Tasks ); break;
	case Task::Watcher::Event::TaskRemoved: Result = Task::Watcher::Events::TaskRemoved::load( Stream, Tasks ); break;
	case Task::Watcher::Event::TaskChanged: Result = Task::Watcher::Events::TaskChanged::load( Stream, Tasks ); break;
	case Task::Watcher::Event::RootTaskMoved: Result = Task::Watcher::Events::RootTaskMoved::load( Stream, Tasks ); break;
	default:
	  qDebug() << "Reading events of type" <<  Type << "is not implemented yet.";
	  Timeshop::Persistent::Loader::skip( Stream );
	  break;
	}
      }
      else Timeshop::Persistent::Loader::skip( Stream );
    }
    else
    {
      qDebug() << "Skip event from this origin (loopback on server).";
      Timeshop::Persistent::Loader::skip( Stream );
    }
    return Result;
  } // read_event( QXmlStreamReader& )
  void Synchronizer::write( QXmlStreamWriter& Stream ) const
  {
    qDebug() << "Storing" << Log.size() << "events.";
    foreach( const QueuedEvent Ev, Log ) //! \todo Make it "righter".
    {
      Stream.writeStartElement( "queued_event" );
      Stream.writeAttribute( "id", Ev.id().str() );
      Stream.writeCharacters( QString::fromUtf8( Ev.data().data(), Ev.data().size() ) );
      Stream.writeEndElement();
    }
  } // write( QXmlStreamWriter& ) const
  void Synchronizer::event( const Event& Ev ) { event_out( Ev ); }
  void Synchronizer::event_in( const Event& Ev )
  {
    qDebug() << "Incoming event, id:" << Ev.id().str();
    if( Ev.id().origin() != Tasks.id() )
      Ev.apply( Tasks );
  } // event_in( const Event& )
  // When there will be log, it's good idea to clean removed tasks from all the queues.
  // For example, we will be unable to inform about new task if it's already deleted, because we don't know it's fields.
  void Synchronizer::event_out( const Event& Ev )
  {
    qDebug() << "Outgoing event, id:" << Ev.id().str();
    Log.push_back( Ev );
    new_event_in_queue();
  } // event_out( const Event& )
  void Synchronizer::new_event_in_queue()
  {
    //! \todo Remove events from the Log upon successful sending only.
    Log.pop_front();
  } // new_event_in_queue()

  Transport::Transport( QTcpSocket* Socket0, QObject* Parent ) : QObject( Parent ), CurrState( State::Disconnected), Socket( 0 ), ToSend( 0 ), Offset( 0 )
  {
    if( Socket0 ) socket( Socket0 );
    Timer.setInterval( 5*1000 ); // Default timeout 5 sec
    Timer.setSingleShot( true );
    connect( &Timer, SIGNAL( timeout() ), SLOT( timeout() ) );
  } // Transport( QTcpSocket*, QObject* )
  Transport::~Transport() { if( Socket ) delete Socket; }
  bool Transport::ready() const
  {
    return( CurrState == State::Ready && ToSend == 0 );
  } // ready() const
  void Transport::socket( QTcpSocket* NewSocket )
  {
    if( Socket )
      delete Socket;
    CurrState = State::Disconnected;
    Socket = NewSocket;
    if( Socket )
    {
      if( Socket->state() == QAbstractSocket::ConnectedState ) CurrState = State::Connected;
      connect( Socket, SIGNAL( connected() ), SLOT( connected() ) );
      connect( Socket, SIGNAL( disconnected() ), SLOT( disconnected() ) );
      connect( Socket, SIGNAL( readyRead() ), SLOT( ready_to_read() ) );
      connect( Socket, SIGNAL( bytesWritten( qint64 ) ), SLOT( bytes_written( qint64 ) ) );
    }
  } // socket( QTcpSocket* )

  void Transport::process_queue( const Synchronizer::QueuedEvent::Queue& Ev )
  {
    if( ready() && Offset < Ev.size() )
    {
      qDebug() << "Sending event.";
      ToSend = Socket->write( Ev[ Offset ].data() );
      if( ToSend < Ev[ Offset ].data().size() )
	qDebug() << "!!! Warning: can't send event data in one block. This shouldn't happen.";
    }
  } // process_queue( const Synchronizer::QueuedEvent::Queue& )
  void Transport::disconnect( const QString& ReasonText )
  {
    qDebug() << "Transport disconnect:" << ReasonText;
    delete Socket;
    Socket = 0;
  } // disconnect( const QString& )
  void Transport::event_is_sent() { Offset++; }
  void Transport::connected() { qDebug() << "Socket connected."; }
  void Transport::disconnected()
  {
    CurrState = State::Disconnected;
    qDebug() << "Socket disconnected.";
    if( Socket )
    {
      Socket->deleteLater();
      Socket = 0;
    }
  } // disconnected()
  void Transport::ready_to_read()
  {
    if( Socket )
    {
      Buffer += Socket->readAll();
      qDebug() << "Received:" << QString::fromUtf8( Buffer );
      int Pos = 0;
      while( Pos < Buffer.size() )
      {
	while( Pos < Buffer.size() && isspace( Buffer[ Pos ] ) ) Pos++;
	if( Pos < Buffer.size() )
	{
	  if( Buffer[ Pos ] == '<' ) // XML block. Note that no comments allowed.
	  {
	    int EndPos = Buffer.indexOf( '>', Pos );
	    if( EndPos > Pos )
	    {
	      if( Buffer[ EndPos-1 ] != '/' ) // Not empty tag
	      {
		for( int I = Pos+1; I < EndPos; I++ )
		  if( isspace( Buffer[ I ] ) )
		    EndPos = I;
		QByteArray Stop = "</" + Buffer.mid( Pos+1, EndPos-Pos-1 ) + '>';
		qDebug() << "Search for" << QString::fromUtf8( Stop );
		EndPos = Buffer.indexOf( Stop, Pos );
		if( EndPos > Pos ) EndPos += Stop.size()-1;
	      }
	      if( EndPos > Pos )
	      {
		Timer.stop();
		xml_block( Buffer.mid( Pos, EndPos-Pos+1 ) );
		Pos = EndPos+1;
	      }
	      else qDebug() << "Transport: Incomplete XML block!";
	    }
	    else qDebug() << "Transport: Incomplete XML start tag!";
	    if( EndPos <= Pos ) Pos = Buffer.size();
	  }
	  else // Text line.
	  {
	    int EndPos = Buffer.indexOf( '\n', Pos );
	    if( EndPos >= 0 )
	    {
	      Timer.stop();
	      text_line( Buffer.mid( Pos, EndPos-Pos+1 ) );
	      Pos = EndPos+1;
	    }
	    else
	    {
	      qDebug() << "Transport: Incomplete text line!";
	      Pos = Buffer.size();
	    }
	  } // Text of XML
	}
      } // Loop through the Buffer
      Buffer.remove( 0, Pos );
      if( !Buffer.isEmpty() ) Timer.start(); // Start timeout for block.
    } // Socket != 0
  } // ready_to_read()
  void Transport::bytes_written( qint64 Count )
  {
    if( ToSend > 0 )
    {
      if( Count > ToSend )
      {
	qDebug() << "!!! Warning: sent more bytes" << Count << "then needed:" << ToSend;
	ToSend = 0;
      }
      else
	ToSend -= Count;
      if( ToSend == 0 )
	event_is_sent();
    }
  } // bytes_written( qint64 )
  void Transport::timeout()
  {
    // Simply disconnect.
    disconnect( "Transport: timeout." );
  } // timeout()
  void Transport::text_line( const QByteArray& Text ) { disconnect( "Transport: enexpected text line:" + QString::fromUtf8( Text ) ); }
  void Transport::xml_block( const QByteArray& Block ) { disconnect( "Transport: enexpected XML block:" + QString::fromUtf8( Block ) ); }
  void Transport::write( QXmlStreamWriter& Stream ) const
  {
    if( !Password.isEmpty() ) Stream.writeTextElement( "password", Password );
    if( !Prompt.isEmpty() ) Stream.writeTextElement( "prompt", Prompt );
    if( Offset > 0 ) Stream.writeTextElement( "offset", QString::number( Offset ) );
  } // write( QXmlStreamWriter& ) const
  bool Transport::load( QXmlStreamReader& Stream )
  {
    while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      load_field( Stream, Stream.name() );
    return !Stream.hasError();
  } // load( QXmlStreamReader& )
  bool Transport::load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown )
  {
    bool Result = true;
    if( Tag == "prompt" )
      Prompt = Stream.readElementText();
    else if( Tag == "password" )
      Password = Stream.readElementText();
    else if( Tag == "offset" )
      Offset = Stream.readElementText().toInt();
    else
    {
      Result = false;
      if( SkipUnknown ) Timeshop::Persistent::Loader::skip( Stream );
    }
    return Result;
  } // load_field( QXmlStreamReader&, const QStringRef& )
  
  Node::Node( TasksFile& Tasks0, const QString& HostName0, quint16 Port0 )
    : Synchronizer( Tasks0 ), HostName( HostName0 ), Port( Port0 )
  {
  } // Node( TasksFile&, int, const QString&, quint16 )
  void Node::start()
  {
    socket( new QTcpSocket( this ) );
    qDebug() << "Connecting to" << HostName << ':' << Port;
    Socket->connectToHost( HostName, Port );
  } // start()
  void Node::connected()
  {
    qDebug() << "Socket connected.";
    CurrState = State::Connected;
    Timer.start();
  } // connected()
  void Node::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeAttribute( "type", "client" );
    Stream.writeTextElement( "host", HostName );
    Stream.writeTextElement( "port", QString::number( Port ) );
    Transport::write( Stream );
    Synchronizer::write( Stream );
  } // write( QXmlStreamWriter& ) const
  bool Node::load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown )
  {
    bool Result = true;
    if( Tag == "host" )
      HostName = Stream.readElementText();
    else if( Tag == "port" )
      Port = Stream.readElementText().toInt();
    else if( !Transport::load_field( Stream, Tag, false ) )
      Result = Synchronizer::load_field( Stream, Tag, SkipUnknown );
    return Result;
  } // load_field( QXmlStreamReader&, const QStringRef&, bool )
  void Node::new_event_in_queue()
  {
    process_queue( Log );
  } // new_event_in_queue()
  void Node::event_is_sent()
  {
    for( Transport::event_is_sent(); Offset > 0; Offset-- )
      Log.pop_front();
    process_queue( Log );
  } // event_is_sent()
  void Node::event_out( const Event& Ev )
  {
    if( Ev.id().origin() == Tasks.id() )
      Synchronizer::event_out( Ev ); // Put event into the queue
  } // event_out( const Event& )
  void Node::text_line( const QByteArray& Text )
  {
    if( CurrState == State::Connected )
      if( Text == Server::Signature )
      {
	wait( "Node send <node>." );
	QXmlStreamWriter Stream( Socket );
	Stream.writeStartElement( "node" );
	Stream.writeAttribute( "id", QString::number( Tasks.id() ) );
	Stream.writeStartElement( "file" );
	Stream.writeTextElement( "uuid", Tasks.uuid() );
	Stream.writeEndElement();
	Stream.writeEndElement();
	CurrState = State::UUID;
	Timer.start();
      }	
      else
	disconnect( "No server signature." );
    else
      qDebug() << "Node: text line in wrong state - ignored.";
  } // text_line( const QByteArray& )
  void Node::xml_block( const QByteArray& Block )
  {
    QXmlStreamReader Stream( Block );
    if( Timeshop::Persistent::Loader::next_subelement( Stream ) )
    {
      qDebug() << "Node: XML block:" << Stream.name();
      switch( CurrState )
      {
      case State::UUID:
	if( Stream.name() == "prompt" )
	  if( Stream.readElementText() == Prompt )
	  {
	    wait( "Node send <password>." );
	    QXmlStreamWriter Out( Socket );
	    Out.writeTextElement( "password", Password );
	    CurrState = State::Password;
	    Timer.start();
	  }
	  else
	    disconnect( "Wrong prompt from the server." );
	else if( Stream.name() == "not_found" )
	  disconnect( "File with UUID " + Tasks.uuid() + " not found on server." );
	else
	  disconnect( "Wrong input (no prompt) from server." );
	break;
      case State::Password:
	if( Stream.name() == "ready" )
	{
	  CurrState = State::Ready;
	  process_queue( Log );
	}
	else
	  disconnect( "No <ready> answer from server." );
	break;
      case State::Ready:
	if( Stream.name() == "event" )
	{
	  if( Event* Ev = read_event( Stream ) )
	  {
	    if( Ev->id().origin() == Tasks.id() )
	      qDebug() << "Node: Skip event from this node (loopback on server).";
	    else
	      event_in( *Ev );
	    delete Ev;
	  }
	  else
	    qDebug() << "Read_event failed.";
	}
	break;
      default:
	qDebug() << "Node: ignore XML block in wrong state:" << CurrState;
	break;
      }
    }
  } // xml_block( const QByteArray& )

  Server::Client::Client( Server& Server0, int Ident0 ) : Srv( Server0 ), Ident( Ident0 ) {}
  Server::Client::~Client() {}
  void Server::Client::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeStartElement( "client" );
    Stream.writeAttribute( "id", QString::number( Ident ) );
    Transport::write( Stream );
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Server::Client::event_in( const Event& Ev ) { Srv.event_in( Ev ); }
  void Server::Client::event_is_sent()
  {
    Transport::event_is_sent();
    Srv.event_is_sent( *this );
  } // event_is_sent()
  void Server::Client::process_queue( const Synchronizer::QueuedEvent::Queue& Ev )
  {
    while( Offset < Ev.size() && Ev[ Offset ].id().origin() == id() )
      Offset++;
    Transport::process_queue( Ev );
  } // process_queue( const Synchronizer::QueuedEvent::Queue& )
  void Server::Client::xml_block( const QByteArray& Block )
  {
    QXmlStreamReader Stream( Block );
    if( Timeshop::Persistent::Loader::next_subelement( Stream ) )
    {
      qDebug() << "Server: XML block:" << Stream.name();
      switch( CurrState )
      {
      case State::Connected:
	if( Stream.name() == "node" )
	{
	  wait( "Server send <prompt>." );
	  QXmlStreamWriter Out( Socket );
	  Out.writeTextElement( "prompt", Prompt );
	  CurrState = State::Password;
	  Timer.start();
	}
	else
	  disconnect( "Server: wrong block from node (expected <node>)." );
	break;
      case State::Password:
	if( Stream.name() == "password" )
	  if( Stream.readElementText() == Password )
	  {
	    wait( "Server send <ready>." );
	    QXmlStreamWriter Out( Socket );
	    Out.writeStartElement( "ready" ); // Simple writeEmptyElement don't closes it but waits for attributes.
	    Out.writeEndElement(); 
	    CurrState = State::Ready;
	  }
	  else
	    disconnect( "Server: wrong password from the node." );
	else
	  disconnect( "Server: wrong block from node (expected <password>)." );
	break;
      case State::Ready:
	if( Stream.name() == "event" )
	{
	  if( Event* Ev = Srv.read_event( Stream ) )
	    event_in( *Ev );
	  else
	    qDebug() << "Read_event_failed.";
	}
	break;
      default:
	qDebug() << "Node: ignore XML block in wrong state:" << CurrState;
	break;
      }
    }
  } // xml_block( const QByteArray& )
  Server::Server( TasksFile& Tasks0, quint16 Port0 ) : Synchronizer( Tasks0 ), Port( Port0 )
  {
  } // Server( TasksFile&, quint16 )
  Server::~Server()
  {
    if( Coordinator ) Coordinator->remove_server( *this );
    while( !Clients.empty() )
    {
      delete Clients.back();
      Clients.pop_back();
    }
  } // ~Server()
  void Server::start()
  {
    if( !Coordinator )
      Coordinator = new MetaServer;
    Coordinator->add_server( *this );
  } // start()
  void Server::event_is_sent( Client& Cli )
  {
    int NewTop = Log.size();
    for( Client::List::const_iterator It = Clients.begin(); NewTop > 0 && It != Clients.end(); It++ )
      if( (*It)->offset() < NewTop )
	NewTop = (*It)->offset();
    if( NewTop > 0 )
    {
      for( int I = 0; I < NewTop; I++ )
	Log.pop_front();
      foreach( Client* C, Clients )
	C->adjust_offset( NewTop );
    }
    Cli.process_queue( Log );
  } // event_is_sent( Client& )
  void Server::write( QXmlStreamWriter& Stream ) const
  {
    Stream.writeAttribute( "type", "server" );
    Stream.writeTextElement( "port", QString::number( Port ) );
    foreach( Client* Cli, Clients )
      Cli->write( Stream );
    Synchronizer::write( Stream );
  } // write( QXmlStreamWriter& ) const
  void Server::new_event_in_queue()
  {
    foreach( Client* Cli, Clients )
      Cli->process_queue( Log );
  } // new_event_in_queue()
  bool Server::load_field( QXmlStreamReader& Stream, const QStringRef& Tag, bool SkipUnknown )
  {
    bool Result = true;
    if( Tag == "port" )
      Port = Stream.readElementText().toInt();
    else if( Tag == "client" )
    {
      int ClientID = -1;
      if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "id", ClientID ) )
      {
	qDebug() << "Load client with ID" << ClientID;
	Client* Cli = new Client( *this, ClientID );
	Cli->load( Stream );
	Clients.push_back( Cli );
      }
      else
      {
	qDebug() << "Encountered client without ID";
	Timeshop::Persistent::Loader::skip( Stream );
      }
    }
    else
      Result = Synchronizer::load_field( Stream, Tag, SkipUnknown );
    return Result;
  } // load_field( QXmlStreamReader&, const QStringRef&, bool )
  MetaServer* Server::Coordinator = 0;
  const char* Server::Signature = "Plans Plant Server\n";

  // MetaServer::Porter
  MetaServer::Porter::Porter( MetaServer& Master0, QTcpSocket* Socket0 ) : Transport( Socket0, &Master0 ), Master( Master0 )
  {
    wait( "Server send Signature." );
    Socket->write( Server::Signature );
    Timer.start();
  } // Porter( MetaServer&, QTcpSocket* )
  MetaServer::Porter::~Porter() { Master.remove_porter( *this ); }
  void MetaServer::Porter::disconnected()
  {
    Transport::disconnected();
    deleteLater();
  } // disconnected()
  void MetaServer::Porter::xml_block( const QByteArray& Block )
  {
    qDebug() << "Porter: XML block:" << QString::fromUtf8( Block );
    QXmlStreamReader Stream( Block );
    if( Timeshop::Persistent::Loader::next_subelement( Stream ) )
    {
      qDebug() << "Porter: XML element:" << Stream.name();
      if( Stream.name() == "node" )
      {
	int NewID = Stream.attributes().value( "id" ).toString().toInt();
	if( NewID > 0 )
	{
	  QString UUID;
	  while( UUID.isEmpty() && Timeshop::Persistent::Loader::next_subelement( Stream ) )
	    if( Stream.name() == "file" )
	      while( UUID.isEmpty() && Timeshop::Persistent::Loader::next_subelement( Stream ) )
		if( Stream.name() == "uuid" )
		  UUID =  Stream.readElementText();
	  if( UUID.isEmpty() )
	    disconnect( "Porter: node without UUID." );
	  else
	  {
	    if( Server::Client* Cli = Master.client_identified( UUID, NewID ) )
	    {
	      qDebug() << "Porter: Client for node" << NewID << "from file with UUID" << UUID << "found. Pass the socket there.";
	      Socket->disconnect( this );
	      Cli->socket( Socket );
	      Socket = 0; // Prevent destruction.
	      Cli->xml_block( Block ); // Let the client parse the rest of the fields.
	    }
	    else
	    {
	      wait( "Server send <not_found>." );
	      QXmlStreamWriter Out( Socket );
	      Out.writeStartElement( "not_found" );
	      Out.writeEndElement();
	      Socket->waitForBytesWritten(); //!< \todo Wait until this is sent by signal, not timeout.
	      disconnect( "Porter: Client for node " + QString::number( NewID ) + " from file with UUID \"" + UUID + "\" not found." );
	    }
	  }
	}
	else
	  disconnect( "Porter: node without ID." );
      }
      else
	disconnect( "Porter: wrong block from node (expected <node>)." );
    }
    else
      disconnect( "Porter: What? No start element in XML block?" );
  } // xml_block( const QByteArray& )
  // MetaServer
  MetaServer::MetaServer( QObject* Parent ) : QObject( Parent ) {}
  MetaServer::~MetaServer()
  {
    foreach( Porter* P, Porters )
      delete P;
  } // ~MetaServer()
  void MetaServer::add_server( Server& NewServer )
  {
    if( !Servers.contains( &NewServer ) )
    {
      Servers.push_back( &NewServer );
      QTcpServer* Listener = 0;
      for( QList<QTcpServer*>::const_iterator It = Listeners.begin(); !Listener && It != Listeners.end(); It++ )
	if( *It && (*It)->serverPort() == NewServer.port() )
	  Listener = *It;
      if( !Listener )
      {
	qDebug() << "MetaServer: Start listener on port" << NewServer.port();
	Listener = new QTcpServer( this );
	Listeners.push_back( Listener );
	connect( Listener, SIGNAL( newConnection() ), SLOT( connection() ) );
	Listener->listen( QHostAddress::Any, NewServer.port() );
      }
    }
  } // add_server( Server& )
  void MetaServer::remove_server( Server& OldServer )
  {
    Servers.removeAll( &OldServer );
    bool StopListener = true;
    for( Server::List::const_iterator It = Servers.begin(); StopListener && It != Servers.end(); It++ )
      if( (*It) && (*It)->port() == OldServer.port() )
	StopListener = false;
    if( StopListener )
      for( QList<QTcpServer*>::const_iterator It = Listeners.begin(); It != Listeners.end(); It++ )
	if( QTcpServer* Listener = *It )
	  if( Listener->serverPort() == OldServer.port() )
	  {
	    qDebug() << "MetaServer: Stop listener on" << Listener->serverPort();
	    Listener->close();
	    Listener->deleteLater();
	  }
  } // remove_server( Server& OldServer )
  void MetaServer::remove_porter( Porter& OldPorter ) { Porters.removeAll( &OldPorter ); }
  Server::Client* MetaServer::client_identified( const QString& FileUUID, int ClientID )
  {
    Server::Client* Result = 0;
    Server* Svr = 0;
    for( Server::List::const_iterator It = Servers.begin(); !Svr && It != Servers.end(); It++ )
      if( (*It) && (*It)->tasks().uuid() == FileUUID )
	Svr = *It;
    if( Svr )
    {
      for( Server::Client::List::const_iterator It = Svr->clients().begin(); !Result && It != Svr->clients().end(); It++ )
	if( (*It) && (*It)->id() == ClientID )
	  Result = *It;
    }
    else
      qDebug() << "Server for file with UUID" << FileUUID << "not found.";
    return Result;
  } // client_identified( const QString&, int )
  void MetaServer::connection()
  {
    qDebug() << "Metaserver: connection signalled.";
    foreach( QTcpServer* Listener, Listeners )
      if( Listener && Listener->hasPendingConnections() )
	while( QTcpSocket* Socket = Listener->nextPendingConnection() )
	{
	  qDebug() << "Incoming client connection from" << Socket->peerAddress() << ':' << Socket->peerPort();
	  Porters.push_back( new Porter( *this, Socket ) );
	}
  } // connection()
} // PlansPlant
