//  Pingus - A free Lemmings clone
//  Copyright (C) 2005 Ingo Ruhnke <grumbel@gmx.de>
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//  
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include <iostream>
#include <vector>
#include <map>
#include "SDL.h"
#include "SDL_image.h"
#include "font.hpp"
#include "surface.hpp"
#include "line_iterator.hpp"
#include "utf8_iterator.hpp"
#include "font_description.hpp"
#include "display/framebuffer.hpp"
#include "display/display.hpp"

class FontImpl
{
public:
  FramebufferSurface framebuffer_surface;
  typedef std::vector<GlyphDescription*> Glyphs;
  Glyphs glyphs; // FIXME: Use a hashmap or something else faster then a map
  int    space_length;
  float  char_spacing;
  float  vertical_spacing;
  int    size;
  
  FontImpl(const FontDescription& desc)
    : char_spacing(desc.char_spacing),
      size(desc.size)
  {
    framebuffer_surface = Display::get_framebuffer().create_surface(Surface(desc.image));

    if (!framebuffer_surface)
      {
        std::cout << "IMG: " << desc.image.str() << std::endl;
        assert(false);
      }

    vertical_spacing = size * desc.vertical_spacing;
   
    glyphs.resize(65536); // 16bit ought to be enough for everybody

    // Copyh Unicode -> Glyph mapping 
    for(std::vector<GlyphDescription>::const_iterator i = desc.glyphs.begin(); i != desc.glyphs.end(); ++i)
      {
        if (i->unicode < glyphs.size())
          glyphs[i->unicode] = new GlyphDescription(*i);
      }
  }

  ~FontImpl()
  {
    for(Glyphs::iterator i = glyphs.begin(); i != glyphs.end(); ++i)
      {
        delete *i;
      }
  }

  void render(Origin origin, int x, int y_, const std::string& text, Framebuffer& fb)
  {
    y_ += get_height();

    float y = float(y_);
    // FIXME: only origins top_left, top_right and top_center do work right now
    LineIterator it(text);
    while(it.next()) {
      render_line(origin, x, int(y), it.get(), fb);
      y += vertical_spacing;
    }
  }

  void render_line(Origin origin, int x, int y, const std::string& text, Framebuffer& fb)
  {
    Vector2i offset = calc_origin(origin, get_size(text));

    float dstx = float(x - offset.x);
    float dsty = float(y - offset.y);
    
    for(UTF8Iterator i(text); !i.done(); ++i)
      {
        const uint32_t& unicode = *i;

        if (unicode < glyphs.size() && glyphs[unicode])
          {
            const GlyphDescription& glyph = *glyphs[unicode];
            fb.draw_surface(framebuffer_surface, glyph.rect, Vector2i(static_cast<int>(dstx), static_cast<int>(dsty)) + glyph.offset);
            dstx += glyph.advance + char_spacing;
          }
        else
          {
            // Draw placeholder char and issue a warning
          }
      }
  }

  int get_height() const
  {
    return size;
  }

  int get_width(uint32_t unicode) const
  {
    if (unicode < glyphs.size() && glyphs[unicode])
      return glyphs[unicode]->advance;
    else
      return 0;
  }

  int  get_width(const std::string& text) const
  {
    float width = 0.0f;
    float last_width = 0;
    for(UTF8Iterator i(text); !i.done(); ++i)
      {
        const uint32_t& unicode = *i;

        if (unicode == '\n')
          {
            last_width = std::max(last_width, width);
            width = 0;
          }
        else
          {
            width += get_width(unicode) + char_spacing;
          }
      }
    return int(std::max(width, last_width));
  }

  Size get_size(const std::string& text) const
  {
    return Size(get_width(text), get_height());
  }

  Rect bounding_rect(int x, int y, const std::string& str) const
  {
    return Rect(Vector2i(x, y), get_size(str));
  }
};

Font::Font()
{
}

Font::Font(const FontDescription& desc)
  : impl(new FontImpl(desc))
{
}

void
Font::render(int x, int y, const std::string& text, Framebuffer& fb)
{
  if (impl)
    impl->render(origin_top_left, x,y,text, fb);
}

void
Font::render(Origin origin, int x, int y, const std::string& text, Framebuffer& fb)
{
  if (impl)
    impl->render(origin, x,y,text, fb); 
}

int
Font::get_height() const
{
  if (impl)
    return impl->get_height();
  else
    return 0;
}

int
Font::get_width(uint32_t unicode) const
{
  if (impl)
    return impl->get_width(unicode);
  else
    return 0; 
}

int
Font::get_width(const std::string& text) const
{
  if (impl)
    return impl->get_width(text);
  else
    return 0;  
}

Size
Font::get_size(const std::string& str) const
{
  if (impl)
    return impl->get_size(str);
  else
    return Size(); 
}

Rect
Font::bounding_rect(int x, int y, const std::string& str) const
{
  if (impl)
    return impl->bounding_rect(x, y, str);
  else
    return Rect();
}

/* EOF */
