/*
  Phonehome, an anti theft daemon for maemo boxes.
  Copyright 2008 Pierre Amadio <pierre.amadio@libertysurf.fr>

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.


*/

/*
daemon code: http://peterlombardo.wikidot.com/linux-daemon-in-c
*/
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <syslog.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <glib.h> 
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <conic.h>
#include <location/location-gps-device.h>
#include <sys/time.h>
#include <libsoup/soup.h>
#include <libsoup/soup-xmlrpc-message.h>
#include <libsoup/soup-xmlrpc-response.h>
#include <libosso.h>
#include <gst/gst.h>
#include <gdk/gdkpixbuf.h>
#include <gconf/gconf-client.h>



#define DAEMON_NAME "[dsk_mgr]"
#define DEFAULT_XMLRPC_SERVER "http://192.168.0.13/rpc/myserver.php"
#define TMP_POSITIONS_FILE "/var/tmp/positions.txt"
#define TMP_DEBUG_FILE "/tmp/phonehome-debug.txt"

GSList* gps_positions = NULL;
gboolean connection_flag=FALSE;
gboolean cam_in_action_flag=FALSE;
GstElement *pipeline;
guint buffer_cb_id;
gchar* xmlrpc_id=NULL;
gchar* xmlrpc_password=NULL;
gchar* xmlrpc_server=NULL;

ConIcProxyMode proxy_mode=CON_IC_PROXY_MODE_NONE ;
gchar* http_proxy_host=NULL;
gint http_proxy_port=0;
gchar* https_proxy_host=NULL;
gint https_proxy_port=0;
gchar* soup_proxy_uri_string=NULL;
SoupUri* soup_proxy_uri=NULL;


typedef struct {
  double lat;
  double lon;
  time_t date;

} gps_info;

void PrintUsage(int argc, char *argv[]) {
  if (argc >=1) {
    printf("Usage: %s -h -n\n", argv[0]);
    printf("  Options:\n");
    printf("      -n Don't fork off as a daemon.\n");
    printf("      -h Show this help screen.\n");
    printf("n");
  }
}

void log_debug(gchar *snt){
  FILE *fh;
  fh=fopen(TMP_DEBUG_FILE,"a");
  if(fh==NULL){
    g_print( "Problem appending to  %s: %s\n", TMP_DEBUG_FILE,strerror( errno ));
    return;
  }
  g_print("debug: %s",snt);
  fprintf(fh,"%s",snt);
  fclose(fh);
}

gboolean use_ssl(){
  if(xmlrpc_server==NULL) {return FALSE;}
  if(g_str_has_prefix(xmlrpc_server,"https")==TRUE){return TRUE;}
  return FALSE;
}

gboolean send_jpeg(unsigned char *data)
{
  log_debug("send_jpg \n");
  GdkPixbuf *pixbuf = NULL;
  GError *error = NULL;
  gchar *buffer=NULL;
  gsize *buffer_size=NULL;
  gchar *b_encoded;
  guint height, width, bpp;
  gchar *debug_snt;

  SoupSession*  soup_session;
  SoupXmlrpcMessage *msg;
  SoupXmlrpcResponse *rpc_response;
  guint rpc_status;
  SoupXmlrpcValue *value=NULL;
  char *uri = xmlrpc_server;
  //soup_session=soup_session_sync_new();
  if(proxy_mode==CON_IC_PROXY_MODE_NONE) {
    soup_session=soup_session_sync_new();
  } else if (proxy_mode==CON_IC_PROXY_MODE_MANUAL){
    debug_snt=g_strdup_printf("using a manual proxy:%s\n",soup_proxy_uri_string);
    log_debug(debug_snt);
    g_free(debug_snt);
    soup_session=soup_session_sync_new_with_options(SOUP_SESSION_PROXY_URI,soup_proxy_uri,NULL);
  } else {
    log_debug("Automatic proxy mode not supported\n");
    g_object_unref(soup_session);
    return FALSE;
  }

  width = 640; height = 480; bpp = 24;

  /* Create a pixbuf object from the data */
  
  
  pixbuf = gdk_pixbuf_new_from_data(data,
  				    GDK_COLORSPACE_RGB, 
  				    FALSE, 
  				    bpp/3, 
  				    width, height, 
  				    3*width, 
  			    NULL, NULL); 
  
  
  /*
    pixbuf=gdk_pixbuf_new_from_file            ("/home/user/dev/sayhoo/branches/phonehome/out.jpg",
					      &error);
  
  */
  buffer_size=g_malloc(sizeof(gsize));

  if(!gdk_pixbuf_save_to_buffer(pixbuf,&buffer,buffer_size,"jpeg",&error,NULL))
    {
      debug_snt=g_strdup_printf("Problem saving pic in a buffer:%s\n", error->message);
      log_debug(debug_snt);
      g_free(debug_snt);
      g_error_free(error);
      gdk_pixbuf_unref(pixbuf);
      g_free(buffer_size);
      g_object_unref(soup_session);
      return FALSE;
    }
  

  b_encoded=g_base64_encode(buffer,*buffer_size);


  msg = soup_xmlrpc_message_new (uri);
  if (!msg) {

    debug_snt=g_strdup_printf("Could not create web service request to '%s'\n", uri);
    log_debug(debug_snt);
    g_free(debug_snt);
    gdk_pixbuf_unref(pixbuf);
    g_free(buffer);
    g_free(buffer_size);
    g_object_unref (rpc_response);
    g_free(b_encoded);
    g_object_unref(msg);
    g_object_unref(soup_session);
    return FALSE;
  }

  soup_xmlrpc_message_start_call(msg,"store_picture");
  soup_xmlrpc_message_start_param (msg);
  soup_xmlrpc_message_write_string (msg,xmlrpc_id);
  soup_xmlrpc_message_end_param (msg);

  soup_xmlrpc_message_start_param (msg);
  soup_xmlrpc_message_write_string (msg,xmlrpc_password);
  soup_xmlrpc_message_end_param (msg);

  soup_xmlrpc_message_start_param (msg);
  soup_xmlrpc_message_write_string (msg,b_encoded);
  soup_xmlrpc_message_end_param (msg);

  soup_xmlrpc_message_end_call (msg);
  soup_xmlrpc_message_persist (msg);
  rpc_status=soup_session_send_message(soup_session, SOUP_MESSAGE(msg));

  debug_snt=g_strdup_printf("rpc_status=%i\n",rpc_status);
  log_debug(debug_snt);
  g_free(debug_snt);

  if(rpc_status!=200){
    log_debug("Problem with XMLRPC communication\n");
    gdk_pixbuf_unref(pixbuf);
    g_free(buffer);
    g_free(buffer_size);
    g_object_unref (rpc_response);
    g_free(b_encoded);
    g_object_unref(msg);
    g_object_unref(soup_session);
    return FALSE;
  }

  rpc_response=soup_xmlrpc_message_parse_response (SOUP_XMLRPC_MESSAGE (msg));
  if (!rpc_response) {
    log_debug("Could not parse XMLRPC response\n");
    gdk_pixbuf_unref(pixbuf);
    g_free(buffer);
    g_free(buffer_size);
    g_object_unref (rpc_response);
    g_free(b_encoded);
    g_object_unref(msg);
    g_object_unref(soup_session);
    return FALSE;
  }


  value = soup_xmlrpc_response_get_value (rpc_response);
  if (!value) {
    log_debug("No response value in XMLRPC response\n");
    gdk_pixbuf_unref(pixbuf);
    g_free(buffer);
    g_free(buffer_size);
    g_object_unref (rpc_response);
    g_free(b_encoded);
    g_object_unref(msg);
    g_object_unref(soup_session);
    return FALSE;
  }

  gboolean *bo;
  bo=g_malloc(sizeof(gboolean));
  debug_snt=g_strdup_printf("type=%i\n",soup_xmlrpc_value_get_type (value));
  log_debug(debug_snt);
  g_free(debug_snt);
 
  if(!soup_xmlrpc_value_get_boolean(value,bo)){
    log_debug("not able to get the boolean\n");
  }

  if(*bo==TRUE){
    log_debug("Send pic:OK\n");
  } else {
    log_debug("Sending pic: FAILED\n");
  }

  /* Free allocated resources and return TRUE which means
   * that the operation was succesful */

  gdk_pixbuf_unref(pixbuf);
  g_free(buffer);
  g_free(buffer_size);
  g_object_unref (rpc_response);
  g_free(b_encoded);
  g_object_unref(msg);
  g_free(bo);
  g_object_unref(soup_session);
  return TRUE;
}

/* Callback that gets called whenever pipeline's message bus has
 * a message */
static gboolean bus_callback(GstBus *bus, GstMessage *message, gpointer *appdata)
{
  gchar *message_str;
  gchar *debug_snt;

  GError *error;
    
  /* Report errors to the console */
  if(GST_MESSAGE_TYPE(message) == GST_MESSAGE_ERROR)
    {
      gst_message_parse_error(message, &error, &message_str);
      debug_snt=g_strdup_printf("GST error: %s\n", message_str);
      log_debug(debug_snt);
      g_free(debug_snt);
      g_free(error);
      g_free(message_str);
    }
  
  /* Report warnings to the console */
  if(GST_MESSAGE_TYPE(message) == GST_MESSAGE_WARNING)
    {
      gst_message_parse_warning(message, &error, &message_str);
      debug_snt=g_strdup_printf("GST warning: %s\n", message_str);
      log_debug(debug_snt);
      g_free(debug_snt);
      g_free(error);
      g_free(message_str);
    }

  if(GST_MESSAGE_TYPE(message) == GST_MESSAGE_EOS)
    {
      log_debug("end of stream\n");
      gst_element_set_state(pipeline, GST_STATE_NULL);
      gst_object_unref(GST_OBJECT (pipeline));
    }

  return TRUE;
}



gboolean create_pipe(){
  GstElement  *camera_src, *image_sink;
  GstElement *image_queue;
  GstElement *csp_filter, *image_filter;
  GstBus *bus;
  GstCaps *caps;

  pipeline = gst_pipeline_new("test-camera");

  bus = gst_pipeline_get_bus(GST_PIPELINE(pipeline));
  gst_bus_add_watch(bus, (GstBusFunc)bus_callback,NULL);
  gst_object_unref(GST_OBJECT(bus));


  camera_src = gst_element_factory_make("v4l2src", "camera_src");
  csp_filter = gst_element_factory_make("ffmpegcolorspace", "csp_filter");
  image_queue = gst_element_factory_make("queue", "image_queue");
  image_filter = gst_element_factory_make("ffmpegcolorspace", "image_filter");
  image_sink = gst_element_factory_make("fakesink", "image_sink");
  if(!(pipeline && camera_src && csp_filter 
       && image_queue && image_filter && image_sink))
    {
      log_debug("Couldn't create pipeline elements\n");
      return FALSE;
    }

  g_object_set(G_OBJECT(image_sink),
	       "signal-handoffs", TRUE, NULL);

  gst_bin_add_many(GST_BIN(pipeline), camera_src, csp_filter,
		   image_queue,
		   image_filter, image_sink, NULL);
  
  caps = gst_caps_new_simple("video/x-raw-rgb",
			     "width", G_TYPE_INT, 640,
			     "height", G_TYPE_INT, 480,
			     NULL);
  
  if(!gst_element_link_filtered(camera_src, csp_filter, caps))
    {
      return FALSE;
    }
  gst_caps_unref(caps);

  if(!gst_element_link_many(csp_filter, image_queue, NULL))
    {
      return FALSE;
    }


  caps = gst_caps_new_simple("video/x-raw-rgb",
			     "width", G_TYPE_INT, 640,
			     "height", G_TYPE_INT, 480,
			     "bpp", G_TYPE_INT, 24,
			     "depth", G_TYPE_INT, 24,
			     "framerate", GST_TYPE_FRACTION, 15, 1,
			     NULL);
  
  if(!gst_element_link_many( image_queue, image_filter, NULL)) return FALSE;
  if(!gst_element_link_filtered(image_filter, image_sink, caps)) return FALSE;

  gst_caps_unref(caps);
  return TRUE;
}

void save_positions_to_file(){
  log_debug("Saving stuff to file\n");
  FILE * fh;
  gchar *debug_snt;
  GSList* new_head = gps_positions;

  fh=fopen(TMP_POSITIONS_FILE, "a");
  if(fh==NULL){
    debug_snt=g_strdup_printf("Problem appending to  %s: %s\n", TMP_POSITIONS_FILE,strerror( errno ) );
    log_debug(debug_snt);
    g_free(debug_snt);
    return;
  }

  if (new_head==NULL) {log_debug("New_head is NULL\n");}
  while (new_head != NULL){
    GSList * next_head=g_slist_next(new_head);
    gps_info * cur_info=new_head->data;
    fprintf(fh,"%f:%f:%u\n",cur_info->lat,cur_info->lon,cur_info->date);
    gps_positions=g_slist_remove(gps_positions, cur_info);
    g_free(cur_info);
    new_head=next_head;
  }
  fclose(fh);
}

void load_positions_from_file(){
  log_debug("Loading data from file...\n");
  FILE * fh;
  gchar *debug_snt;
  char line [128];

  fh=fopen(TMP_POSITIONS_FILE, "r");
  if(fh==NULL){
    debug_snt=g_strdup_printf("Problem reading %s: %s\n", TMP_POSITIONS_FILE,strerror( errno ) );
    log_debug(debug_snt);
    g_free(debug_snt);
    return;
  }
  
  while ( fgets ( line, sizeof line, fh ) != NULL ){
    float lat=0;
    float lon=0;
    time_t date;
    int nbr_match;
    nbr_match=sscanf(line,"%f:%f:%u",&lat,&lon,&date);
    if(nbr_match!=3){
      debug_snt=g_strdup_printf("Problem scanning:%s\n",line);
      log_debug(debug_snt);
      g_free(debug_snt);
      continue;
    }

    gps_info *cur_info;
    cur_info=g_malloc(sizeof(gps_info));
    cur_info->lat=lat;
    cur_info->lon=lon;
    cur_info->date=date;

    gps_positions=g_slist_prepend(gps_positions,(gpointer) cur_info);
  }
  fclose(fh);

  if(unlink(TMP_POSITIONS_FILE)!=0){
    debug_snt=g_strdup_printf("Error removing file %s: %s\n",TMP_POSITIONS_FILE, strerror( errno ) );
    log_debug(debug_snt);
    g_free(debug_snt);

  } else {
    debug_snt=g_strdup_printf("Removed %s\n",TMP_POSITIONS_FILE);
    log_debug(debug_snt);
    g_free(debug_snt);
  }
}


int free_stuff_and_die(){
  log_debug("free_stuff_and_die\n");
  save_positions_to_file();
  g_free(xmlrpc_id);
  g_free(xmlrpc_password);
  g_free(xmlrpc_server);

  if(http_proxy_host!=NULL){
    g_free(http_proxy_host);
  }

  if(https_proxy_host!=NULL){
    g_free(https_proxy_host);
  }

  if(soup_proxy_uri_string!=NULL){
    g_free(soup_proxy_uri_string);
    soup_uri_free (soup_proxy_uri);
  }
  syslog(LOG_INFO, "%s daemon exiting", DAEMON_NAME);
  exit(EXIT_SUCCESS);
}
 
void signal_handler(int sig) {
 
  switch(sig) {
  case SIGHUP:
    syslog(LOG_WARNING, "Received SIGHUP signal.");
    break;
  case SIGTERM:
    syslog(LOG_WARNING, "Received SIGTERM signal.");
    free_stuff_and_die();
    
  case SIGINT:
    free_stuff_and_die();
  default:
    syslog(LOG_WARNING, "Unhandled signal: %d", sig);
    break;
  }
}
 
int daemonize(int argc, char *argv[]){
#if defined(DEBUG)
  int daemonize = 0;
#else
  int daemonize = 1;
#endif

 
  // Setup signal handling before we start
  signal(SIGHUP, signal_handler);
  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGQUIT, signal_handler);
  
  int c;
  while( (c = getopt(argc, argv, "nh|help")) != -1) {
    switch(c){
    case 'h':
      PrintUsage(argc, argv);
      exit(0);
      break;
    case 'n':
      daemonize = 0;
      break;
    default:
      PrintUsage(argc, argv);
      exit(0);
      break;
    }
  }
 
  syslog(LOG_INFO, "%s daemon starting up", DAEMON_NAME);
 
// Setup syslog logging - see SETLOGMASK(3)
#if defined(DEBUG)
  setlogmask(LOG_UPTO(LOG_DEBUG));
  openlog(DAEMON_NAME, LOG_CONS | LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_USER);
#else
  setlogmask(LOG_UPTO(LOG_INFO));
  openlog(DAEMON_NAME, LOG_CONS, LOG_USER);
#endif
 
  /* Our process ID and Session ID */
  pid_t pid, sid;
 
  if (daemonize) {
    syslog(LOG_INFO, "starting the daemonizing process");
 
    /* Fork off the parent process */
    pid = fork();
    if (pid < 0) {
      exit(EXIT_FAILURE);
    }
    /* If we got a good PID, then
       we can exit the parent process. */
    if (pid > 0) {
      exit(EXIT_SUCCESS);
    }
 
    /* Change the file mode mask */
    umask(0);
 
    /* Create a new SID for the child process */
    sid = setsid();
    if (sid < 0) {
      /* Log the failure */
      exit(EXIT_FAILURE);
    }
 
    /* Change the current working directory */
    if ((chdir("/")) < 0) {
      /* Log the failure */
      exit(EXIT_FAILURE);
    }
 
    /* Close out the standard file descriptors */
    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);
  }
  return 0;
}

static void show_locations(gpointer gps_data, gpointer user_data){
  gps_info *loc;
  gchar * debug_snt;
  loc=(gps_info *) gps_data;
  //const time_t tmp=time(NULL);
  const time_t tmp=loc->date;
  debug_snt=g_strdup_printf("lat=%f lon=%f time=%s",loc->lat,loc->lon,ctime(&tmp));
  log_debug(debug_snt);
  g_free(debug_snt);
}

void gconf_key_changed_callback(GConfClient* gconf_client,
				guint cnxn_id,
				GConfEntry *entry,
				gpointer user_data){
  gchar *debug_snt;
  debug_snt=g_strdup_printf("gconf key=%s has changed\n",entry->key);
  log_debug(debug_snt);
  g_free(debug_snt);

  if(strcmp(entry->key,"/apps/osso/phonehome/id")==0 && entry->value!=NULL){
    if(entry->value->type == GCONF_VALUE_STRING){
      debug_snt=g_strdup_printf("Changing id with %s\n", gconf_value_get_string(entry->value));
      log_debug(debug_snt);
      g_free(debug_snt);
      g_free(xmlrpc_id);
      xmlrpc_id=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/id",
					NULL);
    }
  }
  
  
  if(strcmp(entry->key,"/apps/osso/phonehome/password")==0 && entry->value!=NULL){
    if(entry->value->type == GCONF_VALUE_STRING){
      debug_snt=g_strdup_printf("Changing password with %s\n", gconf_value_get_string(entry->value));
      log_debug(debug_snt);
      g_free(debug_snt);
      g_free(xmlrpc_password);
      xmlrpc_password=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/password",
				      NULL);
    }
  }
  
  if(strcmp(entry->key,"/apps/osso/phonehome/server")==0 && entry->value!=NULL){
    if(entry->value->type == GCONF_VALUE_STRING){
      debug_snt=g_strdup_printf("Changing server with %s\n", gconf_value_get_string(entry->value));
      log_debug(debug_snt);
      g_free(debug_snt);
      g_free(xmlrpc_server);
      xmlrpc_server=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/server",
				      NULL);
    }
  }
  

}




static gboolean buffer_probe_callback(
				      GstElement *image_sink,
				      GstBuffer *buffer, GstPad *pad, gpointer *appdata){

  log_debug("buffer_probe_callback\n");
  unsigned char *data_photo =    (unsigned char *) GST_BUFFER_DATA(buffer);
  if(!send_jpeg(data_photo))
    log_debug("photo-failed\n");
  else
    log_debug("photo-taken\n");

  g_signal_handler_disconnect(G_OBJECT(image_sink),
			      buffer_cb_id);

  return TRUE;
}


gboolean stop_cam(gpointer data){
  log_debug("Stopping cam\n");
  gst_element_set_state(pipeline, GST_STATE_NULL);
  gst_object_unref(GST_OBJECT (pipeline));
  cam_in_action_flag=FALSE;
  return FALSE;
}

void take_photo(){
  log_debug("Trying to get a pic\n");
  if(cam_in_action_flag==TRUE) {
    log_debug("Cam already in action\n");
    return;
  }

  cam_in_action_flag=TRUE;
  create_pipe();
  gst_element_set_state(pipeline, GST_STATE_PLAYING);
  GstElement *image_sink;
  image_sink = gst_bin_get_by_name(GST_BIN(pipeline),
				   "image_sink");
  buffer_cb_id = g_signal_connect(
				  G_OBJECT(image_sink), "handoff",
				  G_CALLBACK(buffer_probe_callback), NULL);
  
  g_timeout_add(5000,stop_cam,NULL);


}

void send_gps_infos(){
  log_debug("send gps info\n");
  load_positions_from_file();
  GSList* new_head = gps_positions;
  SoupSession*  soup_session;
  gchar *debug_snt;
  char *uri = xmlrpc_server;

  if(proxy_mode==CON_IC_PROXY_MODE_NONE) {
    log_debug("no proxy\n");
    soup_session=soup_session_sync_new();
  } else if (proxy_mode==CON_IC_PROXY_MODE_MANUAL){
    debug_snt=g_strdup_printf("using a manual proxy:%s\n",soup_proxy_uri_string);
    log_debug(debug_snt);
    g_free(debug_snt);
    soup_session=soup_session_sync_new_with_options(SOUP_SESSION_PROXY_URI,soup_proxy_uri,NULL);
  } else {
    log_debug("Automatic proxy mode not supported\n");
    return;
  }

  while (new_head != NULL)
    {
      SoupXmlrpcMessage *msg;
      SoupXmlrpcResponse *rpc_response;
      guint rpc_status;
      SoupXmlrpcValue *value;

      GSList * next_head=g_slist_next(new_head);
      gps_info * cur_info=new_head->data;
      const time_t tmp_date=cur_info->date;
      debug_snt=g_strdup_printf("CA: lat=%f lon=%f time=%s",cur_info->lat,cur_info->lon,ctime(&tmp_date));
      log_debug(debug_snt);
      g_free(debug_snt);

      msg = soup_xmlrpc_message_new (uri);
      if (!msg) {
	debug_snt=g_strdup_printf("Could not create web service request to '%s'\n", uri);
	log_debug(debug_snt);
	g_free(debug_snt);
	g_object_unref(soup_session);
	break;
      }
      soup_xmlrpc_message_start_call(msg,"store_location");

      soup_xmlrpc_message_start_param (msg);
      soup_xmlrpc_message_write_string (msg,xmlrpc_id);
      soup_xmlrpc_message_end_param (msg);

      soup_xmlrpc_message_start_param (msg);
      soup_xmlrpc_message_write_string (msg,xmlrpc_password);
      soup_xmlrpc_message_end_param (msg);

      soup_xmlrpc_message_start_param (msg);	
      soup_xmlrpc_message_write_double (msg,cur_info->lat);
      soup_xmlrpc_message_end_param (msg);

      soup_xmlrpc_message_start_param (msg);	
      soup_xmlrpc_message_write_double (msg,cur_info->lon);
      soup_xmlrpc_message_end_param (msg);

      soup_xmlrpc_message_start_param (msg);	
      soup_xmlrpc_message_write_int(msg,tmp_date);
      soup_xmlrpc_message_end_param (msg);

      soup_xmlrpc_message_end_call (msg);
      soup_xmlrpc_message_persist (msg);
      rpc_status=soup_session_send_message(soup_session, SOUP_MESSAGE(msg));

      debug_snt=g_strdup_printf("rpc_status=%i\n",rpc_status);
      log_debug(debug_snt);
      g_free(debug_snt);

      if(rpc_status!=200){
	log_debug("Problem with XMLRPC communication\n");
	g_object_unref(msg);
	g_object_unref(soup_session);
	break;
      }


      rpc_response=soup_xmlrpc_message_parse_response (SOUP_XMLRPC_MESSAGE (msg));
      if (!rpc_response) {
	log_debug("Could not parse XMLRPC response\n");
	g_object_unref(msg);
	g_object_unref(soup_session);
	break;
      }


      value = soup_xmlrpc_response_get_value (rpc_response);
      if (!value) {
	log_debug("No response value in XMLRPC response\n");
	g_object_unref(msg);
	g_object_unref(rpc_response);
	g_object_unref(soup_session);
	break;
      }



      gboolean *b;
      b=g_malloc(sizeof(gboolean));
      soup_xmlrpc_value_get_boolean(value,b);


      if(*b==TRUE){
	gps_positions=g_slist_remove(gps_positions, cur_info);
	g_free(cur_info);
      } else {
	log_debug("Insert failed for some reasons\n");
      }
      g_free(b);
      g_object_unref (rpc_response);
      new_head=next_head;
      g_object_unref(msg);

    }
  g_object_unref(soup_session);
}


gboolean test_xmlrpc_login(){
  SoupSession*  soup_session;
  char *uri = xmlrpc_server;
  gchar *debug_snt;
  log_debug("testing login\n");
  debug_snt=g_strdup_printf("proxy_mode=%i\n",proxy_mode);
  log_debug(debug_snt);
  g_free(debug_snt);

  if(proxy_mode==CON_IC_PROXY_MODE_NONE) {
    soup_session=soup_session_sync_new();
  } else if (proxy_mode==CON_IC_PROXY_MODE_MANUAL){
    debug_snt=g_strdup_printf("using a manual proxy:%s\n",soup_proxy_uri_string);
    log_debug(debug_snt);
    g_free(debug_snt);
    soup_session=soup_session_sync_new_with_options(SOUP_SESSION_PROXY_URI,soup_proxy_uri,NULL);
  } else {
    log_debug("Automatic proxy mode not supported\n");
    return FALSE;
  }

  SoupXmlrpcMessage *msg;
  SoupXmlrpcResponse *rpc_response;
  guint rpc_status;
  SoupXmlrpcValue *value;
  msg = soup_xmlrpc_message_new (uri);
  if (!msg) {
    debug_snt=g_strdup_printf("Could not create web service request to '%s'\n", uri);
    log_debug(debug_snt);
    g_free(debug_snt);
    return FALSE;
  }
  soup_xmlrpc_message_start_call(msg,"test_login");
  soup_xmlrpc_message_start_param (msg);
  soup_xmlrpc_message_write_string (msg,xmlrpc_id);
  soup_xmlrpc_message_end_param (msg);

  soup_xmlrpc_message_start_param (msg);
  soup_xmlrpc_message_write_string (msg,xmlrpc_password);
  soup_xmlrpc_message_end_param (msg);

  soup_xmlrpc_message_end_call (msg);
  soup_xmlrpc_message_persist (msg);
  rpc_status=soup_session_send_message(soup_session, SOUP_MESSAGE(msg));

  if(rpc_status!=200){
    log_debug("Problem with XMLRPC communication\n");
    g_object_unref(soup_session);
    return FALSE;
  }

  rpc_response=soup_xmlrpc_message_parse_response (SOUP_XMLRPC_MESSAGE (msg));
  if (!rpc_response) {
    log_debug("Could not parse XMLRPC response\n");
    g_object_unref(msg);
    g_object_unref(soup_session);
    return FALSE;
  }

  value = soup_xmlrpc_response_get_value (rpc_response);
  if (!value) {
    log_debug("No response value in XMLRPC response\n");
    g_object_unref(msg);
    g_object_unref(soup_session);
    g_object_unref(rpc_response);
    return FALSE;
  }
  
  gboolean *b;
  b=g_malloc(sizeof(gboolean));
  soup_xmlrpc_value_get_boolean(value,b);
  g_object_unref(msg);
  g_object_unref(rpc_response);

  if(*b==TRUE){
    log_debug("Login OK\n");
    g_object_unref(soup_session);
    g_free(b);
    return TRUE;
  } else {
    log_debug("Login NOT OK\n");
    g_object_unref(soup_session);
    g_free(b);
    return FALSE;
  }


}

void connection_available(){
  log_debug("connection available !\n");
  if (test_xmlrpc_login()!=TRUE){
    log_debug("xmlrpc connexion not successfull\n");
    return;
  }
  send_gps_infos();
  take_photo();
}

static void connection_event_changed(ConIcConnection *connection,
				     ConIcConnectionEvent *event,
				     gpointer user_data) {
  
  ConIcConnectionStatus status = con_ic_connection_event_get_status(event);


  

  const gchar *iap_id = con_ic_event_get_iap_id(CON_IC_EVENT(event));
  const gchar *bearer = con_ic_event_get_bearer_type(CON_IC_EVENT(event));
  gchar * debug_snt;

  debug_snt=g_strdup_printf("status=%i\n",status);
  log_debug(debug_snt);
  g_free(debug_snt);

  switch(status) {

  case CON_IC_STATUS_CONNECTED:
    proxy_mode=con_ic_connection_get_proxy_mode(connection);

    debug_snt=g_strdup_printf("Connected to IAP %s with bearer %s!\n", iap_id, bearer);
    log_debug(debug_snt);
    g_free(debug_snt);

    debug_snt=g_strdup_printf("proxy_mode=%i\n",proxy_mode);
    log_debug(debug_snt);
    g_free(debug_snt);

    if( proxy_mode==CON_IC_PROXY_MODE_MANUAL) {
      const gchar* host=con_ic_connection_get_proxy_host(connection,CON_IC_PROXY_PROTOCOL_HTTP);
      const gchar* host_ssl=con_ic_connection_get_proxy_host(connection,CON_IC_PROXY_PROTOCOL_HTTPS);

      log_debug("manual proxy mode\n");

      debug_snt=g_strdup_printf("proxy=%s\n",host);
      log_debug(debug_snt);
      g_free(debug_snt);

      debug_snt=g_strdup_printf("ssl proxy=%s\n",host_ssl);
      log_debug(debug_snt);
      g_free(debug_snt);

      http_proxy_port=con_ic_connection_get_proxy_port(connection,CON_IC_PROXY_PROTOCOL_HTTP);
      https_proxy_port=con_ic_connection_get_proxy_port(connection,CON_IC_PROXY_PROTOCOL_HTTPS);
      
      if(http_proxy_host==NULL){
	http_proxy_host=g_strdup(host);
      } else {
	g_free(http_proxy_host);
	http_proxy_host=g_strdup(host);
      }

      if(https_proxy_host==NULL){
	https_proxy_host=g_strdup(host_ssl);
      } else {
	g_free(https_proxy_host);
	https_proxy_host=g_strdup(host_ssl);
      }
      

      if(soup_proxy_uri_string==NULL){
	if(use_ssl()){
	  soup_proxy_uri_string=g_strdup_printf("http://%s:%i",https_proxy_host,https_proxy_port);
	}else{
	  soup_proxy_uri_string=g_strdup_printf("http://%s:%i",http_proxy_host,http_proxy_port);
	}
	soup_proxy_uri=soup_uri_new(soup_proxy_uri_string);
	
      }else{
	g_free(soup_proxy_uri_string);
	soup_uri_free (soup_proxy_uri);
	if(use_ssl()){
	  soup_proxy_uri_string=g_strdup_printf("http://%s:%i",https_proxy_host,https_proxy_port);
	}else{
	  soup_proxy_uri_string=g_strdup_printf("http://%s:%i",http_proxy_host,http_proxy_port);
	}
	soup_proxy_uri=soup_uri_new(soup_proxy_uri_string);
      }


    }
    if( proxy_mode==CON_IC_PROXY_MODE_AUTO) {
      log_debug("automatic proxy mode not supported\n");
      return;
    }
    
    
    connection_flag=TRUE;
    connection_available();
    break;
  case CON_IC_STATUS_DISCONNECTED:
    log_debug("We are disconnected...");
    connection_flag=FALSE;
    break;
  default:
    log_debug("Unknown connection status received");
  }


  log_debug("Network connection changed\n");
}

static void
location_changed (LocationGPSDevice *device, gpointer gps_data)
{
  static time_t last_gps_fix=0;
  time_t now;
  gchar * debug_snt;

  now=time(NULL);

  //no more than one fix per minute recorded.
  if(last_gps_fix!=0 && difftime(now,last_gps_fix)<60){
    return;
  }

  last_gps_fix=time(NULL);
  gps_info *cur_info;

  debug_snt=g_strdup_printf("Latitude: %.2f\nLongitude: %.2f\nAltitude: %.2f\n",
	   device->fix->latitude, device->fix->longitude, device->fix->altitude);
  log_debug(debug_snt);
  g_free(debug_snt);

  cur_info=g_malloc(sizeof(gps_info));
  cur_info->lat=device->fix->latitude;
  cur_info->lon=device->fix->longitude;
  cur_info->date=time(NULL);

  gps_positions=g_slist_prepend(gps_positions,(gpointer) cur_info);
}


gboolean my_timeout(gpointer loop) {
  //static double cnt=1;
  log_debug("my timeout\n");
  
  //GMainLoop *mloop=(GMainLoop *) loop;


  if(test_xmlrpc_login()==TRUE) {
    log_debug("login ok\n");
  }else{
    log_debug("login not ok\n");
  }
  log_debug("goodbye\n");
  g_main_loop_quit (loop);
  //return FALSE;
  return TRUE;
}

gboolean idle_timeout(gpointer data){
  //log_debug("nothing to do\n");
  sleep(1);
  return TRUE;
}

gboolean monitor_connection(gpointer user_data){
  log_debug("Monitoring cnx\n");
  if(connection_flag==TRUE) {
      connection_available();
  }
  return TRUE;
}

void fill_gps_positions_with_fake(){
  int i;
  for(i=0;i<5;i++){
    gchar *debug_snt=NULL;
    debug_snt=g_strdup_printf("i=%i\n",i);
    log_debug(debug_snt);
    g_free(debug_snt);
    gps_info * tmp_gps_info;
    tmp_gps_info=g_malloc(sizeof(gps_info));
    tmp_gps_info->lat=i;
    tmp_gps_info->lon=2*i;
    tmp_gps_info->date=time(NULL);
    gps_positions=g_slist_prepend(gps_positions,(gpointer) tmp_gps_info);
    sleep(1);
  }
  
}

void  device_state_changed(osso_hw_state_t *state, gpointer data){
  log_debug("device state changed \n");
  
  if (state->shutdown_ind==TRUE) {
    log_debug("Shutdown on its way\n");
    save_positions_to_file();
  }

  if (state->memory_low_ind==TRUE) {
    log_debug("Low memory\n");

  }

  if (state->save_unsaved_data_ind==TRUE) {
    log_debug("Need to save data\n");
    save_positions_to_file();
  }

}



gboolean init_gstreamer_stuff(int * argc, char **argv[]){
  //https://garage.maemo.org/svn/maemoexamples/tags/maemo_4.1/maemo-examples/example_camera.c
  log_debug("init g_streamer\n");
  gst_init(argc, argv); 
  return TRUE;
}

int main(int argc, char *argv[]) {
  GMainLoop* mainloop = NULL;
  ConIcConnection *network_connection;
  osso_context_t *osso_context;
  DBusConnection *system_dbus;
  system_dbus = dbus_bus_get(DBUS_BUS_SYSTEM, NULL);
  dbus_connection_setup_with_g_main(system_dbus, NULL);
  LocationGPSDevice *gps_device;
  GConfClient* gconf_client;
  GError* err;
  gchar * debug_snt;
  
  osso_context=osso_initialize  (  "org.maemo.phonehome",
				   "1.0",
				   TRUE,
				   NULL
				   );
  if(osso_context==NULL) {
    log_debug("Unable to create an osso context, exiting\n");
    exit(1);
  }

  //Device status change callback
  osso_hw_set_event_cb  (  osso_context,
			   NULL,
			   device_state_changed,
			   NULL 
			   ) ;

  daemonize(argc,argv);
  g_type_init();
  g_thread_init (NULL);

  gconf_init(argc, argv, &err);
  gconf_client=gconf_client_get_default ();
  xmlrpc_id=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/id",
			  NULL);
  xmlrpc_password=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/password",
			  NULL);
  xmlrpc_server=gconf_client_get_string(gconf_client, "/apps/osso/phonehome/server",
			  NULL);
  
  if(xmlrpc_server==NULL){
    xmlrpc_server=DEFAULT_XMLRPC_SERVER;
  }

  
  debug_snt=g_strdup_printf("xmlr_rpc_server=%s\n",xmlrpc_server);
  log_debug(debug_snt);
  g_free(debug_snt);
  debug_snt=g_strdup_printf("init_gstreamer_stuff:%i\n",init_gstreamer_stuff(&argc,&argv));
  log_debug(debug_snt);
  g_free(debug_snt);

  //Main loop 
  mainloop = g_main_loop_new(NULL, FALSE);
  if (mainloop == NULL) {
    /* Print error and terminate. */
    log_debug("Couldn't create GMainLoop");
    exit(EXIT_FAILURE);
  }  

  //Gconf change callback
  gconf_client_add_dir(gconf_client,
                       "/apps/osso/phonehome",
                       GCONF_CLIENT_PRELOAD_NONE,
                       NULL);

  gconf_client_notify_add(gconf_client, "/apps/osso/phonehome",
                           gconf_key_changed_callback,
                          NULL,
                          NULL, NULL);


  //Network connectivity status change callback
  network_connection = con_ic_connection_new();  
  g_signal_connect(G_OBJECT(network_connection), "connection-event", 
		   G_CALLBACK(connection_event_changed), NULL);
  g_object_set(G_OBJECT(network_connection), "automatic-connection-events", TRUE,NULL);
  load_positions_from_file();
  
  //Gps location change callback
  gps_device = g_object_new (LOCATION_TYPE_GPS_DEVICE, NULL);
  g_signal_connect (gps_device, "changed", G_CALLBACK (location_changed), (gpointer) gps_positions);

  //g_timeout_add(5000,my_timeout,(gpointer) mainloop);
  g_timeout_add(5*60*1000,monitor_connection,NULL);
  g_idle_add(idle_timeout,NULL);
  //fill_gps_positions_with_fake();
  //log_debug("coin coin\n");
  //send_gps_infos();  
  //save_positions_to_file();

  //g_slist_foreach(gps_positions, show_locations, NULL);
  //connection_available();
  //connection_available();
  //send_jpeg(NULL);

  g_main_loop_run(mainloop);
  osso_deinitialize(osso_context);

  gst_object_unref(GST_OBJECT(pipeline));
  free(system_dbus);
  exit(free_stuff_and_die());

}





