/*
 * personal-launcher - Desktop icons to launch applications
 * Copyright (C) 2008  Andrew Olmsted  <andrew.olmsted@gmail.com>
 * 
 * Parts of this file are based on:
 * desktop-plugin-example - Example Desktop Plugin with real transparency
 * Copyright (C) 2008  Tommi Saviranta  <wnd@iki.fi>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "personal-launcher.h"
#include "personal-launcher-setup.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <glib-object.h>
#include <libosso.h>

#include <libintl.h>
#include <libgnomevfs/gnome-vfs.h>

#include <hildon/hildon-banner.h>
#include <libhildondesktop/libhildondesktop.h>

#include <libhildondesktop/hildon-desktop-picture.h>

#include <X11/extensions/Xrender.h>
#include <X11/extensions/Xfixes.h>

#define HOME_DIR g_get_home_dir()
#define IMAGEFILE "/usr/share/pixmaps/plgrip.png"


HD_DEFINE_PLUGIN(PersonalLauncher, personal_launcher, HILDON_DESKTOP_TYPE_HOME_ITEM)

struct _PersonalLauncherPrivate {
	GtkWidget	*desktop;
};

static void create_ui(PersonalLauncher *plugin);

static gboolean launcher_changed(PersonalLauncher *plugin)
{
	plugin->updateTimeout=0;
	plugin->queueRefresh=TRUE;
	gtk_widget_queue_draw(GTK_WIDGET(plugin));

	return FALSE;
}

static void file_changed (GnomeVFSMonitorHandle *handle,
                                  const gchar *monitor_uri,
                                  const gchar *info_uri,
                                  GnomeVFSMonitorEventType event_type,
                                  PersonalLauncher *plugin)
{
	if (!plugin->updateTimeout)
		plugin->updateTimeout=g_timeout_add(500,(GSourceFunc)launcher_changed,plugin);
}

static void applet_destroy(GtkObject *widget)
{
	gtk_object_destroy(widget);
}

static void applet_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	GTK_WIDGET_CLASS(personal_launcher_parent_class)->size_request(
			widget, requisition);
	
}

static gint applet_expose(GtkWidget *widget, GdkEventExpose *event)
{
	PersonalLauncher *plugin = PERSONAL_LAUNCHER(widget);
	PersonalLauncherPrivate *priv = PERSONAL_LAUNCHER_GET_PRIVATE(PERSONAL_LAUNCHER(widget));
	GdkDrawable	*drawable;
	gint		x_offset, y_offset;
	XRenderColor	color;
	Picture		picture;
	Picture		grip,mask;
	gint		width,height;
	XserverRegion	region;

	if (GTK_WIDGET_DRAWABLE(widget) == FALSE) {
		return FALSE;
	}

	gdk_window_get_internal_paint_info(widget->window, &drawable,
			&x_offset, &y_offset);

	picture = hildon_desktop_picture_from_drawable(drawable);
	if (picture == None) {
		return FALSE;
	}
	
	hildon_desktop_picture_and_mask_from_file(IMAGEFILE,
			&grip,
			&mask,
			&width,
			&height);

	if ((plugin->clip.x!=event->area.x - x_offset)||(plugin->clip.y!=event->area.y - y_offset)||(plugin->clip.width!=event->area.width)||(plugin->clip.height!=event->area.height))
	{
		if (!plugin->updateTimeout)
			plugin->updateTimeout=g_timeout_add(1000,(GSourceFunc)launcher_changed,plugin);
	}
	plugin->clip.x = event->area.x - x_offset;
	plugin->clip.y = event->area.y - y_offset;
	plugin->clip.width = event->area.width;
	plugin->clip.height = event->area.height;

	region = XFixesCreateRegion(GDK_DISPLAY(), &plugin->clip, 1);

	XFixesSetPictureClipRegion(GDK_DISPLAY(), picture, 0, 0, region);

	color.red = color.blue = color.green = 0;
	color.alpha = 0;

	XRenderFillRectangle(GDK_DISPLAY(), PictOpSrc, picture, &color,
			0, 0,
			widget->allocation.width,
			widget->allocation.height);

	if (grip) {
		XRenderComposite(GDK_DISPLAY(), PictOpSrc,
				grip,
				mask,
				picture,
				0, 0,
				0, 0,
				widget->allocation.width-width, widget->allocation.height-height,
				widget->allocation.width, widget->allocation.height);
	}

	XFixesDestroyRegion(GDK_DISPLAY(), region);
	XRenderFreePicture(GDK_DISPLAY(), picture);

	if (plugin->queueRefresh)
	{
		create_ui(plugin);
		plugin->queueRefresh=FALSE;
	}

	return GTK_WIDGET_CLASS(personal_launcher_parent_class)->expose_event(
			widget, event);
}

static void applet_realize(GtkWidget *widget)
{
	GTK_WIDGET_CLASS(personal_launcher_parent_class)->realize(widget);
}

static void activation_animation(GtkWidget *widget, GdkEventButton *event, gint trans)
{
	if (trans>=0)
	{
		gint width,height;

		GdkPixbuf *orig=gtk_image_get_pixbuf(gtk_bin_get_child(GTK_BIN(widget)));
		width=gdk_pixbuf_get_width(orig);
		height=gdk_pixbuf_get_height(orig);
		GdkPixbufSimpleAnim *anim=gdk_pixbuf_simple_anim_new(width,height,30);

		gint i;
		for (i=1;i<width/4;i+=2)
		{
			GdkPixbuf *icon=gdk_pixbuf_scale_simple(orig,width-(i*2),height-(i*2),GDK_INTERP_NEAREST);
			GdkPixbuf *good=gdk_pixbuf_new(GDK_COLORSPACE_RGB,TRUE,8,width,height);
			gdk_pixbuf_fill(good,0x00000000);
			gdk_pixbuf_composite(icon,good,0,0,width,height,0,0,1,1,GDK_INTERP_NEAREST,trans);
			gdk_pixbuf_simple_anim_add_frame(anim,good);
			gdk_pixbuf_unref(icon);
			gdk_pixbuf_unref(good);
		}
		gdk_pixbuf_simple_anim_add_frame(anim,orig);
		gtk_container_foreach(GTK_CONTAINER(widget),gtk_widget_destroy,widget);
		GtkWidget *image=gtk_image_new_from_animation(GDK_PIXBUF_ANIMATION(anim));
		gtk_container_add(GTK_CONTAINER(widget),image);
		g_object_unref(G_OBJECT(anim));
		//g_object_unref(G_OBJECT(orig));
		gtk_widget_show_all(widget);
	}
}

static void execute_command(GtkWidget *item, GdkEventButton *event, gint index)
{
	pid_t pid;
	gchar *appName;
	gchar *iconName;
	gchar *executable;
	gchar *service;
	gboolean termFlag,rootFlag;
	gint argc;
	gchar** argv;
	gchar entry[5];
	gchar* filename;
	gboolean fileExists;
	GKeyFile* keyFile;
	GError *error=NULL;
	gboolean result;

	if (index>=0)
	{
		snprintf(entry,5,"%d",index);
		keyFile=g_key_file_new();
		filename=g_build_filename(HOME_DIR,"/.personal_launcher.rc",NULL);
		fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
		if (fileExists)
		{
			appName=g_key_file_get_string(keyFile,entry,"app name",NULL);
			iconName=g_key_file_get_string(keyFile,entry,"icon name",NULL);
			executable=g_key_file_get_string(keyFile,entry,"executable",NULL);
			termFlag=g_key_file_get_boolean(keyFile,entry,"run in term",NULL);
			rootFlag=g_key_file_get_boolean(keyFile,entry,"run as root",NULL);
			service=g_key_file_get_string(keyFile,entry,"service",NULL);
		}
		g_key_file_free(keyFile);

		if (rootFlag)
		{
			executable=g_strconcat("sudo ",executable,NULL);
		}

		if (termFlag)
		{
			osso_context_t *osso=osso_initialize("personal-menu-launch","0.5",TRUE,NULL);
			osso_return_t retVal;
		
			executable = g_strconcat("sh -c \"", g_strescape(executable, NULL), " ; read x\"", NULL);

			retVal=osso_rpc_async_run(osso,"com.nokia.xterm","/com/nokia/xterm","com.nokia.xterm","run_command",NULL,NULL,DBUS_TYPE_STRING,executable,DBUS_TYPE_INVALID);
			if (retVal==OSSO_OK)
			{
				char buf[64];
				g_snprintf(buf, 64, "%s started",appName);
				hildon_banner_show_information(NULL, iconName, buf);
			}
			else
			{
				hildon_banner_show_information(NULL, "qgn_note_gene_syserror", "Execution failed");
			}
			osso_deinitialize(osso);
		}
		else 
		{
			osso_context_t *osso=osso_initialize("personal-menu-launch","0.5",TRUE,NULL);
			if ((g_strcasecmp(service,"")==0) || !(result = osso_rpc_run_with_defaults(osso, service, "top_application", NULL, DBUS_TYPE_INVALID) == OSSO_OK))
			{
				if (g_shell_parse_argv(executable,&argc,&argv,NULL))
				{
					g_spawn_async(NULL,argv,NULL,(GSpawnFlags)G_SPAWN_SEARCH_PATH,NULL,NULL,&pid,&error);
					g_strfreev(argv);
				}
				if (!error)
				{
					char buf[64];
					g_snprintf(buf, 64, "%s started",appName);
					hildon_banner_show_information(NULL, iconName, buf);
					/*free(buf);*/
				}
				if (error)
				{
					hildon_banner_show_information(NULL, "qgn_note_gene_syserror", "Execution failed");
					g_error_free(error);
				}
			}
			osso_deinitialize(osso);
		}
		g_free(filename);
	}
	else
	{
		run_setup(NULL);
	}
	g_free(appName);
	g_free(iconName);
	g_free(executable);
	g_free(service);
}

static void create_ui(PersonalLauncher *plugin)
{
	GtkIconTheme *icon_theme;
	GtkWidget *image;
	GtkWidget *event;
	gchar* filename;
    gboolean fileExists;
    GKeyFile* keyFile;
    gboolean animation;
	int iconSize=64;
	int trans=255;

	if (GTK_IS_FIXED(plugin->layout))
	{
		gtk_container_foreach(GTK_CONTAINER(plugin->layout), (GtkCallback)gtk_widget_destroy, NULL);
	}
	else
	{
    	plugin->layout=gtk_fixed_new();
	}
	keyFile=g_key_file_new();
    filename=g_build_filename(HOME_DIR,"/.personal_launcher.rc",NULL);
    fileExists=g_key_file_load_from_file(keyFile,filename,G_KEY_FILE_KEEP_COMMENTS,NULL);
    if (fileExists)
    {
		int i=0;
		int x=0;
		int y=0;
		int padding=1;
		int minRows=1;
		int numCols=1;
		int entryPerColumn;
		gboolean layoutBool;
		gsize numEntry;
		GError *error=NULL;
		gchar **retVal=g_key_file_get_groups(keyFile,&numEntry);
		char entry[5];
		gboolean found;
		g_strfreev(retVal);

		numEntry--;
		iconSize=g_key_file_get_integer(keyFile,"config","icon size",&error);
		if (error)
		{
			iconSize=40;
			g_error_free(error);
			error=NULL;
		}
		trans=g_key_file_get_integer(keyFile,"config","transparency",&error);
		if (error)
		{
			trans=255;
			g_error_free(error);
			error=NULL;
		}
		animation=g_key_file_get_boolean(keyFile,"config","animation",&error);
		if (error)
		{
			animation=TRUE;
			g_error_free(error);
			error=NULL;
		}
		padding=g_key_file_get_integer(keyFile,"config","padding",&error);
		if (error)
		{
			padding=1;
			g_error_free(error);
			error=NULL;
		}
		layoutBool=g_key_file_get_boolean(keyFile,"config","auto layout",&error);
		if (error)
		{
			layoutBool=TRUE;
			g_error_free(error);
			error=NULL;
		}
		if (!layoutBool)
		{
			minRows=g_key_file_get_integer(keyFile,"config","num rows",&error);
			if (error)
			{
				minRows=1;
				g_error_free(error);
				error=NULL;
			}
			numCols=g_key_file_get_integer(keyFile,"config","num cols",&error);
			if (error)
			{
				numCols=1;
				g_error_free(error);
				error=NULL;
			}
		}
		else
		{
			minRows=plugin->clip.height/(iconSize+padding);
			numCols=plugin->clip.width/(iconSize+padding);
		}
		entryPerColumn=minRows;

		snprintf(entry,5,"%d",i);
		found=g_key_file_has_group(keyFile,entry);
		while (found)
		{
			gchar* appName;
			gchar* iconName;
			gchar* executable=NULL;

			sprintf(entry,"%d",i);
			appName=g_key_file_get_string(keyFile,entry,"app name",NULL);
			iconName=g_key_file_get_string(keyFile,entry,"icon name",NULL);
			executable=g_key_file_get_string(keyFile,entry,"executable",NULL);

			if (executable!=NULL && g_strcasecmp(executable,"")!=0)
			{
				event=gtk_event_box_new();
				gtk_event_box_set_visible_window(GTK_EVENT_BOX(event),FALSE);
				if (iconSize!=0)
				{
					icon_theme = gtk_icon_theme_get_default ();
					GdkPixbuf *icon=gtk_icon_theme_load_icon(icon_theme,iconName,iconSize,0,NULL);
					if (icon==NULL)
					{
						icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",iconSize,0,NULL);
					}
					GdkPixbuf *scale=gdk_pixbuf_scale_simple(icon,iconSize,iconSize,GDK_INTERP_NEAREST);
					GdkPixbuf *comp=gdk_pixbuf_new(GDK_COLORSPACE_RGB,TRUE,8,iconSize,iconSize);
					gdk_pixbuf_fill(comp,0x00000000);
					gdk_pixbuf_composite(scale,comp,0,0,iconSize,iconSize,0,0,1,1,GDK_INTERP_NEAREST,trans);
					image=gtk_image_new_from_pixbuf(comp);

					gdk_pixbuf_unref(icon);
					gdk_pixbuf_unref(scale);
					gdk_pixbuf_unref(comp);
					gtk_container_add(GTK_CONTAINER(event),image);
				}
				if (animation)
				{
					g_signal_connect(G_OBJECT(event),"button-press-event",G_CALLBACK(activation_animation),trans);
				}
				else
				{
					g_signal_connect(G_OBJECT(event),"button-press-event",G_CALLBACK(activation_animation),-1);
				}
				g_signal_connect(G_OBJECT(event),"button-release-event",G_CALLBACK(execute_command),i);
			}

			gtk_fixed_put(GTK_FIXED(plugin->layout),event,x*(iconSize+padding),y*(iconSize+padding));
			y++;
			if (y>entryPerColumn-1)
			{
				y=0;
				x++;
			}

			snprintf(entry,5,"%d",++i);
			found=g_key_file_has_group(keyFile,entry);
			g_free(appName);
			g_free(iconName);
			g_free(executable);
		}
	}
	else
	{
		event=gtk_event_box_new();
		gtk_event_box_set_visible_window(GTK_EVENT_BOX(event),FALSE);
		icon_theme = gtk_icon_theme_get_default ();
		GdkPixbuf *icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_cntlpane",64,0,NULL);
		if (icon==NULL)
		{
			icon=gtk_icon_theme_load_icon(icon_theme,"qgn_list_gene_default_app",64,0,NULL);
		}
		icon=gdk_pixbuf_scale_simple(icon,64,64,GDK_INTERP_NEAREST);
		gdk_pixbuf_composite(icon,icon,0,0,64,64,0,0,1,1,GDK_INTERP_NEAREST,255);
		image=gtk_image_new_from_pixbuf(icon);
		g_object_unref(G_OBJECT(icon));
		gtk_container_add(GTK_CONTAINER(event),image);

		g_signal_connect(G_OBJECT(event),"button-press-event",G_CALLBACK(activation_animation),255);

		g_signal_connect(G_OBJECT(event),"button-release-event",G_CALLBACK(execute_command),-1);

		gtk_fixed_put(GTK_FIXED(plugin->layout),event,0,0);
	}

    g_key_file_free(keyFile);

    g_free(filename);
    if (gtk_bin_get_child(GTK_BIN(plugin)))
    	gtk_widget_show_all(plugin->layout);
    else
 		gtk_container_add(GTK_CONTAINER(plugin),plugin->layout);
 		
 	gtk_widget_show_all(GTK_WIDGET(plugin));
}

static void
settings_popup(PersonalLauncher *plugin)
{
	PersonalLauncherPrivate *priv = PERSONAL_LAUNCHER_GET_PRIVATE(plugin);
	
	run_setup(priv->desktop->window);
}

GtkWidget *settings_menu_entry(HildonDesktopHomeItem *applet, GtkWidget *parent)
{
	PersonalLauncher *plugin = PERSONAL_LAUNCHER(applet);
	PersonalLauncherPrivate *priv = PERSONAL_LAUNCHER_GET_PRIVATE(PERSONAL_LAUNCHER(applet));
	GtkWidget *item;

	priv->desktop = parent;
	item = gtk_menu_item_new_with_label("Personal Launcher");

	g_signal_connect_swapped(G_OBJECT(item), "activate",
			G_CALLBACK(settings_popup), plugin);

	return item;
}

static void personal_launcher_init(PersonalLauncher *plugin)
{
	GtkSettings *settings;
	GdkColormap *rgba_colormap;
	gchar *conf_file;

	plugin->priv = G_TYPE_INSTANCE_GET_PRIVATE(plugin,
			TYPE_PERSONAL_LAUNCHER, PersonalLauncherPrivate);

	settings = gtk_settings_get_default();
	rgba_colormap = gdk_screen_get_rgba_colormap(gdk_screen_get_default());
	
	if (rgba_colormap != NULL) {
		gtk_widget_set_colormap(GTK_WIDGET(plugin), rgba_colormap);
	}
	hildon_desktop_home_item_set_resize_type(
			HILDON_DESKTOP_HOME_ITEM(plugin),
			HILDON_DESKTOP_HOME_ITEM_RESIZE_BOTH);

	plugin->queueRefresh=TRUE;

	gtk_widget_set_name(GTK_WIDGET(plugin), PACKAGE);
	
	conf_file = g_build_filename (g_get_home_dir (), ".personal_launcher.rc", NULL);
	gnome_vfs_init();
	gnome_vfs_monitor_add (&plugin->fileMon, 
                             conf_file,
                             GNOME_VFS_MONITOR_FILE,
                             (GnomeVFSMonitorCallback) file_changed,
                             plugin);
	g_free(conf_file);
}

static void personal_launcher_class_init(PersonalLauncherClass *klass)
{
	HildonDesktopHomeItemClass *applet_class;
	GtkWidgetClass *widget_class;
	GtkObjectClass *gtk_object_class;
	
	g_type_class_add_private(klass, sizeof(PersonalLauncherPrivate));
	applet_class = HILDON_DESKTOP_HOME_ITEM_CLASS(klass);
	widget_class = GTK_WIDGET_CLASS(klass);

	applet_class->settings = settings_menu_entry;

	gtk_object_class = GTK_OBJECT_CLASS(klass);
	
	gtk_object_class->destroy = applet_destroy;

	widget_class->expose_event = applet_expose;
	widget_class->realize = applet_realize;

	widget_class->size_request = applet_size_request;
}
