#ifndef PACKAGE_H
#define PACKAGE_H

/*
  MaemoRate - Rate maemo packages
  Copyright (C) 2007  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <string>

#include <Lum/Images/Image.h>

#include <Lum/Model/Table.h>

#include <Lum/Image.h>
#include <Lum/Panel.h>
#include <Lum/Text.h>

class Package
{
private:
  std::wstring name;
  std::wstring version;
  std::wstring description;
  int          score;

public:
  Package(const std::wstring& name,
          const std::wstring& version,
          const std::wstring& description,
          int score);
  ~Package();

  std::wstring GetName() const;
  std::wstring GetVersion() const;
  std::wstring GetDescription() const;
  int GetScore() const;

  void UpdateScore(int score);
};

typedef Lum::Model::StdTable<Package*>     PackageModel;
typedef Lum::Base::Reference<PackageModel> PackageModelRef;

class PackageDataProvider : public PackageModel::DataProvider
{
private:
  Lum::Images::ImageRef fullStar;
  Lum::Images::ImageRef emptyStar;
  mutable Lum::Object   *nameObject;
  mutable Lum::Object   *ratingObject;

public:
  PackageDataProvider(Lum::Images::Image* fullStar,
                      Lum::Images::Image* emptyStar)
  : fullStar(fullStar),
    emptyStar(emptyStar),
    nameObject(NULL),
    ratingObject(NULL)
  {
    // no code
  }

  ~PackageDataProvider()
  {
    delete nameObject;
    //delete ratingObject; // Makes application crash!
  }

  Lum::Object* GetObject(const PackageModel::Iterator& iter, size_t column) const
  {
    if (column==1) {
      delete nameObject;

      Lum::Panel *panel;
      Lum::Text  *text;

      panel=Lum::VPanel::Create(true,true);

      text=new Lum::Text((*iter)->GetName(),
                         Lum::OS::Font::bold,
                         Lum::Text::left);
      text->AddText(L" "+(*iter)->GetVersion());
      text->SetFlex(true,false);
      panel->Add(text);

      text=new Lum::Text((*iter)->GetDescription(),
                         Lum::OS::Font::italic,
                         Lum::Text::left,
                         Lum::OS::display->GetFont(Lum::OS::Display::fontScaleFootnote));
      text->SetFlex(true,false);
      panel->Add(text);

      nameObject=panel;

      return nameObject;
    }
    else if (column==2) {
      delete ratingObject;

      Lum::Panel *panel=new Lum::HPanel();

      if ((*iter)->GetScore()>=0) {
        for (size_t i=1; i<=5; i++) {
          if (i<=(size_t)(*iter)->GetScore()) {
            panel->Add(Lum::Image::Create(fullStar));
          }
          else {
            panel->Add(Lum::Image::Create(emptyStar));
          }
        }
      }

      ratingObject=panel;

      return ratingObject;
    }
    else {
      assert(false);
    }
  }
};

class PackageComparator : public PackageModel::Comparator
{
public:
  bool operator()(const PackageModel::ElementType& a,
                  const PackageModel::ElementType& b,
                  size_t column, bool down) const
  {
    if (down) {
      return a->GetName()<b->GetName();
    }
    else {
      return a->GetName()>b->GetName();
    }
  }
};

#endif

