/*
 * Copyright: (C) 2009 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  bruce.forsberg@gmail.com
 *
 */


#include "otrplayer.h"
#include "otrplayerfile.h"
#include "otrplayer_thread.h"
#include "id3tag.h"

#include <QtGui>
#include <Qt3Support>
#include <QSize>

#include "MyQ3ListViewItem.h"

#include "aflib/aflibConfig.h"
#include "aflib/aflibFile.h"
#include "aflib/aflibData.h"
#include "aflib/aflibFile.h"
#include "aflib/aflibAudioFile.h"
#include "aflib/aflibAudioPitch.h"
#include "aflib/aflibAudioSampleRateCvt.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <string.h>

#include <cstdio>
#include <cstdlib>
#include <iostream>

using std::list;

#define NUM_THREADS 2

#define PLAY_DELTA 1024 * 4
#define MP3_DATA_CFG "otrplayer.cfg"
#define FILE_LINE 0
#define INFO_LINE 1
#define ID3_LINE  2

#define FILE_FIELD   0
#define LENGTH_FIELD 1
#define TOTAL_FIELD  2
#define KBPS_FIELD   3
#define RATE_FIELD   4
#define DIR_FIELD    5

#ifdef MAEMO
#include <libosso.h>
#if 0
#include <osso-multimedia-interface.h>
#endif
extern osso_context_t *osso_context; 
#endif


// Constructor
otrplayer::otrplayer(QApplication *app, bool debug)
    : mainForm()
{
   bool result;

   _debug = debug;

   if (_debug)
      std::cout << "otrplayer: Entering constructor" << std::endl;

   _app = app;
   _input = NULL;
   _output = NULL;
   _pitch = NULL;
   _sample_rate = NULL;
   _stop = TRUE;
   _current_position = 0;
   _source_current_position = 0;
   _channels = 0;
   _pitch_enabled = FALSE;
   _item = NULL;
   _mount_while_play = FALSE;
   _file_was_playing_when_unmounted = FALSE;
   _sample_rate_conversion_factor = 1.0;

#ifdef MAEMO
   aboutAct = new QAction(tr("&About"), this);
   aboutAct->setStatusTip(tr("Show the application's About box"));
   result = connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));
   if (result == false)
   {
      if (_debug)
         std::cout << "Could not connect About menu" << std::endl;
   }

   helpMenu = menuBar()->addMenu(tr("&Help"));
   helpMenu->addAction(aboutAct);
#endif

   // Show sort indicator
   playList->setShowSortIndicator(true);

   if (_debug)
      std::cout << "otrplayer: creating otrplayerfile" << std::endl;
   _otr_file = new otrplayerfile();

   // Initialize Config tab 
   if (_debug)
      std::cout << "otrplayer: calling initButtons" << std::endl;
   initButtons();

   // Initialize Dir tab from config file
   if (_debug)
      std::cout << "otrplayer: calling initDirs" << std::endl;
   initDirs();

   pitchSlider->setEnabled(_pitch_enabled);
   timeSlider->setRange(0, 100);
   infoEdit->setReadOnly(TRUE);

   lookForDecoders();

   if (_debug)
      std::cout << "otrplayer: calling getFiles" << std::endl;
   getFiles();

   // Create a timer to operate as a work procedure
   if (_debug)
      std::cout << "otrplayer: creating timer" << std::endl;
   _timer = new QTimer(this);
   connect(_timer, SIGNAL(timeout()), this, SLOT(run()));
}

// Destructor
otrplayer::~otrplayer()
{
   if (_debug)
      std::cout << "otrplayer: calling destructor" << std::endl;
   delete _timer;

   if (_stop == FALSE)
      stop();

   delete _otr_file;
   delete _input;
   delete _output;
   delete _pitch;
   delete _sample_rate;
}


void
otrplayer::about()
{
   if (_debug)
      std::cout << "otrplayer: about called" << std::endl;
   QMessageBox::about(this, tr("About Otrplayer"),
      tr("<img src=\"/usr/share/icons/hicolor/scalable/hildon/otrplayer.png\" align=\"middle\" />"
      "<h1 align=\"center\"><b>Otrplayer 0.50</b></h1>"
      "<h5 align=\"center\">OTR Audio Player</h5>"
      "<h5 align=\"center\">(C) 2009 Bruce Forsberg</h5>"
      "<h5 align=\"center\">http://otrplayer.garage.maemo.org</h5>"));
}


void
otrplayer::keyPressEvent(QKeyEvent *e)
{
   bool action = false;

   if (_debug)
      std::cout << "otrplayer: Pressed key:" << std::hex << e->key() << std::endl;

   // This is the Plus key on the top of the NOKIA TABLETS
   if (e->key() == Qt::Key_F7)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as PLUS" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::CALENDAR_KEY));
   }
   // This is the Minus key of the top of the NOKIA TABLETS
   else if (e->key() == Qt::Key_F8)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as MINUS" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::ADDRESS_KEY));
   }
   // Swap Key on the Nokia Tablets
   else if (e->key() == Qt::Key_F5)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as Swap" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::MENU_KEY));
   }
   // Far right key of the SL-5500 PDA, Mail
   else if (e->key() == Qt::Key_F13)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as EMAIL" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::EMAIL_KEY));
   }
   // Escape Key on the Nokia Tablets
   else if (e->key() == Qt::Key_Escape)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as Cancel" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::HOME_KEY));
   }
   else
   {
      if (_debug)
         std::cout << "otrplayer: Key not recognized" << std::endl;
   }

   if (action == false)
      e->ignore();
   else
      e->accept();
}


bool
otrplayer::performKeyEvent(int func_event)
{
   int  value;
   bool action = true;
   bool ret_value;


   // No Selection
   if (func_event == 0)
      action = false;
   // Volume Up
   else if (func_event == 1)
      volumeChange(5);
   // Volume Down
   else if (func_event == 2)
      volumeChange(-5);
   // Fast Forward
   else if (func_event == 3)
   {
      ret_value = _otr_file->getAdvance(0, value);
      if (ret_value == FALSE)
         value = 30;
      seekSeconds(value);
   }
   // Fast Rewind
   else if (func_event == 4)
   {
      ret_value = _otr_file->getAdvance(1, value);
      if (ret_value == FALSE)
         value = -30;
      seekSeconds(value);
   }
   // Play
   else if (func_event == 5)
      play();
   // Stop
   else if (func_event == 6)
      stop();
   // Play / Stop
   else if (func_event == 7)
   {
      if (_stop == TRUE)
         play();
      else
         stop();
   }
   else
      action = false;

   return (action);
}


void
otrplayer::setup(QString& file)
{
   aflibConfig  input_config;
   aflibStatus  status;
   QString      str, secs, mins;
   QString      local_str = file.toLocal8Bit();


   if (_debug)
      std::cout << "setup: Enter" << std::endl;
   delete _input;
   delete _output;
   delete _pitch;
   delete _sample_rate;

   if (_debug)
      std::cout << "setup: create aflibAudioFile" << std::endl;
   // IF an ogg extension then use AUTO type
   if (file.endsWith(".mp3", Qt::CaseInsensitive) == true)
      _input = new aflibAudioFile(AFLIB_MPEG_TYPE, local_str.toStdString(),
               &input_config, &status);
   else
      _input = new aflibAudioFile(AFLIB_AUTO_TYPE, local_str.toStdString(),
               &input_config, &status);
   _input->setCacheEnable(FALSE);

   if (status == AFLIB_SUCCESS)
   {
       _sample_rate_conversion_factor = 44100.0 / input_config.getSamplesPerSecond();

       if (_debug)
          std::cout << "setup: create aflibAudioPitch" << std::endl;
      _pitch = new aflibAudioPitch(*_input, _sample_rate_conversion_factor);
      // IF close enough to 1.0 then only enable pitch control is user requested
      if (_sample_rate_conversion_factor < 1.0000001 && _sample_rate_conversion_factor > 0.9999999)
         _pitch->enable(_pitch_enabled);
      // ELSE we need pitch control to sample convert
      else
         _pitch->enable(TRUE);
      _pitch->setOutputSampleRate(TRUE, 44100);
      _pitch->setCacheEnable(FALSE);
      pitchCallback(pitchSlider->value());

       if (_debug)
          std::cout << "setup: create output aflibAudioFile with rate " << _pitch->getOutputConfig().getSamplesPerSecond() << std::endl;
      aflibConfig output_config(_pitch->getOutputConfig());
      _output = new aflibAudioFile(*_pitch, AFLIB_DEV_TYPE, "/dev/dsp",
            &output_config, &status);
      _output->setCacheEnable(FALSE);

      _total_size = output_config.getTotalSamples();
      _samples_per_second = output_config.getSamplesPerSecond();
      _channels = output_config.getChannels();
   }
   else
   { 
      delete _input;
      _input = NULL;
      _output = NULL;
      _pitch = NULL;
      _sample_rate = NULL;
      _stop = TRUE;
      QMessageBox::information(this, "Warning",
         "File is not currently available.\n" + file, QMessageBox::Ok);

      stop();
   }
}


void
otrplayer::getFiles()
{
   QStringList  lst, tree;
   QStringList::Iterator  it, itt, it_off;
   MyQ3ListViewItem  *item = NULL;
   int          value;
   QString      str;
   QStringList  tmp_lst;
   QString      filepath;
   int          sec, total_sec;
   static bool  madplay_found = TRUE;
   static bool  displayed_warning = FALSE;
   QStringList  file_list;
   QStringList  offset_list;
   QStringList  old_list;
   QString      dir_save;
   QString      file_save;
   bool         value_saved = FALSE;
   int          j;


   otrplayer_thread *thread[NUM_THREADS];
   for (j = 0; j < NUM_THREADS; j++)
   {
      thread[j] = new otrplayer_thread();
   }


   if (_debug)
      std::cout << "getFiles: Entering getFiles" << std::endl;

   // IF there is a currently selected item then save it.
   item = (MyQ3ListViewItem *)playList->currentItem();
   if (item != 0)
   {
      dir_save = item->text(DIR_FIELD);
      file_save= item->text(FILE_FIELD);
      value_saved = TRUE;
   }
   else
   {
      QString last_item;
      int     pos;

      _otr_file->getPlayedFile(last_item);
      if (!last_item.isEmpty())
      {
         pos = last_item.lastIndexOf("/");
         dir_save = last_item.left(pos);
         file_save = last_item.right(last_item.length() - pos - 1);
         value_saved = TRUE;
      }
   }
   if (_debug)
   {
      if (value_saved == TRUE)
      {
         std::cout << "getFiles: dir saved:" << dir_save.toStdString() << std::endl;
         std::cout << "getFiles: file saved:" << file_save.toStdString() << std::endl;
      }
   }

   if (_debug)
      std::cout << "getFiles: clear Entries" << std::endl;

   // Clear all previous entries
   file_list.clear();
   offset_list.clear();

   // Clear all entries in list GUI
   playList->clear();

   if (_debug)
      std::cout << "getFiles: get search directories" << std::endl;
   // Get directories to search
   _otr_file->getDirs(old_list, tree);

   for (itt = old_list.begin(), it = tree.begin(); itt != old_list.end(); ++itt, ++it)
   {
      if ((*it) == "Yes")
         buildDirTree((*itt), lst);
      else
      {
         QDir  dir(*itt);
         if (dir.exists())
         {
            lst.append(*itt);
         }
      }
   }

   if (_debug)
      std::cout << "getFiles: get mp3 files from each directory" << std::endl;
   QStringList filters;
   filters << "*.mp3" << "*.ogg" << "*.wav";
   // Loop for every directory tree
   for (itt = lst.begin(); itt != lst.end(); ++itt)
   {
      QDir  dir(*itt);

      dir.setNameFilters(filters);
      tmp_lst = dir.entryList();
      filepath = (*itt);
      filepath.append("/");

      for (it = tmp_lst.begin(); it != tmp_lst.end(); ++it)
         file_list.append(filepath + *it);
   }

   if (_debug)
      std::cout << "getFiles: Update list of files" << std::endl;
   // Update list of files
   _otr_file->process(file_list, offset_list);
   if (_debug)
     std::cout << "getFiles: Done Update list of files" << std::endl;

   if (_debug)
      std::cout << "getFiles: Total number of files are: " << file_list.size() << std::endl;

   bool continue_run = true;
   it = file_list.begin();
   it_off = offset_list.begin();
   QString  str1, str2, str3;
   while (continue_run)
   {
      for (j = 0; it != file_list.end() && j < NUM_THREADS; ++it, ++it_off, j++)
      {
         thread[j]->setName(*it);
         thread[j]->setOffset((*it_off).toLong());
      }
      if (j != NUM_THREADS)
         continue_run = false;
      int num_threads = j;

#if 0
      if (_debug)
      {
         QString  latin_str = (*it).toLatin1();
         std::cout << "getFiles: get info for:" << latin_str.toStdString() << std::endl;
         std::cout << "getFiles: local8bit:" << local_str.toStdString() << std::endl;
      }
#endif

      for (int jj = 0; jj < num_threads; jj++)
      {
         thread[jj]->start();
      }
      for (int jj = 0; jj < num_threads; jj++)
      {
         thread[jj]->wait();
      }

      for (int jj = 0; jj < num_threads; jj++)
      {
         QString& name = thread[jj]->getName();
         value = thread[jj]->getKBPS();

         total_sec = thread[jj]->getTotalSec();
         secondsToQString(total_sec, str1);

         sec = thread[jj]->getSeconds();
         secondsToQString(sec, str2);

         str.setNum(value);
         str3.setNum(thread[jj]->getSamplesPerSecond());

         QChar chr('/');
         int slash;
         slash = name.lastIndexOf(chr);
         item = new MyQ3ListViewItem( playList, item,
            name.right(name.length() - slash - 1),
            str2, str1, str, str3, name.left(slash));
      }
   }

   // IF madplay was not found and have not displayed warning yet
   if (madplay_found == FALSE && displayed_warning == FALSE)
   {
      displayed_warning = TRUE;
      QMessageBox::information(this, "Warning", "Madplay utility not found.\nPlease install the package for madplay.",
         QMessageBox::Ok);
   }

   // Don't allow resizing of columns. We do this after filling all fields
   // for the first time so the initial sizing is correct.
   if (_debug)
      std::cout << "getFiles: fix column sizes" << std::endl;
   playList->setColumnWidthMode(FILE_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(LENGTH_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(TOTAL_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(KBPS_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(RATE_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(DIR_FIELD, Q3ListView::Manual);

   // Set previous item or first item if no previous
   if (_debug)
      std::cout << "getFiles: set previous item" << std::endl;
   if (playList->firstChild() != 0)
      playList->setCurrentItem(playList->firstChild());
   if (value_saved == TRUE)
   {
      Q3ListViewItemIterator  t(playList);

      // Loop thru entire list
      for ( ; t.current(); t++)
      {
         MyQ3ListViewItem  *itm = (MyQ3ListViewItem *)t.current();

         if (( itm->text(FILE_FIELD) == file_save) &&
              (itm->text(DIR_FIELD) == dir_save))
         {
            playList->setCurrentItem(itm);
            break;
         }
      }
   }

   for (j = 0; j < NUM_THREADS; j++)
   {
      delete thread[j];
   }
}


void
otrplayer::play()
{
   if (_debug)
      std::cout << "play: play called" << std::endl;
   // IF playing then ignore.
   if (_stop == FALSE)
      return;

   // Call our run routine when app loop has nothing to do
   _timer->start(0);

   _item = (MyQ3ListViewItem *)playList->selectedItem();
   if (_item == NULL)
      return;

   _stop = FALSE;

   prepareNextFile();
}

void
otrplayer::stop()
{
   QString  str;
   MyQ3ListViewItem  *item;


   if (_debug)
      std::cout << "stop: stop called" << std::endl;

   _timer->stop();

   _stop = TRUE;
   delete _input;
   delete _pitch;
   delete _output;
   delete _sample_rate;
   _pitch = NULL;
   _input = NULL;
   _output = NULL;
   _sample_rate = NULL;

   item = (MyQ3ListViewItem *)playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   // Update offset of file
   if (_current_position == -1)
      _otr_file->storeOffset(str, (long)_current_position);
   else
   {
      _otr_file->storeOffset(str, (long)_source_current_position );
   }

   // IF a mount or unmount occurred while playing then refresh file display
   if (_mount_while_play == TRUE)
   {
      _mount_while_play = FALSE;
      getFiles();
   }
}

void
otrplayer::slider(int value)
{
   QString str;
   int sec;

   _slider_value = value;

   sec = _total_size * _sample_rate_conversion_factor * value / 100 /_samples_per_second;
   secondsToQString(sec, str);
   _item->setText(LENGTH_FIELD, str);
}

void
otrplayer::sliderUpdate()
{
   if (_debug)
   {
      std::cout << "Slider old _current_position " << std::dec << _current_position << std::endl;
      std::cout << "Slider old _source_current_position " << _source_current_position << std::endl;
   }
   long long delta = (_total_size / 100 * _slider_value) - _source_current_position;
   _source_current_position += delta;
   _current_position = _source_current_position;
   stop();
   play();
   if (_debug)
   {
      std::cout << "Slider delta" << delta << std::endl;
      std::cout << "Slider new _current_position " << _current_position << std::endl;
      std::cout << "Slider new _source_current_position " << _source_current_position << std::endl;
   }
}


void
otrplayer::pitchCallback(int value)
{
   if (_pitch)
      if (_pitch_enabled == TRUE)
         _pitch->setFactor( (double)value/-500.0 + _sample_rate_conversion_factor);
      else 
         _pitch->setFactor(_sample_rate_conversion_factor);
}


void
otrplayer::pitchEnable()
{
   _pitch_enabled = !_pitch_enabled;
   pitchSlider->setEnabled(_pitch_enabled);
   // IF close enough to 1.0 then only enable pitch control is user requested
   if (_pitch)
   {
      if (_sample_rate_conversion_factor < 1.0000001 && _sample_rate_conversion_factor > 0.999999)
         _pitch->enable(_pitch_enabled);
      // ELSE we need pitch control to sample convert
      else
         _pitch->enable(TRUE);
   }

   // We need to reset pitch factor is needed so we call this
   pitchCallback(pitchSlider->sliderPosition());
}


void
otrplayer::secondsToQString(
   int sec,
   QString&  str)
{
   QString      secs, mins;


   mins.setNum(sec/60);
   secs.setNum(sec%60);
   str = mins;
   if (sec%60 < 10)
      str.append(":0");
   else
      str.append(":");
   str.append(secs);
}


void
otrplayer::QStringToSeconds(
   int& sec,
   QString  str)
{
   int  min;
   QTextStream ts(&str);
   QChar   dummy;


   ts >> min >> dummy >> sec;

   sec = sec + min * 60;
}


void
otrplayer::seekSeconds(
   int sec)
{
   _source_current_position += (long long)(sec * _samples_per_second / _sample_rate_conversion_factor);
   if (_source_current_position < 0)
      _source_current_position = 0;
   _current_position = _source_current_position;

   stop();
   play();
}


void
otrplayer::buildDirTree(
   QString  str,
   QStringList&  lst)
{
   QStringList::Iterator  it;
   QStringList  dir_lst;
   QDir  dir(str);

   if (dir.exists())
   {
      lst.append(str);

      dir.setFilter(QDir::Dirs | QDir::NoSymLinks);
      dir_lst = dir.entryList();

      for (it = dir_lst.begin(); it != dir_lst.end(); ++it)
      {
         if ((*it != ".") && (*it != ".."))
            buildDirTree(dir.absolutePath() + "/" + *it, lst);
      }
   }
}


void
otrplayer::displayID3(QString& file_str)
{
   QString local_str = file_str.toLocal8Bit();

   
   if (_debug)
   {
      std::cout << "displayID3: get info for:" << file_str.toStdString() << std::endl;
      std::cout << "displayID3: local8bit:" << local_str.toStdString() << std::endl;
   }
   int fddd = open(local_str.toLatin1(), O_RDONLY);
   ID3Tag *id3 = new ID3Tag;
   if (id3->read(fddd) == TRUE)
   {
      int count = ID3_LINE;
      QString  str;

      if (!id3->title().isEmpty())
         infoEdit->insertLine("Title - " + id3->title(), count++);
      if (!id3->artist().isEmpty())
         infoEdit->insertLine("Artist - " + id3->artist(), count++);
      if (!id3->album().isEmpty())
         infoEdit->insertLine("Album - " + id3->album(), count++);
      if (id3->year() != 0)
         infoEdit->insertLine("Year - " + str.setNum(id3->year()), count++);
      if (!id3->comment().isEmpty())
         infoEdit->insertLine("Comment - " + id3->comment(), count++);
      if (id3->track() != 0)
         infoEdit->insertLine("Track - " + str.setNum(id3->track()), count++);
   }
   delete id3;
   ::close(fddd);
}


void
otrplayer::run()
{
   int             delta;
   aflibStatus     status;
   static int      old_sec = -1;
   int             sec;
   QString         str, file_str;


   // IF currently playing
   if (_stop == FALSE)
   {
      if (_current_position == -1)
         status = AFLIB_END_OF_FILE;
      else
      {
         // must reset delta since process can change it
         delta = PLAY_DELTA;

         _output->process(status, _current_position, delta);

         //std::cout << "Position: " << _current_position << " Delta: " << delta << std::endl;

         _current_position += delta;
         _source_current_position += (long long)(delta / _sample_rate_conversion_factor);

         // Update time, only every second
         if (_samples_per_second == 0)
            sec = 0;
         else
            sec = _source_current_position * _sample_rate_conversion_factor /_samples_per_second;
         if (sec != old_sec)
         {
            secondsToQString(sec, str);
            if (_total_size == 0)
               timeSlider->setValue(0);
            else
               timeSlider->setValue((int)( 100 * _source_current_position / _total_size));
            _item->setText(LENGTH_FIELD, str);
         }
         old_sec = sec;
      }

      if (status != AFLIB_SUCCESS)
      {
         file_str = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);

         // -1 means we are at end of the file
         _current_position = -1;

         // Update offset of file
         _otr_file->storeOffset(file_str, (long)_current_position);

         // Advanced to the next file
         Q3ListViewItemIterator  it(_item);
         it++;
         if (it.current() != NULL)
         {
            playList->setSelected(it.current(), TRUE);
            _item = (MyQ3ListViewItem *)playList->selectedItem();
            prepareNextFile();
         }
         else
         {
            stop();
         }
      }
   }
}


void
otrplayer::prepareNextFile()
{
   QString         info_str, str1;

   _file = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);

   setup(_file);

   infoEdit->clear();
   infoEdit->insertLine(_item->text(FILE_FIELD), FILE_LINE);

   info_str = _item->text(KBPS_FIELD);
   info_str.append(" kbps, ");
   str1.setNum(_samples_per_second);
   info_str.append(str1);
   info_str.append(" Hz, ");
   if (_channels == 1)
      info_str.append("Mono");
   else
      info_str.append("Stereo");
   infoEdit->insertLine(info_str, INFO_LINE);

   displayID3(_file);

   _current_position = _otr_file->getOffset(_file);
   _source_current_position = _current_position;

   // Store this as the last played file
   _otr_file->storePlayedFile(_file);
}


void
otrplayer::ff()
{
   MyQ3ListViewItem  *item;
   QString         str, str1, str2;
   int             secs;


   item = (MyQ3ListViewItem *)playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   QStringToSeconds(secs, item->text(TOTAL_FIELD));

   // Update offset of file
   _otr_file->storeOffset(str, (long)-1);

   if (_stop == FALSE)
   {
      str1 = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);
      if (str == str1)
         _current_position = -1;
   }

   secondsToQString(secs, str2);
   item->setText(LENGTH_FIELD, str2);
}


void
otrplayer::rev()
{
   MyQ3ListViewItem  *item;
   QString         str, str1, str2;


   item = (MyQ3ListViewItem *)playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   // Update offset of file
   _otr_file->storeOffset(str, (long)0);

   if (_stop == FALSE)
   {
      str1 = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);
      if (str == str1)
      {
         _current_position = 0;
         _source_current_position = 0;
      }
   }

   secondsToQString(0, str2);
   item->setText(LENGTH_FIELD, str2);
}


void
otrplayer::initButtons()
{
   int           menu_sel;
   int           value;
   int           font;


   menu_sel = _otr_file->getButton(otrplayerfile::CALENDAR_KEY);
   calendarBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::ADDRESS_KEY);
   addressBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::HOME_KEY);
   homeBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::MENU_KEY);
   menuBox->setCurrentItem(menu_sel);

#if 0
   menu_sel = _otr_file->getButton(otrplayerfile::EMAIL_KEY);
   emailBox->setCurrentItem(menu_sel);
#endif

   // Fast Forward advance button value
   if (_otr_file->getAdvance(0, value) == true)
   {
      ffEdit->setValue(value);
   }
   else
      ffEdit->setValue(30);

   if (_otr_file->getAdvance(1, value) == true)
   {
      rewEdit->setValue(value);
   }
   else
      rewEdit->setValue(-30);

   // Set the font for the playList
   if (_otr_file->getFont(font) == false)
     font = 1;

   // Set the button on and font
   fontGroup->setButton(font);
   fontSize(font);
}


void
otrplayer::volumeChange(int value)
{
#if 0
   static osso_rpc_t vol;

	vol.type = DBUS_TYPE_DOUBLE;
	vol.value.d = 5.0;

	osso_rpc_run(osso_context,
      OSSO_MULTIMEDIA_SERVICE, OSSO_MULTIMEDIA_OBJECT_PATH, OSSO_MULTIMEDIA_SOUND_INTERFACE,
      OSSO_MULTIMEDIA_SET_VOLUME_REQ, &vol, DBUS_TYPE_DOUBLE, NULL, DBUS_TYPE_INVALID);
#endif
}


void
otrplayer::calendarCallback(int value)
{
   _otr_file->setButton(otrplayerfile::CALENDAR_KEY, value);
}


void
otrplayer::addressCallback(int value)
{
   _otr_file->setButton(otrplayerfile::ADDRESS_KEY, value);
}


void
otrplayer::homeCallback(int value)
{
   _otr_file->setButton(otrplayerfile::HOME_KEY, value);
}


void
otrplayer::menuCallback(int value)
{
   _otr_file->setButton(otrplayerfile::MENU_KEY, value);
}

#if 0
void
otrplayer::emailCallback(int value)
{
   _otr_file->setButton(otrplayerfile::EMAIL_KEY, value);
}
#endif

void
otrplayer::initDirs()
{
   QStringList  lst;
   QStringList  tree;
   MyQ3ListViewItem  *item = NULL;
   QStringList::Iterator  it, itt;


   _otr_file->getDirs(lst, tree);
   if (lst.isEmpty())
   {
      _otr_file->setDir("/media/mmc1", "Yes");
      _otr_file->setDir("/media/mmc2", "Yes");

      _otr_file->getDirs(lst, tree);
   }

   dirView->clear();

   for (it = lst.begin(), itt = tree.begin(); it != lst.end(); it++, itt++)
   {
      item = new MyQ3ListViewItem( dirView, item, (*it), (*itt));
   }
}


void
otrplayer::setDir()
{
   if (dirEdit->text().isEmpty())
   {
      QMessageBox::information(this, "Warning", "Must enter a Directory.",
         QMessageBox::Ok);
   }
   else
   {
      if (treeBox->isChecked())
         _otr_file->setDir(dirEdit->text(), "Yes");
      else
         _otr_file->setDir(dirEdit->text(), "No");

      // Display new entry
      initDirs();

      // Stop playing file
      stop();

      // Reload all files
      getFiles();
   }
}


void
otrplayer::deleteDir()
{
   MyQ3ListViewItem   *item;


   item = (MyQ3ListViewItem *)dirView->selectedItem();
   if (item != NULL)
   {
      _otr_file->deleteDir(item->text(0));

      // Display new entry
      initDirs();

      // Stop playing file
      stop();

      // Reload all files
      getFiles();
   }
}


void
otrplayer::ffCallback(int value)
{
   _otr_file->setAdvance(0, value);
}


void
otrplayer::rewCallback(int value)
{
   _otr_file->setAdvance(1, value);
}


void
otrplayer::removeFiles()
{
   Q3ListViewItemIterator  it(playList);
   Q3ListViewItemIterator  itt(playList);
   MyQ3ListViewItem  *item;
   int   num_items = 0;
   QString  msg_str, num_str;


   // Loop thru entire list
   for ( ; it.current(); it++)
   {
      QString  str;
      long     offset;
      QDir     dir;

      item = (MyQ3ListViewItem *)it.current();
      str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

      // IF offset is -1 then remove
      offset = _otr_file->getOffset(str);
      if (offset == -1)
         num_items++;
   }

   if (num_items > 0)
   {
      msg_str = "There are " + num_str.setNum(num_items) + " files to remove\n";
      msg_str.append("Are you sure you want them removed?");

      switch( QMessageBox::warning( this, "Remove Files",
                          msg_str,
                          "Yes, Remove", "No, Do Not Remove", 0,
                          0, 1 ))
      {
         case 0: // Yes
            // Loop thru entire list
            for ( ; itt.current(); itt++)
            {
               QString  str;
               long     offset;
               QDir     dir;

               item = (MyQ3ListViewItem *)itt.current();
               str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

               // IF offset is -1 then remove
               offset = _otr_file->getOffset(str);
               if (offset == -1)
                  dir.remove(str);
            }

            // Stop playing file
            stop();

            // Reload all files
            getFiles();

         break;
         case 1: // No
         break;
      }
   }
}


void
otrplayer::fontSize(int size)
{
   QFont font(  playList->font() );
   font.setFamily( "helvetica" );

   if (size == 0)
      playList->setFont( bigButton->font() );
   else if (size == 2)
      playList->setFont( smallButton->font() );
   else 
      playList->setFont( medButton->font() );

   _otr_file->setFont(size);
}


void
otrplayer::lookForDecoders()
{
   list<string> formats;
   list<string> descriptions;
   list<string>::iterator  it;


   // Get all the found formats
   aflibFile::returnSupportedFormats(formats, descriptions);

   for (it = formats.begin(); it != formats.end(); it++)
   {
      if ( (*it) != "DEVICE") 
      {
         QString str;

         str = "Found Decoder: ";
         str.append((*it).c_str());
         infoEdit->insertLine(str, -1);
      }
   }
}

