/*
 * Copyright: (C) 2003 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */


#include <qpe/qpeapplication.h>
#include <qlistview.h>
#include <qslider.h>
#include <qmultilineedit.h>

#include "pthread.h"

#include "otrplayer_audio.h"
#include "otrplayerfile.h"
#include "id3tag.h"
#include "aflib/aflibConfig.h"
#include "aflib/aflibFile.h"
#include "aflib/aflibAudioFile.h"
#include "aflib/aflibAudioPitch.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>



#define PLAY_DELTA 512
#define FILE_LINE 0
#define INFO_LINE 1
#define ID3_LINE  2



// Constructor
otrplayer_audio::otrplayer_audio(
   QListView *lst,
   QSlider   *slider,
   QMultiLineEdit *edit,
   QApplication  *app)
{
   _input = NULL;
   _output = NULL;
   _pitch = NULL;
   _current_position = 0;
   _channels = 0;
   _app = app;
 
   _playList = lst;
   _timeSlider = slider;
   _infoEdit = edit;

   _pitch_enabled = FALSE;
   _stop = TRUE;
}

// Destructor
otrplayer_audio::~otrplayer_audio()
{
   delete  _input;
   delete  _output;
   delete  _pitch;
}


void
otrplayer_audio::setup(
QString& file,
int&       samples_per_second,
int&       channels)
{
   aflibConfig  input_config;
   aflibStatus  status;
   QString      str, secs, mins;


   delete _input;
   delete _output;
   delete _pitch;

   _input = new aflibAudioFile(AFLIB_AUTO_TYPE, file.latin1(),
               &input_config, &status);
   _input->setCacheEnable(FALSE);

   _pitch = new aflibAudioPitch(*_input, 1.0);
   _pitch->enable(_pitch_enabled);
   _pitch->setCacheEnable(FALSE);

   aflibConfig output_config(_input->getInputConfig());
   _output = new aflibAudioFile(*_pitch, AFLIB_DEV_TYPE, "/dev/dsp",
         &output_config, &status);
   _output->setCacheEnable(FALSE);

   _total_size = output_config.getTotalSamples();
   _samples_per_second = output_config.getSamplesPerSecond();
   _channels = output_config.getChannels();

   samples_per_second = _samples_per_second;
   channels = _channels;
}


void *
run(void *parm)
{
   aflibStatus     status;
   int             delta;
   QString         str, file_str, info_str, str1;
   int             sec;
   QListViewItem  *item;
   int             old_sec = -1;
   otrplayer_audio *p;
   int             samples_per_second, channels;
   QApplication   *app;

   p = (otrplayer_audio *)parm;

   app = p->getApp();

   app->lock();
   item = p->getPlayList()->selectedItem();
   QListViewItemIterator  it(item);
   app->unlock();

   if (item == NULL)
      return(NULL);

   do
   {
      app->lock();
      file_str = item->text(4);
      file_str.append("/");
      file_str.append(item->text(0));

      p->setup(file_str, samples_per_second, channels);

      p->getInfoEdit()->clear();
      p->getInfoEdit()->insertLine(item->text(0), FILE_LINE);

      info_str = item->text(1);
      info_str.append(" kbps, ");
      str1.setNum(samples_per_second);
      info_str.append(str1);
      info_str.append(" Hz, ");
      if (channels == 1)
         info_str.append("Mono");
      else
         info_str.append("Stereo");
      p->getInfoEdit()->insertLine(info_str, INFO_LINE);

      p->displayID3(file_str);

      // Get offset of file
      otrplayerfile *mp3file = new otrplayerfile;
      p->setCurrentPosition(mp3file->getOffset(file_str));
      delete mp3file;

      p->setStop(FALSE);
      status = AFLIB_SUCCESS;
      app->unlock();

      while ((status == AFLIB_SUCCESS) && (p->getStop()== FALSE))
      {
         // must reset delta since process can change it
         delta = PLAY_DELTA;

         p->getOutput()->process(status, p->getCurrentPosition(), delta);

         p->setCurrentPosition(p->getCurrentPosition() + delta);

         // Update time, only every second
         sec = p->getCurrentPosition() / p->getSamplesSecond();
         if (sec != old_sec)
         {
            app->lock();
            p->secondsToQString(sec, str);
            p->getSlider()->setValue((int)( 100 * p->getCurrentPosition()/p->getTotalSize()));
            item->setText(3, str);
            p->getApp()->processEvents();
            app->unlock();
         }
         old_sec = sec;
      }

      app->lock();
      // Update offset of file
      otrplayerfile mp3files;
      mp3files.storeOffset(file_str, (long)p->getCurrentPosition());

      // Don't advance if user pressed STOP
      if (p->getStop() == FALSE)
      {
         it++;
         if (it.current() != NULL)
         {
            p->getPlayList()->setSelected(it.current(), TRUE);
            item = p->getPlayList()->selectedItem();
         }
      }
      app->unlock();
   }
   while (it.current() != NULL && p->getStop() == FALSE);

   p->setStop(TRUE);
   return (NULL);
}


void
otrplayer_audio::play()
{
   pthread_t  thread;
   pthread_attr_t attr;


   pthread_attr_init(&attr);
   pthread_attr_setstacksize(&attr, 1024 * 128);
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

   pthread_create(&thread, &attr, (void * (*)(void *))run, this);
}


void
otrplayer_audio::setPitchEnabled(bool value)
{
   _pitch_enabled = value;
   if (_pitch)
      _pitch->enable(_pitch_enabled);
}


bool
otrplayer_audio::getPitchEnabled()
{
   return _pitch_enabled;
}


void
otrplayer_audio::pitchFactor(double val)
{
   if (_pitch)
      _pitch->setFactor(val);
}


void
otrplayer_audio::getData(
   QString&   filepath,
   long long& sams_per_sec,
   long long& total_sams,
   int&       kbps)
{
   aflibAudioFile  *temp = NULL;
   aflibConfig  input_config;
   aflibStatus  status;
   bool         ret_value;


   sams_per_sec = 0;
   total_sams = 0;

   temp = new aflibAudioFile(AFLIB_AUTO_TYPE, filepath.latin1(),
      &input_config, &status);
   if (temp)
   {
      ret_value = temp->getItem("kbps", (void *)&kbps);
      if (ret_value == FALSE)
         kbps = 0;
   }
   else
      kbps = 0;

   sams_per_sec = input_config.getSamplesPerSecond();
   total_sams = input_config.getTotalSamples();

   delete temp;
}


void
otrplayer_audio::setStop(bool stop)
{
   _stop = stop;
}


bool
otrplayer_audio::getStop()
{
   return _stop;
}

void
otrplayer_audio::seekSeconds(
   int sec)
{
   _current_position += (sec * _samples_per_second);
   if (_current_position < 0)
      _current_position = 0;
}


long long
otrplayer_audio::getTotalSize()
{
   return _total_size;
}


void
otrplayer_audio::setCurrentPosition(long long pos)
{
   _current_position = pos;
}


long long
otrplayer_audio::getCurrentPosition()
{
   return (_current_position);
}


void
otrplayer_audio::secondsToQString(
   int sec,
   QString&  str)
{
   QString      secs, mins;


   mins.setNum(sec/60);
   secs.setNum(sec%60);
   str = mins;
   if (sec%60 < 10)
      str.append(":0");
   else
      str.append(":");
   str.append(secs);
}


aflibAudioFile *
otrplayer_audio::getOutput()
{
   return (_output);
}


QListView *
otrplayer_audio::getPlayList()
{
   return (_playList);
}


QSlider *
otrplayer_audio::getSlider()
{
   return (_timeSlider);
}


long long
otrplayer_audio::getSamplesSecond()
{
   return (_samples_per_second);
}


QApplication *
otrplayer_audio::getApp()
{
   return (_app);
}


QMultiLineEdit *
otrplayer_audio::getInfoEdit()
{
   return (_infoEdit);
}


void
otrplayer_audio::displayID3(QString& file_str)
{
   int fddd = open(file_str.latin1(), O_RDONLY);
   ID3Tag *id3 = new ID3Tag;
   if (id3->read(fddd) == TRUE)
   {
      int count = ID3_LINE;
      QString  str;

      if (!id3->title().isEmpty())
         _infoEdit->insertLine("Title - " + id3->title(), count++);
      if (!id3->artist().isEmpty())
         _infoEdit->insertLine("Artist - " + id3->artist(), count++);
      if (!id3->album().isEmpty())
         _infoEdit->insertLine("Album - " + id3->album(), count++);
      if (id3->year() != 0)
         _infoEdit->insertLine("Year - " + str.setNum(id3->year()), count++);
      if (!id3->comment().isEmpty())
         _infoEdit->insertLine("Comment - " + id3->comment(), count++);
      if (id3->track() != 0)
         _infoEdit->insertLine("Track - " + str.setNum(id3->track()), count++);
   }
   delete id3;
   std::close(fddd);
}

