/*
 * Copyright: (C) 2003 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */

// class for Ogg audio file reading and writing

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif



#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "vorbis/ivorbiscodec.h"
#include "vorbis/ivorbisfile.h"

#include "aflibOggFile.h"
#include "aflibData.h"
#include "aflibFileItem.h"


#define MODULE_NAME "aflibOggFile"

extern "C"
{
   aflibFile *
   getAFileObject() { return ((aflibFile *)new aflibOggFile()); }

	void
	query(list<aflibFileItem*>& support_list)
	{
		aflibFileItem* item;
		item	= new aflibFileItem();

		item->setFormat("OGG");
		item->setDescription("Ogg Format");
		item->setName(MODULE_NAME);
        item->setExtension(".ogg");
        item->setMagic("0(O), 1(g), 2(g)");
		support_list.push_back(item);
	}
}


aflibOggFile::aflibOggFile()
{
   _vf = new OggVorbis_File;
   _fd = 0;
   _kbps = 0;
   _total_samples = 0;
   _current_sample = 0;
   _bytes_per_sample = 0;

   _size = AFLIB_SIZE_UNDEFINED;
}

aflibOggFile::~aflibOggFile()
{
   if (_fd != NULL)
      ov_clear(_vf);
   delete _vf;
}

aflibStatus
aflibOggFile::afopen(
   const char * file,
   aflibConfig* cfg)
{
   aflibStatus status = AFLIB_SUCCESS;
   aflibConfig  input_cfg;
   vorbis_info *vi;

   _file = file;

   if ((_fd = fopen(file, "r")) == NULL )
      return (AFLIB_ERROR_OPEN);

   if(ov_open(_fd, _vf, NULL, 0) < 0)
   {
      // Only close _fd if ov_open fails
      fclose (_fd);
      return (AFLIB_ERROR_OPEN);
   }

   // Store user specified results as defaults
   if (cfg != NULL)
   {
      input_cfg = *cfg;
   }

   vi = ov_info(_vf, -1);

   input_cfg.setDataOrientation(AFLIB_INTERLEAVE);
   input_cfg.setDataEndian(AFLIB_ENDIAN_LITTLE);
   input_cfg.setChannels(vi->channels);
   input_cfg.setTotalSamples(ov_pcm_total(_vf, -1));
   input_cfg.setSamplesPerSecond(vi->rate);
   input_cfg.setSampleSize(AFLIB_DATA_16S);

   _kbps = (int)ov_bitrate(_vf, -1) / 1000;
   _total_samples = ov_pcm_total(_vf, -1);
   _bytes_per_sample = 2 * vi->channels;

   ov_pcm_seek(_vf, 0);
   _current_sample = 0;

   // Set the input and output audio configuration data
   setInputConfig(input_cfg);
   setOutputConfig(input_cfg);

   return (status);
}

aflibStatus
aflibOggFile::afcreate(
   const char * file,
   const aflibConfig& config)
{
   return ( AFLIB_ERROR_UNSUPPORTED );
}


aflibStatus
aflibOggFile::afread(
   aflibData& data,
   long long position )
{
   void * p_data;
   long   new_length;
   long   partial_length;
   long   total_length;
   aflibStatus  status = AFLIB_SUCCESS;
   int    cur;


   data.setConfig(getInputConfig());
   p_data = data.getDataPointer();

   // Seek to correct position
   if ((position != -1) && (_current_sample != position))
   {
      ov_pcm_seek(_vf, position);
      _current_sample = position;
   }

   // Don't read past total samples available
   if (_total_samples < (position + data.getLength()))
   {
      new_length = _total_samples - position;
      if (new_length < 0)
         new_length = 0;
   }
   else
   {
      new_length = data.getLength();
   }

   if (new_length != 0)
   {
      partial_length = new_length = new_length * _bytes_per_sample;
      total_length = 0;
      while ((partial_length = ov_read( _vf, (char *)p_data + total_length,
          partial_length, &cur)) != 0)
      {
         total_length += partial_length;
         partial_length = new_length - total_length;
         if (partial_length == 0)
            break;
      }
      new_length = total_length / _bytes_per_sample;
   }

   _current_sample += new_length;

   // IF we reached the end of the file then return error
   if (new_length == 0)
   {
      status = AFLIB_END_OF_FILE;
      data.adjustLength(0);
   }
   // IF we read the last chunk
   else if (new_length != data.getLength())
   {
      // When we only have a partial read then readjust the length of data
      data.adjustLength(new_length);
   }

   return (status);
}

aflibStatus
aflibOggFile::afwrite(
   aflibData& data,
   long long position )
{
   return (AFLIB_ERROR_UNSUPPORTED);
}


bool
aflibOggFile::isDataSizeSupported(aflib_data_size size)
{
   // If handle not yet allocated then indicate sizes supported
   if (_fd == NULL)
      return (size == AFLIB_DATA_16S);

   return (size == _size);
}
 
bool
aflibOggFile::isEndianSupported(aflib_data_endian end)
{
   // Linux device only supports little endian
   return (end == AFLIB_ENDIAN_LITTLE);
}

bool
aflibOggFile::isSampleRateSupported(int& rate)
{
   int value;
   bool ret_value = FALSE;


   // If handle not yet allocated then indicate any sample rate is supported
   if (_fd == NULL)
      return (TRUE);

   // Get the rate of the data
   value = getOutputConfig().getSamplesPerSecond();

   // IF same rate then TRUE else return desired rate
   if (rate == value)
      ret_value = TRUE;
   else
      rate = value;

   return (ret_value);
}


bool
aflibOggFile::getItem(const char* item, void* value)
{
    bool status = FALSE;

    if (!strcmp(item, "kbps"))
    {
       *((int *)value) = _kbps;
       status = TRUE;
    }

    return status;
}

