/*
 * Copyright: (C) 2003 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */


// class for MP3 audio file reading 

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string>
#include <iostream>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>


#include "aflibMadFile.h"
#include "aflibData.h"

#include "aflibFileItem.h"
	
#define MODULE_NAME "aflibMadFile"

#define TEMP_DIR			"/tmp/"

extern "C"
{
   aflibFile *
   getAFileObject() { return ((aflibFile *)new aflibMadFile()); }

	void
	query(list<aflibFileItem*>& support_list)
	{

		aflibFileItem* item;
		item	= new aflibFileItem();

		item->setFormat("MPEG");
		item->setDescription("madplay Wrapper");
		item->setExtension(".mpg");
		item->setExtension(".mp3");
		item->setExtension(".mp2");
		item->setName(MODULE_NAME);
		item->setMagic("0(R), 1(I), 2(F), 3(F), 8(W), 9(A), 10(V), 11(E), -1(d), 1(a), 2(t), 3(a), -2(a), 8()");
		item->setMagic("0(I), 1(D), 2(3)");
		item->setMagic("0()");
		support_list.push_back(item);
	
	}
} 


aflibMadFile::aflibMadFile():
_fd(0),
_sample_rate(0),
_channels(0),
_dual_channel(FALSE),
_current_sample(0),
_mpeg_cmd("madplay -q --output=raw:- "),
_kbps(0)
{
}

aflibMadFile::~aflibMadFile()
{
   // Should be moved to a close function
	if(_fd != NULL)
       pclose(_fd);
}

aflibStatus
aflibMadFile::afopen(
   const char * file,
   aflibConfig* cfg)
{
   // This function will open an existing MP3 file.

   string cmd_str, temppath;
   aflibConfig  input_cfg(*cfg);
   char  buf[10240];
   char temp[] = "mpeg-XXXXXX";
   long info_size = 0;
   aflibData  data(1);
   char    *temp_buf;
   struct stat  stat_buf;
   long long    samples;
   int  status_system;


   if ( mktemp(temp) == NULL ) return (AFLIB_ERROR_OPEN);
   temppath = TEMP_DIR; 
   temppath += temp;	
   cmd_str = "madplay";
   cmd_str += " -v --time=00:00:00.001 --output=raw:";
   cmd_str += "/dev/null \"";
   cmd_str += file;
   cmd_str += "\" 2> ";
   cmd_str += temppath;

   status_system = system(cmd_str.c_str());
   // 32512 is 127 shifted left 8 places
   // This means madplay was not found
   if (status_system == -1 || status_system == 32512)
      return (AFLIB_NOT_FOUND);

   if ((_fd = fopen(temppath.c_str(), "r")) == NULL )
      return (AFLIB_ERROR_OPEN);

   /* Read info from temp file */
   info_size = fread(buf,1,10240,_fd);	
   fclose(_fd);
   _fd = NULL;
   unlink(temppath.c_str());

   buf[info_size] = 0;

#if 0
   // This strings means that madplay can't decode this file
   if (strstr(buf, "error: frame 0: lost synchronization"))
      return (AFLIB_ERROR_OPEN);
#endif

   _dual_channel = FALSE;
   if (strstr(buf, ", single channel,"))
      _channels = 1;
   else if (strstr(buf, ", dual channel,"))
   {
      _dual_channel = TRUE;
      _channels = 2;
   }
   else
      _channels = 2;
   input_cfg.setChannels(_channels);

   temp_buf = strstr(buf, "Layer");
   if (temp_buf)
   {
      temp_buf = strstr(temp_buf, ",");
      if (temp_buf)
      {
         temp_buf++;
         sscanf(temp_buf, "%d", &_kbps);
      }
   }
  
   temp_buf = strstr(buf, "kbps,");
   if (temp_buf)
   {
      temp_buf += strlen("kbps,");
      sscanf(temp_buf, "%d", &_sample_rate);
   }
   else
   {
      _sample_rate = 44100;
   }
 
   input_cfg.setSamplesPerSecond(_sample_rate);

   input_cfg.setSampleSize(AFLIB_DATA_16S);
   input_cfg.setDataOrientation(AFLIB_INTERLEAVE);
   input_cfg.setDataEndian(data.getHostEndian());

   if (_kbps != 0)
   {
      stat(file, &stat_buf);
      samples = (long long)((double)stat_buf.st_size / _kbps * 8.0 /
                 1000.0 * _sample_rate);
      input_cfg.setTotalSamples(samples);
   }
   else
      input_cfg.setTotalSamples(0);

   // Set input and output audio configuration data
   setInputConfig(input_cfg);
   setOutputConfig(input_cfg);
   _filename = file;

   return (AFLIB_SUCCESS);
}

aflibStatus
aflibMadFile::afcreate(
   const char * file,
   const aflibConfig& cfg)
{
   return( AFLIB_ERROR_UNSUPPORTED );
}

aflibStatus
aflibMadFile::afread(
   aflibData& data,
   long long position )
{

   void * p_data;
   long   total_length;
   long   new_length = 0;
   aflibStatus  status = AFLIB_SUCCESS;
   bool   data_seek = FALSE;
   int    seconds;
   int    hours;
   int    minutes;
   char   time_str[100];


   data.setConfig(getInputConfig());
   total_length = data.getTotalAdjustLength();
   p_data = data.getDataPointer();

   // Close pipe and resync if necessary
   if ((position != -1) && (_current_sample != position))
   {
      if (_fd)
      {
         pclose(_fd);
         _fd = NULL;
      }
      data_seek = TRUE;
   }

   if(_fd == NULL)
   {
   	  string cmd_str = _mpeg_cmd;
      // IF dual channel output both channels
      if (_dual_channel)
         cmd_str += "-S ";

      if (data_seek)
      {
         seconds = position / _sample_rate;
         hours = seconds / 3600;
         seconds -= (hours * 3600);
         minutes = seconds / 60;
         seconds -= (minutes * 60);
         sprintf(time_str, "%2.2d:%2.2d:%2.2d ", hours, minutes, seconds);
         cmd_str += "--start=";
         cmd_str += time_str;
         _current_sample = position;
      }
      else
         _current_sample = 0;

      cmd_str += "\"";
      cmd_str += _filename;
      cmd_str += "\" 2>/dev/null";

      if((_fd = popen(cmd_str.c_str(),"r")) == NULL )
         return (AFLIB_ERROR_OPEN);
   }

   new_length = fread(p_data, 1,total_length, _fd);
   _current_sample += (new_length / 2 / _channels);

   if (new_length != total_length)
   {
      data.adjustLength(new_length);
      if (new_length == 0)
   	  {
         if( _fd )
            pclose( _fd ); 
         _fd = NULL; 
         _current_sample = 0;
         status = AFLIB_END_OF_FILE;
      }
   }

   return(status);
}

aflibStatus
aflibMadFile::afwrite(
   aflibData& data,
   long long position )
{
   return( AFLIB_ERROR_UNSUPPORTED );
}

bool
aflibMadFile::isDataSizeSupported(aflib_data_size size)
{
   return (size == AFLIB_DATA_16S);
}

bool
aflibMadFile::isEndianSupported(aflib_data_endian end)
{
   return (end == AFLIB_ENDIAN_LITTLE);
}

bool
aflibMadFile::isSampleRateSupported(int& rate)
{
   bool ret_value = FALSE;
 
 
   // If not yet allocated then 
   if (_sample_rate == 0)
      return (rate == 22050 || rate == 44100 || rate == 16000);

   // IF same rate then TRUE else return desired rate
   if (rate == _sample_rate)
      ret_value = TRUE;
   else
      rate = _sample_rate;
 
   return (ret_value);
}

bool
aflibMadFile::getItem(const char* item, void* value)
{
    bool status = FALSE;

    if (!strcmp(item, "kbps"))
    {
       *((int *)value) = _kbps;
       status = TRUE;
    }

	return status;
}
