/*
 * Copyright: (C) 1999-2001 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */

// class for Device audio file reading and writing

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <esd.h>

#include "aflibDevFile.h"
#include "aflibData.h"
#include "aflibFileItem.h"

#define OPEN_BUFFER     0.5
#define CREATE_BUFFER   0.2

#define MODULE_NAME "aflibDevFile"

extern "C"
{
   aflibFile *
   getAFileObject() { return ((aflibFile *)new aflibDevFile()); }

   void
   query(list<aflibFileItem*>& support_list)
   {
      aflibFileItem* item;
      item	= new aflibFileItem();

      item->setFormat("DEVICE");
      item->setDescription("OSS Device");
      item->setName(MODULE_NAME);
      support_list.push_back(item);
   }
}


aflibDevFile::aflibDevFile()
{
   _fd_int = -1;
   _size = AFLIB_SIZE_UNDEFINED;
}

aflibDevFile::~aflibDevFile()
{
   if (_fd_int != -1)
   {
      close(_fd_int);
   }
}

aflibStatus
aflibDevFile::afopen(
   const char * file,
   aflibConfig* cfg)
{
   aflibStatus status = AFLIB_SUCCESS;
   aflibConfig  input_cfg;

   _create_mode = FALSE;
   _file = file;
   int snd_format;
   esd_format_t format = ESD_STREAM | ESD_PLAY;


   // Store defaults if user has specified
   if (cfg != NULL)
   {
      input_cfg = *cfg;
   }

   if (cfg != NULL)
   {
      if (cfg->getBitsPerSample() == 8)
      {
         snd_format = 8;
      }
      else if (cfg->getBitsPerSample() == 16)
      {
         snd_format = 16;
      }
      else
      {
         // Some unknown setting
         snd_format = 8;
      }
   }
   else
   {
      snd_format = 16;
   }
   input_cfg.setDataOrientation(AFLIB_INTERLEAVE);
   input_cfg.setDataEndian(AFLIB_ENDIAN_LITTLE);

   if (snd_format == 16)
   {
      input_cfg.setSampleSize(AFLIB_DATA_16S);
      _size = AFLIB_DATA_16S;
   }
   else
   {
      input_cfg.setSampleSize(AFLIB_DATA_8U);
      _size = AFLIB_DATA_8U;
   }

   if (cfg != NULL)
   {
      input_cfg.setChannels(cfg->getChannels());
   }
   else
   {
      input_cfg.setChannels(1);
   }

   _snd_stereo = input_cfg.getChannels() - 1;

   if (cfg != NULL)
   {
      _snd_speed = cfg->getSamplesPerSecond();
   }
   else
   {
      _snd_speed = 44100;
   }
   input_cfg.setSamplesPerSecond(_snd_speed);

   // Set the input and output audio configuration data
   setInputConfig(input_cfg);
   setOutputConfig(input_cfg);

   return (status);
}

aflibStatus
aflibDevFile::afcreate(
   const char * file,
   const aflibConfig& config)
{
   aflibStatus status = AFLIB_SUCCESS;
   aflibConfig output_cfg(config);
   esd_server_info_t   *esd_info;

#if 0
   _fd_int = esd_open_sound(NULL);
   esd_info = esd_get_server_info(_fd_int);
   if (esd_info)
   {
      if (getenv("AFLIB_DEBUG"))
      {
         std::cout << "Format is: " << esd_info->format << std::endl;
         std::cout << "Sample Rate of esd server is: " << esd_info->rate << std::endl;
      }
      esd_free_server_info(esd_info);
   }
#endif


   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate called" << std::endl;

   _create_mode = TRUE;
   _file = file;
   esd_format_t format = ESD_STREAM | ESD_PLAY;

   if ((config.getBitsPerSample() == 16) && (config.getDataOrientation() == AFLIB_INTERLEAVE))
   {
      _size = AFLIB_DATA_16S;
      format |= ESD_BITS16;
   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate ESD_BITS16 selected" << std::endl;
   }
   else if (config.getBitsPerSample() == 8)
   {
      _size = AFLIB_DATA_8U;
      format |= ESD_BITS8;
   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate ESD_BITS8 selected" << std::endl;
   }
   else
   {
      std::cerr << "Unsupported sample format" << std::endl;
      // close (_fd_int);
      _fd_int = -1;
      return (AFLIB_ERROR_UNSUPPORTED);
   }

   output_cfg.setSampleSize(_size);

   _snd_stereo = config.getChannels() - 1;
   if (_snd_stereo == 0)
   {
      format |= ESD_MONO;
   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate ESD_MONO selected" << std::endl;
   }
   else
   {
      format |= ESD_STEREO;
   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate ESD_STEREO selected" << std::endl;
   }

   _snd_speed = config.getSamplesPerSecond();   
   if (getenv("AFLIB_DEBUG"))
      std::cout << "afcreate samples per second of " << _snd_speed << std::endl;

   // Set input and output audio configuration
   setInputConfig(config);
   setOutputConfig(output_cfg);

   _fd_int= esd_play_stream(format, _snd_speed, NULL, "otrplayer");
   if (_fd_int <= 0)
   {
      std::cout << "Unable to open device" << std::endl;
      return (AFLIB_ERROR_OPEN);
   }


   return(status);
}


aflibStatus
aflibDevFile::afread(
   aflibData& data,
   long long position )
{
   void * p_data;
   long   total_length;
   aflibStatus  status = AFLIB_SUCCESS;

   data.setConfig(getInputConfig());
   total_length = data.getTotalLength();
   p_data = data.getDataPointer();

   read(_fd_int, p_data, total_length);

   return (status);
}

aflibStatus
aflibDevFile::afwrite(
   aflibData& data,
   long long position )
{
   void * p_data;
   long   total_length;
   aflibStatus  status = AFLIB_SUCCESS;


   if (data.getLength() == data.getOrigLength())
   {
      total_length = data.getTotalLength();
   }
   else
   {
      total_length = (long)(data.getTotalLength() *
         ((double)data.getLength() / (double)data.getOrigLength()));
   }

   p_data = data.getDataPointer();

   if (write(_fd_int, p_data, total_length) == -1)
   {
      perror("");
      std::cerr << "Failed to write to audio device " << std::endl;
   }

   return (status);
}


/*! \brief Processes unique information for the Linux sound device.

   This will process any unique information for the Linux audio device. It currently
   supports one item. It is AFLIB_DEV_ITEM_BUFFER. It will expect a value that is of
   type double. This will create a buffer in the hardware, if it is supported, of
   the length specified in seconds. 
*/
bool
aflibDevFile::setItem(
   const char * item,
   const void * value)
{
   bool state = FALSE;

   // IF user wants to set buffer length
   if (strcmp(item, AFLIB_DEV_ITEM_BUFFER) == 0)
   {
      state = TRUE;
   }

   return (state);
}


bool
aflibDevFile::isDataSizeSupported(aflib_data_size size)
{
   // If handle not yet allocated then indicate sizes supported
   if (_fd_int == -1)
      return (size == AFLIB_DATA_8U || size == AFLIB_DATA_16S);

   return (size == _size);
}
 
bool
aflibDevFile::isEndianSupported(aflib_data_endian end)
{
   // Linux device only supports little endian
   return (end == AFLIB_ENDIAN_LITTLE);
}

bool
aflibDevFile::isSampleRateSupported(int& rate)
{
   int value;
   bool ret_value = FALSE;


   // If handle not yet allocated then indicate any sample rate is supported
   if (_fd_int == -1)
      return (TRUE);

   // Get the rate of the data
   value = getOutputConfig().getSamplesPerSecond();

   // IF same rate then TRUE else return desired rate
   if (rate == value)
      ret_value = TRUE;
   else
      rate = value;

   return (ret_value);
}

