/*
 * Copyright: (C) 1999-2001 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "aflibAudioSampleRateCvt.h"
#include "aflibConverter.h"
#include "aflibData.h"


/*! \brief Basic constructor to place object in a chain

    Different sampling methods can be employed. By setting linear_interpolation to
    TRUE you are selecting the fastest and least quality method of sampling. With this
    set to TRUE the high_quality and filter_interpolation have no effect. This method
    just does a simple interpolation process when resampling occurs. Thus alias
    distoration can occur with this mode. To use high quality resampling the
    linear_interpolation flag should be set to FALSE.
    With linear_interpolation set to FALSE then the high_quality flag can be set. If
    it is set to FALSE then a small filter will be used. If TRUE a large filter will
    be used. The large filter is the best quality and also is the most CPU intensive.
    For most applications the small filter should be more than adequate. With the small
    and large filter a third parameter can be set, filter_interpolation. With
    filter_interpolation set then the filter coefficients used for both the small and
    large filtering will be interpolated as well.
    Defaults for the 3 filter parameters are set in the constructor which will create
    a small filter. This should be sufficient for most cases.
    For a description of factor see the setFactor member function description.
*/
aflibAudioSampleRateCvt::aflibAudioSampleRateCvt(
   aflibAudio& audio,
   double factor,
   bool  linear_interpolation,
   bool  high_quality,
   bool  filter_interpolation) : aflibAudio(audio)
{
   _output_samples = 0;
   _initial = TRUE;
   _next_output_position = 0;
   _next_input_position = 0;
   _save_samples = 0;
   _init_chan = 0;
   _in_array = NULL;
   _in_array_size = 0;
   _out_array = NULL;
   _out_array_size = 0;
   _prev_in_count = 0;
   _vol = 1.0;

   _converter = new aflibConverter(high_quality, linear_interpolation,
      filter_interpolation);

   setFactor(factor);

   // For some factor rates we need extra data at the end to extrapolate
   if (_factor <= 1.0)
      _extra_sample = 50;
   else
      _extra_sample = (int)_factor + 50;

   // By default enable caching in base class
   setCacheEnable(TRUE);
}


/*! \brief Destructor
*/
aflibAudioSampleRateCvt::~aflibAudioSampleRateCvt()
{
   delete _converter;
   delete [] _in_array;
   delete [] _out_array;
}

/*! \brief Sets the sample rate conversion factor.

     This will set the sample rate conversion factor. Virtually any value, within
     reason, can be set. For instance to convert 22050 to 44100 a factor of 0.5
     should be used. To convert 44100 to 22050 a factor of 2.0 should be used. This
     will change the factor that was set in the constructor. This function can also
     be used to change the final amplitude. This only applies to the small and large
     filters and not the linear mode.
*/
void
aflibAudioSampleRateCvt::setFactor(
   double factor,
   double volume)
{
   _factor = factor;
   _vol = volume;

   // calculate the output configuration now that we have a new factor
   const aflibConfig config = getInputConfig();
   setInputConfig(config);

   _initial = TRUE;
}


/*! \brief Sets the input and output audio data configuration of this object.

   This function overrides the aflibAudio base class function. It will change
   the output rate sample rate in the output audio configuration.
*/
void
aflibAudioSampleRateCvt::setInputConfig(const aflibConfig& cfg)
{
   aflibConfig config = cfg;

   config.setSamplesPerSecond((int)(cfg.getSamplesPerSecond() * _factor + 0.5));
   config.setTotalSamples((long long)(cfg.getTotalSamples() * _factor + 0.5));

   // call the base classes setInputConfig to store input configuration
   aflibAudio::setInputConfig(cfg);
   // Store output configuration
   setOutputConfig(config);
}


/*! \brief Performs the sample rate conversion of a segment of data.

   This is the function that does all the work. It does many things. It takes an
   input buffer of data from its parent and adds it to the end of any previously
   unused data from the previous pass. This is done so that data can be streamed.
   The new data is sent to the converter. The convertered data is then saved to the
   output buffer to be returned to the next object. Any unused data is saved for the
   next pass. NOTE: one can disable the caching in the aflibMemCache base class,
   which is enabled by default for this class, and instead use a simple caching
   implemented in this function. 
*/
aflibStatus
aflibAudioSampleRateCvt::compute_segment(
   list<aflibData *>& data,
   long long position) 
{
   long temp;
   int out_count, in_count, old_in_count, nChans;
   long  i;
   int   c, counter;
   aflibConfig newConfig((*data.begin())->getConfig());


   nChans   = (*data.begin())->getConfig().getChannels();
   (*(data.begin()))->getLength(temp);
   in_count = (int)temp;
   old_in_count = (int)temp;

   // If we need to re initialize the rate converter then do so
   if ((_initial == TRUE) || (_init_chan != nChans))
   {
      _init_chan = nChans;
      _converter->initialize(_factor, nChans, _vol);
   }

   // This is set in process. This may have to change in future because this assumes
   // this func gets called after process with data object in that order.
   out_count  = _output_samples;

   // We will reuse the in_array if at all possible
   if ((_in_array) && ((nChans * (in_count + _save_samples)) <= _in_array_size))
   {
      // Reuse previously allocated memory
   }
   else
   {
      // Free old memory and allocate new memory
      _in_array_size = nChans * (in_count + _save_samples);
      delete [] _in_array;
      _in_array = new short[_in_array_size];
   }

   // We will reuse the out_array if at all possible
   if ((_out_array) && (nChans * out_count <= _out_array_size))
   {
      // Reuse previously allocated memory
   }
   else
   {
      // Free old memory and allocate new memory
      _out_array_size = nChans * out_count;
      delete [] _out_array;
      _out_array = new short[_out_array_size];
   }

   // If in_array has different alignment we may need to shift some data
   if ((_save_samples != 0) && (_prev_in_count != (in_count + _save_samples)))
   {
      // IF current array is smaller than previous one saved
      if (_prev_in_count > in_count + _save_samples)
      {
         for (c = 1; c < nChans; c++)
         {
            for (i = 0; i < _save_samples; i++)
            {
               _in_array[c * (in_count + _save_samples) + i] =
                  _in_array[c * _prev_in_count + i];
            }
         }
      }
      // ELSE current array is larger
      else
      {
         for (c = nChans-1; c >= 1; c--)
         {
            for (i = _save_samples-1; i >= 0; i--)
            {
               _in_array[c * (in_count + _save_samples) + i] =
                  _in_array[c * _prev_in_count + i];
            }
         }
      }
   }

   for (c = 0; c < nChans; c++)
   {
      for (i = 0; i < in_count; i++)
      {
         _in_array[c * (in_count + _save_samples) + _save_samples + i] =
            (short)(*data.begin())->getSample(i, c);
      }
   }

   newConfig.setSamplesPerSecond((int)(newConfig.getSamplesPerSecond() * _factor + 0.5));
   newConfig.setTotalSamples((long long)(newConfig.getTotalSamples() * _factor + 0.5));
   aflibData * newData = new aflibData(newConfig, out_count);

   // Since in_count is returned we need to add _save_samples
   in_count += _save_samples;
   // Perform the conversion
   _converter->resample(in_count, out_count, _in_array, _out_array);

   // Use in_count returned from above to calculate next position
   _next_input_position =  position - _save_samples + in_count;

   // Store output data into a new data structure
   counter = 0;
   for (c = 0; c < nChans; c++)
   {
      for (i = 0; i < newData->getLength(); i++)
      {
         newData->setSample((int)(_out_array[counter++]), i, c);
      }
   }

   // IF data caching is enabled then don't cache in this class
   if (getCacheEnable() == TRUE)
   {
      _save_samples = 0;
   }
   else
   {
      // Save extra samples in buffer to use next pass
      _save_samples = old_in_count + _save_samples - in_count;
   }
   for (c = 0; c < nChans; c++)
   {
      for (i = 0; i < _save_samples; i++)
      {
         _in_array[i + c * (in_count + _save_samples)] =
            _in_array[i + c * (in_count + _save_samples) + in_count];
      }
   }
   _prev_in_count = in_count + _save_samples;


   *(*data.begin()) = *newData;
   delete newData;

   _initial = FALSE;

   return (AFLIB_SUCCESS);
}


/*! \brief Performs some needed preprocessing before base class function is called.

   This function will take the position and number of samples requested and
   based on several factors will determine what position and number of samples
   is required from its parent. It will also try as best as possible to keep 
   the data streaming. This is so it can read from a device or a format that
   does not have random access. When this function is done it will call the
   aflibAudio base classes function of the same name.
*/
aflibData *
aflibAudioSampleRateCvt::process(
   aflibStatus& ret_status,
   long long position,
   int& num_samples,
   bool free_memory) 
{
   long long save_position = position;
   int       save_samples  = num_samples;
   static bool  prev_enable = FALSE;


   _output_samples = num_samples;

   if (getEnable() == TRUE)
   {
      // Change the number of samples we need based on sampling factor
      num_samples = (int)ceil(num_samples / _factor) + _extra_sample;
      if ((position > 0) && (prev_enable != FALSE))
      {
         // If next data block in a sequence
         if (position == _next_output_position)
         {
            // Read input data starting right where we left off plus ignore left over data
            position = _next_input_position + _save_samples;

            // We will use _save_samples from previous run thus we don't need them
            num_samples -= _save_samples;
         }
         else
         {
            position = (long long)floor(position / _factor);

            // Since not in a sequence then throw data away
            _save_samples = 0;
         }
      }
      else
      {
         // Since not in a sequence then throw data away
         _save_samples = 0;
      }
      prev_enable = TRUE;
   }
   else
   {
      prev_enable = FALSE;
   }
   
   _next_output_position = save_position + save_samples;

   return (aflibAudio::process(ret_status, position, num_samples, free_memory));
}


/*! \brief Inform base class that this object only supports 16bit signed data.

   This overrides the virtual function in the base class. This algorithm works
   only on 16 bit signed data. Base class must convert the data.
*/
bool
aflibAudioSampleRateCvt::isDataSizeSupported(aflib_data_size size)
{
   bool state = FALSE;

   if (size == AFLIB_DATA_16S)
      state = TRUE;

   return (state);
}


 

