/*
 * Copyright: (C) 2000-2001 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif



#include "aflibAudioMixer.h"
#include "aflibData.h"


/*! \brief Constructor that requires a parent.
*/
aflibAudioMixer::aflibAudioMixer(
   aflibAudio& audio) : aflibAudio(audio)
{
}


/*! \brief Constructor that does not require a parent.
*/
aflibAudioMixer::aflibAudioMixer() : aflibAudio()
{
}


/*! \brief Destructor.
*/
aflibAudioMixer::~aflibAudioMixer()
{
   //delAllMix();
}


/*! \brief Adds a mix element.

    This will add an element to be mixed. Any input specified must exist. The first
    input is number 1. The amplitude can be specified as any value between 1 and 100.
    This corresponds to 1 and 100 percent of full scale. The in_chan is the channel
    number of the input to use. The out_chan is the same but indicates the output 
    channel to use. The first channel for inputs and outputs start at 0. For example
    to mix a stereo input to a mono output one would call addMix twice with
    (1, 0, 0, 50) & (1, 1, 0, 50). This would take channels 0 and 1
    of input 1, divide each by two, and add both together and output as channel 0. If
    inputs differ in their configuration then they will be configure to the greater
    of eachs configs. If the mix was added successfully then AFLIB_SUCCESS will be
    returned. A mix with same input, in_chan, out_chan values will replace the old
    values.
*/
aflibStatus
aflibAudioMixer::addMix(
   int  input,
   int  in_chan,
   int  out_chan,
   int  amp)
{
   aflibStatus status = AFLIB_SUCCESS;

   // need to check if in list. If so remove.
   delMix(input, in_chan, out_chan);

   aflibMixerItem item(input, in_chan, out_chan, amp);
   _mix_item.insert(item);

   // Invalidate chain since it has changed
   setNodeProcessed(FALSE);

   return (status);
}


/*! \brief Deletes a mix element.

    This will delele a current mix element. If successful then AFLIB_SUCCESS will be
    returned else if the item was not found then AFLIB_NOT_FOUND will be returned. 
*/
aflibStatus
aflibAudioMixer::delMix(
   int  input,
   int  in_chan,
   int  out_chan)
{
   aflibStatus status = AFLIB_NOT_FOUND;
   set<aflibMixerItem, less < aflibMixerItem> >::iterator it;

   // Loop thru all items
   for (it = _mix_item.begin(); it != _mix_item.end(); it++)
   {
      // Look for the item to delete
      if ((input == (*it).getInput()) &&
          (in_chan == (*it).getInChannel()) &&
          (out_chan == (*it).getOutChannel()))
      {
         _mix_item.erase(it);
         status = AFLIB_SUCCESS;
         break;
      }
   }

   // Invalidate chain since it has changed
   setNodeProcessed(FALSE);

   return (status);
}


/*! \brief Deletes all mix elements.
 
    This will delele all current mix elements.
*/
void
aflibAudioMixer::delAllMix()
{
   // Erase all mix items
   if (_mix_item.size() != 0)
   {
      _mix_item.erase( _mix_item.begin(), _mix_item.end());
   }                                                                                        
}


/*! \brief Returns the current number of active mix elements. 
*/
int
aflibAudioMixer::getNumOfMixs()
{
   return (_mix_item.size());                                                              
}


/*! \brief Returns information on a specific mix element.

    This will retrieve the current values for a specific mix element. One should call
    getNumOfMixes first. Mix numbers start at 1.
*/ 
aflibStatus
aflibAudioMixer::getMix(
   int   mix_num,
   int&  input,
   int&  in_chan,
   int&  out_chan,
   int&  amp)
{
   aflibStatus status = AFLIB_NOT_FOUND;
   set<aflibMixerItem, less < aflibMixerItem> >::iterator it;
   int j;
 
   input = 0;
   in_chan = 0;
   out_chan= 0;
   amp = 0;
 
   if (mix_num <= (int)_mix_item.size())
   {
      for (it = _mix_item.begin(), j = 1; it != _mix_item.end(); it++, j++)
      {
         if (j == mix_num)
         {
            input = (*it).getInput();
            in_chan = (*it).getInChannel();
            out_chan= (*it).getOutChannel();
            amp = (*it).getAmplitude();
            status = AFLIB_SUCCESS;
            break;
         }
      }
   }

   return (status);
}


void
aflibAudioMixer::setInputConfig(const aflibConfig& cfg)
{
   // This function overrides the aflibAudio base class function. It will change
   // the output configuration based on the mixing of the inputs. It will select the
   // greatest sample rate of all inputs. It will select the greatest data size amoung
   // all the inputs. It will select the last parents endian type. 

   map<int, aflibAudio *, less<int> >  parent_list = getParents();
   map<int, aflibAudio *, less<int> >::iterator it;
   set<aflibMixerItem, less < aflibMixerItem> >::iterator it_item;
   int  sample_rate = 0;
   aflib_data_endian  endian = AFLIB_ENDIAN_LITTLE;
   aflib_data_size    size = AFLIB_DATA_8U;
   aflibConfig out_cfg(cfg);
   int   chan_num = 0;
   int   chan_cur = -1;


   // Look at every parents data configuration
   for (it = parent_list.begin(); it != parent_list.end(); it++)
   {
      const aflibConfig& new_cfg = ((*it).second)->getOutputConfig();

      // Pick the biggest sample rate
      if (new_cfg.getSamplesPerSecond() > sample_rate)
      {
         sample_rate = new_cfg.getSamplesPerSecond();
      }

      // pick last endian config. It does not really matter
      endian = new_cfg.getDataEndian();

      // Pick 16S, 16U, 8S, or 8U in that order
      if (size != AFLIB_DATA_16S)
      {
         if (new_cfg.getSampleSize() == AFLIB_DATA_16S)
         {
            size = AFLIB_DATA_16S;
         }
         else if (new_cfg.getSampleSize() == AFLIB_DATA_16U)
         {
            size = AFLIB_DATA_16U;
         }
         else if (size != AFLIB_DATA_16U)
         {
            if (new_cfg.getSampleSize() == AFLIB_DATA_8S)
            {
               size = AFLIB_DATA_8S;
            }
            else if (size != AFLIB_DATA_8S)
            {
               size = AFLIB_DATA_8U;
            }
         }
      }
   }

   // Find number of output channels
   for (it_item = _mix_item.begin(); it_item != _mix_item.end(); it_item++)
   {
      if (chan_cur != (*it_item).getOutChannel())
      {
         chan_num++;
         chan_cur = (*it_item).getOutChannel();
      }
   }

   // Set and Store the output configuration
   out_cfg.setSamplesPerSecond(sample_rate);
   out_cfg.setSampleSize(size);
   out_cfg.setDataEndian(endian);
   out_cfg.setChannels(chan_num);
   setOutputConfig(out_cfg);

   // Set the input config to be the same as the output. This is what we need inputted
   // into each input. It will force the base classes to make the conversion.
   aflibAudio::setInputConfig(cfg);
   aflibAudio::setOutputConfig(out_cfg);
}


aflibStatus
aflibAudioMixer::compute_segment(
   list<aflibData *>& data,
   long long position)
{
   // This function is the one that does the work. It will take the list of audio data
   // passed in and based on the mix information stored will generate an output. It will
   // assume all data passed in is of the same length. If it is not then the output data
   // will be te longest of the inputs. If will then assume the missing data in the other
   // inputs are zero. The results will be passed back into the first element of the
   // list. This function could use many optimizations.


   int  i, k;
   int  input;
   int  in_chan;
   int  out_chan;
   int  amp;
   int  size;
   int  num_mixs;
   long length = 0;;
   double mult;
   int value;
   aflibData * out_data;
   map<int, aflibAudio *, less<int> > audio_list = getParents();
   list<aflibData *>::iterator it;
   map<int, aflibAudio *, less<int> >::iterator it_map;


   // Get common data we need
   size = data.size();
   num_mixs = getNumOfMixs();
   if (size == 0)
   {
      return(AFLIB_NO_DATA);
   }

   // Find the largest size of all data. Should all be equal.
   for (it = data.begin(); it != data.end() ; it++)
      if (length < (*it)->getLength())
         length = (*it)->getLength();

   // Allocate data array and zero all data
   out_data = new aflibData(getOutputConfig(), length);
   out_data->zeroData();

   // Mixes are ordered by output channel
   for (i = 1; i <= num_mixs; i++)
   {
      getMix(i, input, in_chan, out_chan, amp);

      for (it = data.begin(), it_map = audio_list.begin(); it != data.end(); it++, it_map++)
      {
         // IF this is the input we are looking for
         if (input == (*it_map).first)
         {
            mult = (double)amp / 100.0;

            // NOTE data objects and audio_list should be in the same order
            // store each element
            for (k = 0; k < length; k++)
            {
               // calculate new data to store
               value = (int)((*it)->getSample(k, in_chan) * mult);

               // sum new data with existing data in cell and store
               out_data->setSample(
                  out_data->getSample(k, out_chan) + value, k, out_chan);
            }
            break;
         }
      }
   }

   // remove all data elements and replace with one
   for (it = data.begin(); it != data.end() ; it++)
   {
      delete (*it);
   }
   data.erase(data.begin(), data.end());
   data.push_back(out_data);

   return (AFLIB_SUCCESS);
}

void
aflibAudioMixer::parentWasDestroyed(int parent_id)
{
   // If a parent is destroyed then walk through the mix element list and delete them.

   int num_segs;
   int i;
   int input, in_chan, out_chan, amp;
 
 
   num_segs = getNumOfMixs();
 
   // Count backwards so that num_segs changing does not affect us
   // Remove all mixes from this input
   for (i = num_segs; i != 0; i--)
   {
      getMix(i, input, in_chan, out_chan, amp);
      if (parent_id == input)
      {
         delMix(input, in_chan, out_chan);
      }
   }
}

bool
aflibAudioMixer::isDataSizeSupported(aflib_data_size size)
{
   // This overrides the virtual function in the base class.
 
   bool state = FALSE;
 
   if (size == getInputConfig().getSampleSize())
      state = TRUE;
 
   return (state);
}
 
bool
aflibAudioMixer::isEndianSupported(aflib_data_endian end)
{
   // This overrides the virtual function in the base class.
 
   bool state = FALSE;
 
   if (end == getInputConfig().getDataEndian())
      state = TRUE;
 
   return (state);
}

bool
aflibAudioMixer::isSampleRateSupported(int& rate)
{
   // This overrides the virtual function in the base class. See if the rate requested
   // is the rate that we have computed that we will be outputting.

   int value;
   bool ret_value = FALSE;

   // Get the rate of the data
   value = getOutputConfig().getSamplesPerSecond();
 
   // IF same rate then TRUE else return desired rate
   if (rate == value)
      ret_value = TRUE;
   else
      rate = value;
 
   return (ret_value);
}

