/*
 * Copyright: (C) 2003 Bruce W. Forsberg
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 *
 *   Bruce Forsberg  forsberg@tns.net
 *
 */


#include "otrplayer.h"
#include "otrplayerfile.h"
#include "id3tag.h"

#include <QKeyEvent>
#include <qapplication.h>
#include <qlabel.h>
#include <qslider.h>
#include <qdir.h>
#include <qpalette.h>
#include <q3listview.h>
#include <q3multilineedit.h>
#include <qtimer.h>
#include <qtextstream.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qradiobutton.h>
#include <q3buttongroup.h>

#include "aflib/aflibConfig.h"
#include "aflib/aflibFile.h"
#include "aflib/aflibAudioFile.h"
#include "aflib/aflibAudioPitch.h"


#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <string.h>

#include <cstdio>
#include <cstdlib>
#include <iostream>

using std::list;


#define PLAY_DELTA 512
#define MP3_DATA_CFG "otrplayer.cfg"
#define FILE_LINE 0
#define INFO_LINE 1
#define ID3_LINE  2

#define FILE_FIELD   0
#define LENGTH_FIELD 1
#define TOTAL_FIELD  2
#define KBPS_FIELD   3
#define RATE_FIELD   4
#define DIR_FIELD    5



// Constructor
otrplayer::otrplayer(QApplication *app, bool debug)
    : mainForm()
{
   _debug = debug;

   if (_debug)
      std::cout << "otrplayer: Entering constructor" << std::endl;

   _app = app;
   _input = NULL;
   _output = NULL;
   _pitch = NULL;
   _stop = TRUE;
   _current_position = 0;
   _channels = 0;
   _pitch_enabled = FALSE;
   _item = NULL;
   _mount_while_play = FALSE;
   _file_was_playing_when_unmounted = FALSE;

   if (_debug)
      std::cout << "otrplayer: creating otrplayerfile" << std::endl;
   _otr_file = new otrplayerfile();

   // Initialize Config tab 
   if (_debug)
      std::cout << "otrplayer: calling initButtons" << std::endl;
   initButtons();

   // Initialize Dir tab from config file
   if (_debug)
      std::cout << "otrplayer: calling initDirs" << std::endl;
   initDirs();

   pitchSlider->setEnabled(_pitch_enabled);
   timeSlider->setRange(0, 100);
   infoEdit->setReadOnly(TRUE);

   lookForDecoders();

   if (_debug)
      std::cout << "otrplayer: calling getFiles" << std::endl;
   getFiles();

   // Create a timer to operate as a work procedure
   if (_debug)
      std::cout << "otrplayer: creating timer" << std::endl;
   _timer = new QTimer(this);
   connect(_timer, SIGNAL(timeout()), this, SLOT(run()));

#if 0
   if (_debug)
      std::cout << "otrplayer: calling grabKeyboard" << std::endl;
   QPEApplication::grabKeyboard();

   if (_debug)
      std::cout << "otrplayer: creating QCopChannel" << std::endl;
   QCopChannel* pcmciaChannel = new QCopChannel( "QPE/Card", this );
   connect( pcmciaChannel, SIGNAL(received(const QCString &, const QByteArray &)),
     this, SLOT(pcmciaMessage(const QCString &, const QByteArray &)) );
#endif
}

// Destructor
otrplayer::~otrplayer()
{
   if (_debug)
      std::cout << "otrplayer: calling destructor" << std::endl;
   delete _timer;

   if (_stop == FALSE)
      stop();

   delete _otr_file;
   delete _input;
   delete _output;
   delete _pitch;

   // QPEApplication::ungrabKeyboard();
}


void
otrplayer::keyPressEvent(QKeyEvent *e)
{
   bool action = false;

   if (_debug)
      std::cout << "otrplayer: Pressed key:" << e->key() << std::endl;

   // Far left key on the SL-5500 PDA, activity
   if (e->key() == Qt::Key_F7)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as PLUS" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::CALENDAR_KEY));
   }
   // Next left ley on the SL-5500 PDA, contacts
   else if (e->key() == Qt::Key_F8)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as MINUS" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::ADDRESS_KEY));
   }
   // Second far right key of the SL-5500 PDA, Menu
   else if (e->key() == Qt::Key_F11)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as MENU" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::MENU_KEY));
   }
   // Far right key of the SL-5500 PDA, Mail
   else if (e->key() == Qt::Key_F13)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as EMAIL" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::EMAIL_KEY));
   }
   // Cancel Key
   else if (e->key() == Qt::Key_Escape)
   {
      if (_debug)
         std::cout << "otrplayer: Key recognized as Cancel" << std::endl;
      action = performKeyEvent(_otr_file->getButton(otrplayerfile::HOME_KEY));
   }
   else
   {
      if (_debug)
         std::cout << "otrplayer: Key not recognized" << std::endl;
   }

   if (action == false)
      e->ignore();
   else
      e->accept();
}


bool
otrplayer::performKeyEvent(int func_event)
{
   int  value;
   bool action = true;


   // No Selection
   if (func_event == 0)
      action = false;
   // Volume Up
   else if (func_event == 1)
      volumeChange(5);
   // Volume Down
   else if (func_event == 2)
      volumeChange(-5);
   // Fast Forward
   else if (func_event == 3)
   {
      _otr_file->getAdvance(0, value);
      seekSeconds(value);
   }
   // Fast Rewind
   else if (func_event == 4)
   {
      _otr_file->getAdvance(1, value);
      seekSeconds(value);
   }
   // Play
   else if (func_event == 5)
      play();
   // Stop
   else if (func_event == 6)
      stop();
   // Play / Stop
   else if (func_event == 7)
   {
      if (_stop == TRUE)
         play();
      else
         stop();
   }
   else
      action = false;

   return (action);
}


void
otrplayer::setup(QString& file)
{
   aflibConfig  input_config;
   aflibStatus  status;
   QString      str, secs, mins;
   QString      local_str = file.toLocal8Bit();


   if (_debug)
      std::cout << "setup: Enter" << std::endl;
   delete _input;
   delete _output;
   delete _pitch;

   if (_debug)
      std::cout << "setup: create aflibAudioFile" << std::endl;
   // IF an ogg extension then use AUTO type
   if (file.endsWith(".mp3", Qt::CaseInsensitive) == true)
      _input = new aflibAudioFile(AFLIB_MPEG_TYPE, local_str.toStdString(),
               &input_config, &status);
   else
      _input = new aflibAudioFile(AFLIB_AUTO_TYPE, local_str.toStdString(),
               &input_config, &status);
   _input->setCacheEnable(FALSE);

   if (status == AFLIB_SUCCESS)
   {
       if (_debug)
          std::cout << "setup: create aflibAudioPitch" << std::endl;
      _pitch = new aflibAudioPitch(*_input, 1.0);
      _pitch->enable(_pitch_enabled);
      _pitch->setCacheEnable(FALSE);
      pitchCallback(pitchSlider->value());

       if (_debug)
          std::cout << "setup: create output aflibAudioFile" << std::endl;
      aflibConfig output_config(_input->getInputConfig());
      _output = new aflibAudioFile(*_pitch, AFLIB_DEV_TYPE, "/dev/dsp",
            &output_config, &status);
      _output->setCacheEnable(FALSE);

      _total_size = output_config.getTotalSamples();
      _samples_per_second = output_config.getSamplesPerSecond();
      _channels = output_config.getChannels();
   }
   else
   { 
      delete _input;
      _input = NULL;
      _output = NULL;
      _pitch = NULL;
      _stop = TRUE;
      QMessageBox::information(this, "Warning",
         "File is not currently available.\n" + file, QMessageBox::Ok);

      stop();
   }
}


void
otrplayer::getFiles()
{
   QStringList  lst, tree;
   QStringList::Iterator  it, itt;
   Q3ListViewItem  *item = NULL;
   aflibAudioFile  *temp = NULL;
   aflibConfig  input_config;
   aflibStatus  status;
   int          value;
   QString      str;
   bool         ret_value;
   QStringList  tmp_lst;
   QString      filepath;
   int          sec, total_sec;
   static bool  madplay_found = TRUE;
   static bool  displayed_warning = FALSE;
   QStringList  file_list;
   QStringList  old_list;
   QString      dir_save;
   QString      file_save;
   bool         value_saved = FALSE;

   if (_debug)
      std::cout << "getFiles: Entering getFiles" << std::endl;

   // IF there is a currently selected item then save it.
   item = playList->currentItem();
   if (item != 0)
   {
      dir_save = item->text(DIR_FIELD);
      file_save= item->text(FILE_FIELD);
      value_saved = TRUE;
   }
   else
   {
      QString last_item;
      int     pos;

      _otr_file->getPlayedFile(last_item);
      if (!last_item.isEmpty())
      {
         pos = last_item.lastIndexOf("/");
         dir_save = last_item.left(pos);
         file_save = last_item.right(last_item.length() - pos - 1);
         value_saved = TRUE;
      }
   }
   if (_debug)
   {
      if (value_saved == TRUE)
      {
         std::cout << "getFiles: dir saved:" << dir_save.toStdString() << std::endl;
         std::cout << "getFiles: file saved:" << file_save.toStdString() << std::endl;
      }
   }

   if (_debug)
      std::cout << "getFiles: clear Entries" << std::endl;

   // Clear all previous entries
   file_list.clear();

   // Clear all entries in list GUI
   playList->clear();

   if (_debug)
      std::cout << "getFiles: get search directories" << std::endl;
   // Get directories to search
   _otr_file->getDirs(old_list, tree);

   for (itt = old_list.begin(), it = tree.begin(); itt != old_list.end(); ++itt, ++it)
   {
      if ((*it) == "Yes")
         buildDirTree((*itt), lst);
      else
      {
         QDir  dir(*itt);
         if (dir.exists())
         {
            lst.append(*itt);
         }
      }
   }

   if (_debug)
      std::cout << "getFiles: get mp3 files from each directory" << std::endl;
   // Loop for every directory tree
   for (itt = lst.begin(); itt != lst.end(); ++itt)
   {
      QDir  dir(*itt);
      QStringList filters;

      filters << "*.mp3" << "*.ogg" << "*.wav";
      dir.setNameFilters(filters);
      tmp_lst = dir.entryList();

      for (it = tmp_lst.begin(); it != tmp_lst.end(); ++it)
      {
         filepath = (*itt);
         filepath.append("/");
         filepath.append(*it);

         file_list.append(filepath);
      }
   }

   if (_debug)
      std::cout << "getFiles: Update list of files" << std::endl;
   // Update list of files
   _otr_file->process(file_list);

   for (it = file_list.begin(); it != file_list.end(); ++it)
   {
      QString  str1, str2, str3, secs, mins;
      QString  latin_str = (*it).toLatin1();
      QString  local_str = (*it).toLocal8Bit();

      if (_debug)
      {
         std::cout << "getFiles: get info for:" << latin_str.toStdString() << std::endl;
         std::cout << "getFiles: local8bit:" << local_str.toStdString() << std::endl;
      }

      // IF an ogg extension then use AUTO type
      if ((*it).endsWith(".mp3", Qt::CaseInsensitive) == true)
         temp = new aflibAudioFile(AFLIB_MPEG_TYPE, local_str.toStdString(),
               &input_config, &status);
      else
         temp = new aflibAudioFile(AFLIB_AUTO_TYPE, local_str.toStdString(),
               &input_config, &status);

      // Did we not find madplay
      if (status == AFLIB_NOT_FOUND)
         madplay_found = FALSE;

      if ((status == AFLIB_ERROR_INITIALIZATION_FAILURE) ||
          (status == AFLIB_ERROR_OPEN))
      {
         QString str = "Can\'t recognize file ";
         str.append((*it).toLatin1());
         infoEdit->insertLine(str, -1);
      }
      else
      {
         if (temp)
         {
            ret_value = temp->getItem("kbps", (void *)&value);
            if (ret_value == FALSE)
               value = 0;
         }
         else
         {
            if (_debug)
               std::cout << "getFiles: ERROR creating aflibAudioFile" << std::endl;
            value = 0;
         }

         if (input_config.getSamplesPerSecond() != 0)
         {
            total_sec = input_config.getTotalSamples()/input_config.getSamplesPerSecond();
            // round up to next second
            total_sec++;
         }
         else 
            total_sec = 0;
         secondsToQString(total_sec, str1);

         if (input_config.getSamplesPerSecond() != 0)
         {
            // IF -1 we are at end of file
            if (_otr_file->getOffset(*it) == -1)
               sec = total_sec;
            else
               sec = _otr_file->getOffset(*it)/input_config.getSamplesPerSecond();
         }
         else
            sec = 0;
         secondsToQString(sec, str2);

         if (_debug)
            std::cout << "getFiles: Update info in list box" << std::endl;

         str.setNum(value);
         str3.setNum(input_config.getSamplesPerSecond());

         QChar chr('/');
         int slash;
         slash = (*it).lastIndexOf(chr);
         item = new Q3ListViewItem( playList, item,
            (*it).right((*it).length() - slash - 1),
            str2, str1, str, str3, (*it).left(slash));
      }
      delete temp;
   }

   // IF madplay was not found and have not displayed warning yet
   if (madplay_found == FALSE && displayed_warning == FALSE)
   {
      displayed_warning = TRUE;
      QMessageBox::information(this, "Warning", "Madplay utility not found.\nPlease install the IPK for madplay.",
         QMessageBox::Ok);
   }

   // Don't allow resizing of columns. We do this after filling all fields
   // for the first time so the initial sizing is correct.
   if (_debug)
      std::cout << "getFiles: fix column sizes" << std::endl;
   playList->setColumnWidthMode(FILE_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(LENGTH_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(TOTAL_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(KBPS_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(RATE_FIELD, Q3ListView::Manual);
   playList->setColumnWidthMode(DIR_FIELD, Q3ListView::Manual);

   // Set previous item or first item if no previous
   if (_debug)
      std::cout << "getFiles: set previous item" << std::endl;
   if (playList->firstChild() != 0)
      playList->setCurrentItem(playList->firstChild());
   if (value_saved == TRUE)
   {
      Q3ListViewItemIterator  t(playList);

      // Loop thru entire list
      for ( ; t.current(); t++)
      {
         Q3ListViewItem  *itm = t.current();

         if (( itm->text(FILE_FIELD) == file_save) &&
              (itm->text(DIR_FIELD) == dir_save))
         {
            playList->setCurrentItem(itm);
            break;
         }
      }
   }
}


void
otrplayer::play()
{
   if (_debug)
      std::cout << "play: play called" << std::endl;
   // IF playing then ignore.
   if (_stop == FALSE)
      return;

   // Call our run routine when app loop has nothing to do
   _timer->start(0);

#if 0
   // Stop unit from going into suspend mode during playing
   QCopEnvelope("QPE/System", "setScreenSaverMode(int)" ) << QPEApplication::DisableSuspend;
#endif

   _item = playList->selectedItem();
   if (_item == NULL)
      return;

   _stop = FALSE;

   prepareNextFile();
}

void
otrplayer::stop()
{
   QString  str;
   Q3ListViewItem  *item;


   if (_debug)
      std::cout << "stop: stop called" << std::endl;

#if 0
   // Reenable suspend mode
   QCopEnvelope("QPE/System", "setScreenSaverMode(int)" ) << QPEApplication::Enable;
#endif

   _timer->stop();

   _stop = TRUE;
   delete _input;
   delete _pitch;
   delete _output;
   _pitch = NULL;
   _input = NULL;
   _output = NULL;

   item = playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   // Update offset of file
   _otr_file->storeOffset(str, (long)_current_position);

   // IF a mount or unmount occurred while playing then refresh file display
   if (_mount_while_play == TRUE)
   {
      _mount_while_play = FALSE;
      getFiles();
   }
}

void
otrplayer::slider(int value)
{
   _current_position = _total_size / 100 * value;
}


void
otrplayer::pitchCallback(int value)
{
   if (_pitch)
      _pitch->setFactor( (double)value/-500.0 + 1.0);
}


void
otrplayer::pitchEnable()
{
   _pitch_enabled = !_pitch_enabled;
   pitchSlider->setEnabled(_pitch_enabled);
   if (_pitch)
      _pitch->enable(_pitch_enabled);
}


void
otrplayer::secondsToQString(
   int sec,
   QString&  str)
{
   QString      secs, mins;


   mins.setNum(sec/60);
   secs.setNum(sec%60);
   str = mins;
   if (sec%60 < 10)
      str.append(":0");
   else
      str.append(":");
   str.append(secs);
}


void
otrplayer::QStringToSeconds(
   int& sec,
   QString  str)
{
   int  colon;
   int  min;
   QTextStream ts(&str);
   QChar   dummy;


   ts >> min >> dummy >> sec;

   sec = sec + min * 60;
}


void
otrplayer::seekSeconds(
   int sec)
{
   _current_position += (sec * _samples_per_second);
   if (_current_position < 0)
      _current_position = 0;
}


void
otrplayer::buildDirTree(
   QString  str,
   QStringList&  lst)
{
   QStringList::Iterator  it;
   QStringList  dir_lst;
   QDir  dir(str);

   if (dir.exists())
   {
      lst.append(str);

      dir.setFilter(QDir::Dirs | QDir::NoSymLinks);
      dir_lst = dir.entryList();

      for (it = dir_lst.begin(); it != dir_lst.end(); ++it)
      {
         if ((*it != ".") && (*it != ".."))
            buildDirTree(dir.absolutePath() + "/" + *it, lst);
      }
   }
}


void
otrplayer::displayID3(QString& file_str)
{
   QString local_str = file_str.toLocal8Bit();

   
   if (_debug)
   {
      std::cout << "displayID3: get info for:" << file_str.toStdString() << std::endl;
      std::cout << "displayID3: local8bit:" << local_str.toStdString() << std::endl;
   }
   int fddd = open(local_str.toLatin1(), O_RDONLY);
   ID3Tag *id3 = new ID3Tag;
   if (id3->read(fddd) == TRUE)
   {
      int count = ID3_LINE;
      QString  str;

      if (!id3->title().isEmpty())
         infoEdit->insertLine("Title - " + id3->title(), count++);
      if (!id3->artist().isEmpty())
         infoEdit->insertLine("Artist - " + id3->artist(), count++);
      if (!id3->album().isEmpty())
         infoEdit->insertLine("Album - " + id3->album(), count++);
      if (id3->year() != 0)
         infoEdit->insertLine("Year - " + str.setNum(id3->year()), count++);
      if (!id3->comment().isEmpty())
         infoEdit->insertLine("Comment - " + id3->comment(), count++);
      if (id3->track() != 0)
         infoEdit->insertLine("Track - " + str.setNum(id3->track()), count++);
   }
   delete id3;
   ::close(fddd);
}


void
otrplayer::run()
{
   int             delta;
   aflibStatus     status;
   static int      old_sec = -1;
   int             sec;
   QString         str, file_str;


   // IF currently playing
   if (_stop == FALSE)
   {
      if (_current_position == -1)
         status = AFLIB_END_OF_FILE;
      else
      {
         // must reset delta since process can change it
         delta = PLAY_DELTA;

         _output->process(status, _current_position, delta);

         _current_position += delta;

         // Update time, only every second
         if (_samples_per_second == 0)
            sec = 0;
         else
            sec = _current_position/_samples_per_second;
         if (sec != old_sec)
         {
            secondsToQString(sec, str);
            if (_total_size == 0)
               timeSlider->setValue(0);
            else
               timeSlider->setValue((int)( 100 * _current_position/_total_size));
            _item->setText(LENGTH_FIELD, str);
         }
         old_sec = sec;
      }

      if (status != AFLIB_SUCCESS)
      {
         file_str = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);

         // -1 means we are at end of the file
         _current_position = -1;

         // Update offset of file
         _otr_file->storeOffset(file_str, (long)_current_position);

         // Advanced to the next file
         Q3ListViewItemIterator  it(_item);
         it++;
         if (it.current() != NULL)
         {
            playList->setSelected(it.current(), TRUE);
            _item = playList->selectedItem();
            prepareNextFile();
         }
         else
         {
            stop();
         }
      }
   }
}


void
otrplayer::prepareNextFile()
{
   QString         info_str, str1;

   _file = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);

   setup(_file);

   infoEdit->clear();
   infoEdit->insertLine(_item->text(FILE_FIELD), FILE_LINE);

   info_str = _item->text(KBPS_FIELD);
   info_str.append(" kbps, ");
   str1.setNum(_samples_per_second);
   info_str.append(str1);
   info_str.append(" Hz, ");
   if (_channels == 1)
      info_str.append("Mono");
   else
      info_str.append("Stereo");
   infoEdit->insertLine(info_str, INFO_LINE);

   displayID3(_file);

   _current_position = _otr_file->getOffset(_file);

   // Store this as the last played file
   _otr_file->storePlayedFile(_file);
}


void
otrplayer::ff()
{
   Q3ListViewItem  *item;
   QString         str, str1, str2;
   int             secs;


   item = playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   QStringToSeconds(secs, item->text(TOTAL_FIELD));

   // Update offset of file
   _otr_file->storeOffset(str, (long)-1);

   if (_stop == FALSE)
   {
      str1 = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);
      if (str == str1)
         _current_position = -1;
   }

   secondsToQString(secs, str2);
   item->setText(LENGTH_FIELD, str2);
}


void
otrplayer::rev()
{
   Q3ListViewItem  *item;
   QString         str, str1, str2;


   item = playList->selectedItem();
   if (item == NULL)
      return;

   str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

   // Update offset of file
   _otr_file->storeOffset(str, (long)0);

   if (_stop == FALSE)
   {
      str1 = _item->text(DIR_FIELD) + "/" + _item->text(FILE_FIELD);
      if (str == str1)
         _current_position = 0;
   }

   secondsToQString(0, str2);
   item->setText(LENGTH_FIELD, str2);
}


void
otrplayer::initButtons()
{
   int           menu_sel;
   int           value;
   int           font;


   menu_sel = _otr_file->getButton(otrplayerfile::CALENDAR_KEY);
   calendarBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::ADDRESS_KEY);
   addressBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::HOME_KEY);
   homeBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::MENU_KEY);
   menuBox->setCurrentItem(menu_sel);

   menu_sel = _otr_file->getButton(otrplayerfile::EMAIL_KEY);
   emailBox->setCurrentItem(menu_sel);

   // Fast Forward advance button value
   if (_otr_file->getAdvance(0, value) == true)
   {
      ffEdit->setValue(value);
   }
   else
      ffEdit->setValue(30);

   if (_otr_file->getAdvance(1, value) == true)
   {
      rewEdit->setValue(value);
   }
   else
      rewEdit->setValue(-30);

   // Set the font for the playList
   if (_otr_file->getFont(font) == false)
     font = 1;

   // Set the button on and font
   fontGroup->setButton(font);
   fontSize(font);
}


void
otrplayer::volumeChange(int value)
{
   int vol;
   int new_vol;

#if 0
   // We do this to flush results before QCop call
   {
      Config cfg("qpe");
      cfg.setGroup("Volume");
      vol = cfg.readNumEntry("VolumePercent", 0);

      vol += value;
      cfg.writeEntry("VolumePercent",(int)vol);
   }
   QCopEnvelope( "QPE/System", "volumeChange(bool)" ) << FALSE;
#endif

   int fd = open("/dev/mixer", O_RDWR);
   ioctl( fd , MIXER_READ(0), &vol );

   new_vol = vol & 0xff;
   new_vol += value;
   if (new_vol > 100)
      new_vol = 100;
   else if (new_vol < 0)
      new_vol = 0;

   vol = new_vol < 8 | new_vol;
   
   ioctl( fd , MIXER_WRITE(0), &vol );
   ::close( fd );
}


void
otrplayer::calendarCallback(int value)
{
   _otr_file->setButton(otrplayerfile::CALENDAR_KEY, value);
}


void
otrplayer::addressCallback(int value)
{
   _otr_file->setButton(otrplayerfile::ADDRESS_KEY, value);
}


void
otrplayer::homeCallback(int value)
{
   _otr_file->setButton(otrplayerfile::HOME_KEY, value);
}


void
otrplayer::menuCallback(int value)
{
   _otr_file->setButton(otrplayerfile::MENU_KEY, value);
}


void
otrplayer::emailCallback(int value)
{
   _otr_file->setButton(otrplayerfile::EMAIL_KEY, value);
}


void
otrplayer::initDirs()
{
   QStringList  lst;
   QStringList  tree;
   Q3ListViewItem  *item = NULL;
   QStringList::Iterator  it, itt;


   _otr_file->getDirs(lst, tree);
   if (lst.isEmpty())
   {
      _otr_file->setDir("/media/mmc1", "Yes");
      _otr_file->setDir("/media/mmc2", "Yes");

      _otr_file->getDirs(lst, tree);
   }

   dirView->clear();

   for (it = lst.begin(), itt = tree.begin(); it != lst.end(); it++, itt++)
   {
      item = new Q3ListViewItem( dirView, item, (*it), (*itt));
   }
}


void
otrplayer::setDir()
{
   if (dirEdit->text().isEmpty())
   {
      QMessageBox::information(this, "Warning", "Must enter a Directory.",
         QMessageBox::Ok);
   }
   else
   {
      if (treeBox->isChecked())
         _otr_file->setDir(dirEdit->text(), "Yes");
      else
         _otr_file->setDir(dirEdit->text(), "No");

      // Display new entry
      initDirs();

      // Stop playing file
      stop();

      // Reload all files
      getFiles();
   }
}


void
otrplayer::deleteDir()
{
   Q3ListViewItem   *item;


   item = dirView->selectedItem();
   if (item != NULL)
   {
      _otr_file->deleteDir(item->text(0));

      // Display new entry
      initDirs();

      // Stop playing file
      stop();

      // Reload all files
      getFiles();
   }
}


void
otrplayer::ffCallback(int value)
{
   _otr_file->setAdvance(0, value);
}


void
otrplayer::rewCallback(int value)
{
   _otr_file->setAdvance(1, value);
}


void
otrplayer::removeFiles()
{
   Q3ListViewItemIterator  it(playList);
   Q3ListViewItemIterator  itt(playList);
   Q3ListViewItem  *item;
   int   num_items = 0;
   QString  msg_str, num_str;


   // Loop thru entire list
   for ( ; it.current(); it++)
   {
      QString  str;
      long     offset;
      QDir     dir;

      item = it.current();
      str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

      // IF offset is -1 then remove
      offset = _otr_file->getOffset(str);
      if (offset == -1)
         num_items++;
   }

   if (num_items > 0)
   {
      msg_str = "There are " + num_str.setNum(num_items) + " files to remove\n";
      msg_str.append("Are you sure you want them removed?");

      switch( QMessageBox::warning( this, "Remove Files",
                          msg_str,
                          "Yes, Remove", "No, Do Not Remove", 0,
                          0, 1 ))
      {
         case 0: // Yes
            // Loop thru entire list
            for ( ; itt.current(); itt++)
            {
               QString  str;
               long     offset;
               QDir     dir;

               item = itt.current();
               str = item->text(DIR_FIELD) + "/" + item->text(FILE_FIELD);

               // IF offset is -1 then remove
               offset = _otr_file->getOffset(str);
               if (offset == -1)
                  dir.remove(str);
            }

            // Stop playing file
            stop();

            // Reload all files
            getFiles();

         break;
         case 1: // No
         break;
      }
   }
}


void
otrplayer::fontSize(int size)
{
   QFont font(  playList->font() );
   font.setFamily( "helvetica" );

   if (size == 0)
      playList->setFont( bigButton->font() );
   else if (size == 2)
      playList->setFont( smallButton->font() );
   else 
      playList->setFont( medButton->font() );

   _otr_file->setFont(size);
}


#if 0
void
otrplayer::pcmciaMessage(const QCString &msg, const QByteArray &byt)
{
   int i;


   if ( msg == "mtabChanged()" )
   {
      // IF currently playing then stop while CF ejects.
      // If one powers down while in play mode the CF card will
      // eject and this message will be sent. We stop play mode
      // delay then start again once the unit has been powered
      // back on. This is a kludge but the only way I kind find
      // to detect when a CF is unmounted and mounted. If we keep
      // on playing during power down without doing this then when
      // we power on madplay will error since the file will not be
      // available.
      if (_stop == FALSE)
      {
         stop();

         // sleep for 4 seconds
         for (i =0; i < 4; i++)
            sleep(1);

         play();
      }
      // Need to still delay in case someone presses play right after
      // power on. This allows for CF card to mount.
      else
      {
         // sleep for 4 seconds
         for (i =0; i < 4; i++)
            sleep(1);
      }
   }
}

#endif


void
otrplayer::lookForDecoders()
{
   list<string> formats;
   list<string> descriptions;
   list<string>::iterator  it;


   // Get all the found formats
   aflibFile::returnSupportedFormats(formats, descriptions);

   for (it = formats.begin(); it != formats.end(); it++)
   {
      if ( (*it) != "DEVICE") 
      {
         QString str;

         str = "Found Decoder: ";
         str.append((*it).c_str());
         infoEdit->insertLine(str, -1);
      }
   }
}


