#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <libintl.h>
#include <locale.h>
#include <glib.h>
#include <dbus/dbus-glib.h>
#include <hildon/hildon-notification.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>

#define OSSO_MEDIA_SERVER_SERVICE "com.nokia.osso_media_server"
#define OSSO_MEDIA_SERVER_PATH "/com/nokia/osso_media_server"
#define OSSO_MEDIA_SERVER_MUSIC_IFACE "com.nokia.osso_media_server.music"
#define OSSO_MEDIA_SERVER_DETAILS_SIG "details_received"
#define OSSO_MEDIA_SERVER_STATE_SIG "state_changed"
#define OSSO_MEDIA_SERVER_STATE_STOPPED "stopped"

static DBusGConnection *sess_dbus_conn = NULL;
static DBusGProxy *media_server_proxy = NULL;
static HildonNotification *notification = NULL;
static GtkWidget *banner = NULL;

static guint timeout = 10;
static gboolean use_hildon_banner = FALSE;

static GOptionEntry entries[] = 
{
	{"timeout", 't', 0, G_OPTION_ARG_INT, &timeout, "The time that the notification will be on the screen for", "TIMEOUT"},
	{"use-hildon-banner", 'u', 0, G_OPTION_ARG_NONE, &use_hildon_banner, "Use a HildonBanner instead of a HildonNotification", NULL},
	{NULL}
};

static void sig_handler (gint sig G_GNUC_UNUSED)
{
	gtk_main_quit ();
}

static void notification_close (void)
{
	if (!use_hildon_banner && notification)
	{
		notify_notification_close (NOTIFY_NOTIFICATION (notification), NULL);
		notification = NULL;
	}
	else if (use_hildon_banner && banner)
	{
		gtk_widget_destroy (banner);
		banner = NULL;
	}
}

static gboolean on_banner_delete_event (const GtkWidget *widget G_GNUC_UNUSED, const GdkEvent *event G_GNUC_UNUSED, gconstpointer user_data G_GNUC_UNUSED)
{
	notification_close ();
	return TRUE;
}

static void on_state_changed(const DBusGProxy *proxy G_GNUC_UNUSED, const gchar *state, gconstpointer user_data G_GNUC_UNUSED)
{
	if (!strcmp (state, OSSO_MEDIA_SERVER_STATE_STOPPED))
		notification_close ();
}

static void on_details_recieved (const DBusGProxy *proxy G_GNUC_UNUSED, GHashTable *hash, gconstpointer user_data G_GNUC_UNUSED)
{
	gchar *artist = (gchar *) g_hash_table_lookup (hash, "artist");
	gchar *title = (gchar *) g_hash_table_lookup (hash, "title");

	if (!artist && !title)
		return;

	if (!artist)
		artist = dgettext ("mediaplayer", "medi_li_nowplay_unknown_artist");

	if (!title)
		title = " ";

	if (!use_hildon_banner)
	{
		notification = hildon_notification_new (artist, title, "qgn_list_medi", "media");
		notify_notification_set_timeout (NOTIFY_NOTIFICATION (notification), timeout * 1000);
		notify_notification_show (NOTIFY_NOTIFICATION (notification), NULL);
	}
	else
	{
		banner = hildon_banner_show_informationf (NULL, "qgn_indi_gene_play", "%s - %s", artist, title);
		hildon_banner_set_timeout (HILDON_BANNER (banner), timeout * 1000);
		g_signal_connect (banner, "destroy", G_CALLBACK (on_banner_delete_event), NULL);
	}
}

int main (int argc, char **argv)
{
	GOptionContext *context = g_option_context_new (NULL);
	g_option_context_set_summary (context, "Now Playing Notifier, the Nice Way to be notified on song changes");
	g_option_context_add_main_entries (context, entries, NULL);
	g_option_context_add_group (context, gtk_get_option_group (FALSE));

	if (!g_option_context_parse (context, &argc, &argv, NULL))
		g_printerr ("Error parsing options.");

	g_option_context_free (context);

	gtk_init (&argc, &argv);
	g_set_application_name ("Now Playing Notifier");
	if (!use_hildon_banner)
		notify_init (g_get_application_name ());

	sess_dbus_conn = dbus_g_bus_get (DBUS_BUS_SESSION, NULL);
	g_assert (sess_dbus_conn);
	
	media_server_proxy = dbus_g_proxy_new_for_name (sess_dbus_conn, OSSO_MEDIA_SERVER_SERVICE, OSSO_MEDIA_SERVER_PATH, OSSO_MEDIA_SERVER_MUSIC_IFACE);
	g_assert (media_server_proxy);

	dbus_g_proxy_add_signal (media_server_proxy, OSSO_MEDIA_SERVER_DETAILS_SIG, dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING), G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (media_server_proxy, OSSO_MEDIA_SERVER_DETAILS_SIG, G_CALLBACK (on_details_recieved), NULL, NULL);

	dbus_g_proxy_add_signal (media_server_proxy, OSSO_MEDIA_SERVER_STATE_SIG, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (media_server_proxy, OSSO_MEDIA_SERVER_STATE_SIG, G_CALLBACK (on_state_changed), NULL, NULL);

	signal (SIGINT, sig_handler);
	signal (SIGQUIT, sig_handler);
	signal (SIGTERM, sig_handler);

	gtk_main ();

	dbus_g_proxy_disconnect_signal (media_server_proxy, OSSO_MEDIA_SERVER_DETAILS_SIG, G_CALLBACK (on_details_recieved), NULL);
	notification_close ();
	g_object_unref (media_server_proxy);
	dbus_g_connection_unref (sess_dbus_conn);
	if (!use_hildon_banner)
		notify_uninit ();

	return EXIT_SUCCESS;
}
