/**
 * @file	moncellsd.c
 * @author  Zach Habersang
 * @brief 	System Monitor Server Daemon
 *
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
 
// TODO: write a handshake between client and server such as a version compatibility send
 
#include "moncellsd.h"
#include "info.h"

// threads, must be linked... -lpthread
#include <pthread.h> 

// daemonizing
#include <pwd.h>
#include <grp.h>
#include <sys/resource.h>
#include <sys/stat.h>
#include <fcntl.h>

#define DEFAULT_WORKING_DIR "/tmp/"
#define DEFAULT_RUNNING_USERNAME "root"
#define DEFAULT_RUNNING_GROUP "daemon"

// errno
#include <errno.h>

// Monitor Server Signal Struct
struct {
	unsigned int signalLog[66];

	unsigned int signalReceived:1;
	int lastSignalReceived;

	unsigned int shutdownInitiated:1;

	struct sigaction handler;
} MS_signals;

struct information info; // global system information instance


/** STAND ALONE MAIN! :: Master Server Main Func */
int main(int argc, char *argv[])
{
	int option;
	int run_foreground = 0;

	// Check command line arguements
	while ((option = getopt(argc, argv, "hf")) != -1) 
	{
		switch (option) 
		{
			case 'f': // Run in foreground
				run_foreground = 1;
				break;
			case 'h': // Help, print usage and exit
				printuse_exit();
				break;
		}
	}

	// Start server in foreground
	if (run_foreground)
		serv_init();
		
	// Daemonize and start server in background
	else
	{
		switch(daemonize()) 
		{
			case -1: // Error
				ServDieWithError("Failed to fork()", (int)NULL);
				break;

			case 1: // Parent
				break;
				
			default: // Child
				MS_signals_init(NULL);	
				monitor_init(); // do initial monitoring operations
				serv_init();
				break;
		}
	}
	return EXIT_SUCCESS;
}


/** Need to do these before a client connects, only once! */
void monitor_init(void)
{
	// do this beforehand so usage doesn't show up as 0
	if (update_top() < 0)
		ServERR("update_stat() failed.");
	
	if (get_freq() < 0)
		ServERR("get_freq() failed.");
}


/** Start the Network Monitor Server */
void serv_init(void)
{
	int servsock;
	int clntsock;
    struct sockaddr_in servAddr;
	struct sockaddr_in clntAddr;
	socklen_t clntlen;

	// Create socket
    if(-1 == (servsock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)))
        ServDieWithError("Failed to create socket.", 0);
 
	// Set address
    memset(&servAddr, 0, sizeof(servAddr));
    servAddr.sin_family = AF_INET;
    servAddr.sin_port = htons(PORT);
    servAddr.sin_addr.s_addr = INADDR_ANY;
 
	// Bind to address
    if(-1 == bind(servsock, (const void *)&servAddr, sizeof(servAddr)))
        ServDieWithError("Failed to bind address", servsock);

	// Listen for incoming connections
    if(-1 == listen(servsock, MAXPENDING))
        ServDieWithError("Failed to listen on address", servsock);
		
	// infinite server loop
	for (;;)
	{
		clntlen = sizeof(struct sockaddr);
		clntsock = accept(servsock, (struct sockaddr *) &clntAddr, &clntlen);
		if (DEBUG) printf("Handling client %s\n", inet_ntoa(clntAddr.sin_addr));		
		recvclnt(clntsock, clntAddr);
	}
		
}


void recvclnt(int clntsock, struct sockaddr_in clntAddr) 
{
	pthread_t threadID;
	struct ThreadArgs *threadArgs;
	
	// create separate memory for client argument 
	if ((threadArgs = (struct ThreadArgs *)malloc(sizeof(struct ThreadArgs))) == NULL)
		ServDieWithError("thread malloc() failed", clntsock);
		
	// save client socket
	threadArgs->socket = clntsock;
	threadArgs->address = clntAddr;
	
	// create client thread
	if (pthread_create(&threadID, NULL, ServerThreadMain, (void *) threadArgs) != 0)
		ServDieWithError("pthread_create() failed", clntsock);
	if (DEBUG) printf("with thread %ld\n", (long int) threadID);
}


void *ServerThreadMain(void *threadArgs)
{
	int clntsock;
	struct sockaddr_in clntAddr;
	
	// Gurantees that thread resources are deallocated upon return
	pthread_detach(pthread_self());
	
	// Extract socket file descriptor from argument, free thread arugments
	clntsock = ((struct ThreadArgs *) threadArgs) -> socket;
	clntAddr = ((struct ThreadArgs *) threadArgs) -> address;
	free(threadArgs);
	
	handleClient(clntsock, clntAddr);
	
	return (NULL);
}

void printuse_exit(void)
{
	printf(":: Network Monitor Server ::\n");
	printf("\t -h \t help (this message now)\n");
	printf("\t -f \t run in foreground\n");
	exit(EXIT_SUCCESS);
}


/** 
 * Print error message, close the socket, and exit with an error.
 * @param errorMsg The error message to print
 * @param servsock The server socket
 */
void ServDieWithError(char *errorMsg, int sock)
{
	if (sock) close(sock);
	perror(errorMsg);
	fprintf(stderr, "::MONCELLSD SERVER FATAL ERROR: %s : ERRNO : %d", errorMsg, errno);
	syslog(LOG_ERR, "::MONCELLSD SERVER FATAL ERROR: %s : ERRNO : %d", errorMsg, errno);
	exit(EXIT_FAILURE);
}


/** 
 * Print error message, continue running server
 * @param errorMsg The error message to print
 */
int ServERR(char *errorMsg)
{
	perror(errorMsg);
	fprintf(stderr, "::MONCELLSD SERVER WARNING: %s : ERRNO : %d", errorMsg, errno);
	syslog(LOG_ERR, "::MONCELLSD SERVER WARNING: %s : ERRNO : %d", errorMsg, errno);
	return -1;
}


/**
 * Handle the communication between the server and connected client.
 * @param clntsock The client that connected
 */
void handleClient(int clntsock, struct sockaddr_in clntAddr)
{	
	for (;;)
	{
		if (Build_Monitor_Data(clntsock, clntAddr) < 0)
			break;
		sleep(INTERVAL_OF_DATA_UPDATES);
	}
	
	close(clntsock);
}


// MONITOR SERVER PROTOCOL :: RAW TEXT PROTOCOL :: NICE AND SIMPLE
//
// 			"<key>:<data>\n"
//
int Build_Monitor_Data(int clntsock, struct sockaddr_in clntAddr) 
{		
	// --------------------------------------------------
	// SYS :: System Information
	// --------------------------------------------------
	if (update_uptime() < 0)
	{
		if (sendclnt(clntsock, "%s%c%s%c", SYS, CMD_DELIMIT, ERROR_DATA_RETURN, MSG_DELIMIT) < 0)
			return -1;		
	}	
	else
	{
		uname(&info.uname_s);
		// data: uptime, sysname, nodename, release, machine
		char buf[128];
		if (sendclnt(clntsock, "%s%c%s%c%s%c%s%c%s%c%s%c", 
		SYS, CMD_DELIMIT, get_formatted_uptime(buf), CMD_DELIMIT, info.uname_s.sysname, CMD_DELIMIT, info.uname_s.nodename,
		CMD_DELIMIT, info.uname_s.release, CMD_DELIMIT, info.uname_s.machine, MSG_DELIMIT) < 0)
			return -1;
		
	}
	
	// --------------------------------------------------
	// CPU :: Loads, Usage, Freq, Count
	// --------------------------------------------------
	if (update_stat() < 0)
	{
		if (sendclnt(clntsock, "%s%c%s%c", CPU, CMD_DELIMIT, ERROR_DATA_RETURN, MSG_DELIMIT) < 0)
			return -1;		
	}	
	else
	{
		update_load_average();
		
		// data: loads, cpu usage, freq, cpu count
		if (sendclnt(clntsock, "%s%c%.2f %.2f %.2f%c%.0f%c%.0f%c%u%c", 
		CPU, CMD_DELIMIT, info.loadavg[0], info.loadavg[1], info.loadavg[2], CMD_DELIMIT, (100*info.cpu_usage[0]), CMD_DELIMIT, info.freq[0],
		CMD_DELIMIT, info.cpu_count, MSG_DELIMIT) < 0)
			return -1;	
		
	}
	
	// --------------------------------------------------
	// MEM :: RAM, SWAP
	// --------------------------------------------------
	if (update_meminfo() < 0)
	{
		if (sendclnt(clntsock, "%s%c%s%c", MEM, CMD_DELIMIT, ERROR_DATA_RETURN, MSG_DELIMIT) < 0)
			return -1;		
	}	
	else
	{
		// data: memused, memmax, memusage, swapused, swapmax, swapusage
		if (sendclnt(clntsock, "%s%c%llu%c%llu%c%i%c%llu%c%llu%c%i%c", 
		MEM, CMD_DELIMIT, (info.mem/1024), CMD_DELIMIT, (info.memmax/1024), CMD_DELIMIT, get_memory_per_usage(), CMD_DELIMIT, (info.swap/1024),
		CMD_DELIMIT, (info.swapmax/1024), CMD_DELIMIT, get_swap_per_usage(), MSG_DELIMIT) < 0)
			return -1;	
		
	}
	
	// --------------------------------------------------
	// PROCS :: TOTAL PROCS, RUN PROCS, TOP
	// --------------------------------------------------
	if ((update_top() < 0) && (update_total_processes() < 0))
	{
		if (sendclnt(clntsock, "%s%c%s%c", PROCS, CMD_DELIMIT, ERROR_DATA_RETURN, MSG_DELIMIT) < 0)
			return -1;		
	}	
	else
	{
		// build top 1 to 5
		char top1[TLEN], top2[TLEN], top3[TLEN], top4[TLEN], top5[TLEN];
		snprintf(top1, TLEN, "%s %d %.2f %.2f", info.cpu[0]->name, info.cpu[0]->pid, info.cpu[0]->amount, info.cpu[0]->totalmem);	
		snprintf(top2, TLEN, "%s %d %.2f %.2f", info.cpu[1]->name, info.cpu[1]->pid, info.cpu[1]->amount, info.cpu[1]->totalmem);	
		snprintf(top3, TLEN, "%s %d %.2f %.2f", info.cpu[2]->name, info.cpu[2]->pid, info.cpu[2]->amount, info.cpu[2]->totalmem);	
		snprintf(top4, TLEN, "%s %d %.2f %.2f", info.cpu[3]->name, info.cpu[3]->pid, info.cpu[3]->amount, info.cpu[3]->totalmem);	
		snprintf(top5, TLEN, "%s %d %.2f %.2f", info.cpu[4]->name, info.cpu[4]->pid, info.cpu[4]->amount, info.cpu[4]->totalmem);	
		
		// data: total procs, running procs, top1, top2, top3, top4, top5
		if (sendclnt(clntsock, "%s%c%hu%c%hu%c%s%c%s%c%s%c%s%c%s%c", 
		PROCS, CMD_DELIMIT, info.procs, CMD_DELIMIT, info.run_procs, CMD_DELIMIT, top1, CMD_DELIMIT, top2,
		CMD_DELIMIT, top3, CMD_DELIMIT, top4, CMD_DELIMIT, top5, MSG_DELIMIT) < 0)
			return -1;	
		
	}
			
	return 0;
}


/** Send a command through our socket to the IRC-Server */
int sendclnt(int clntsock, char *str, ...) 
{
	int bytes_sent;
	va_list tmpl;
	char tmp[MAXSENDSIZE];
	
	// Send command string
	va_start(tmpl, str);
	vsprintf(tmp, str, tmpl);
	
	// Send data to client
	// ------------------------------------------------
	// BUGFIX :: Server shutsdown when client Ctrl+C's
	// MSG_NOSIGNAL: 
	// This flag requests that the implementation does not to send a 
	// SIGPIPE signal on errors on stream oriented sockets when the 
	// other end breaks the connection. The EPIPE error is still returned 
	// as normal. THIS IS FOR LINUX ONLY!
	// see: http://www.wlug.org.nz/MSG_NOSIGNAL
	bytes_sent =  send(clntsock, tmp, strlen(tmp), MSG_NOSIGNAL);
	
	if (DEBUG) printf("Sending data: %s\n", tmp);
	va_end(tmpl);
	return bytes_sent;
}


/**
 *	daemonize forks the current process and return the parent and the child is the new "main" process
 *	@return int Process ID with -1 being a fail, 1 being the parent
 */
int daemonize(void) 
{
	pid_t pid;

	// Daemonize
	if ((pid = fork()) < 0 )
		return pid; // Error
	else if (pid != 0)
		return 1; // Parent

	// Child process ID
	pid = getpid();

	// Become session leader
	setsid();

	// Clear file mode creation mask
	umask(0);

	// Change working directory
	chdir(DEFAULT_WORKING_DIR);

	// Change the uid and gid
	if (getuid() != 0)
		ServDieWithError("You must be root to run this.", (int)NULL);

	// Get the default uid and gid
	struct passwd *euser = getpwnam(DEFAULT_RUNNING_USERNAME);
	if (euser == NULL)
	    ServDieWithError("Failed to find user info:", (int)NULL);
	if (setuid(euser->pw_uid) < 0)
		ServDieWithError("setuid:", (int)NULL);

	struct group *egroup = getgrnam(DEFAULT_RUNNING_GROUP);
	if (egroup == NULL)
		ServDieWithError("Failed to find group info:", (int)NULL);
	if (setgid(egroup->gr_gid) < 0)
		ServDieWithError("setgid:", (int)NULL);

	// Done, success!
	syslog(LOG_INFO, "Launched at pid (%d)", getpid());
	return pid;
}


/**
 * Handle signals
 * @param int Signal received
 */
void ServerSignalHandler(int signal) 
{
	MS_signals.signalLog[signal]++;
	MS_signals.lastSignalReceived = signal;

	// -------------------------------------------
	// WRITE WHAT EACH SIGNAL DOES HERE!!!!
	// -------------------------------------------
	switch(signal) 
	{
		case SIGINT:
			//if (MS_signals.signalLog[SIGINT] >= 2) 
			ServDieWithError("SIGINT Received", 0);
			printf("Received CTRL+C\n");
		break;
		
		case SIGIO: // Socket ready for I/O :: Default: Ignore
		
		case SIGPIPE: // Attempt to write to a closed socket :: Default: Termination
		
		case SIGALRM: // Expiration of an alarm timer :: Default: Termination
		
		case SIGCHLD: // Child process exit :: Default: Ignore
		
		case SIGABRT: // Abort
		
		case SIGHUP: // Hangup

		default:
			MS_signals.signalReceived = 1;
			char buf[128];
			snprintf(buf , 128, "Received Signal (%d)", signal);
			ServDieWithError(buf, 0);
	}
}


/** Initialize signal handlers */
void MS_signals_init(void (*func)(int signal)) {
	unsigned int i = 0;

	// Assign function provided or default local function
	if (func == NULL)
		func = ServerSignalHandler;

	// Setup signal handlers
	MS_signals.handler.sa_handler = func;

	// Initialize set to full
	if (sigfillset(&MS_signals.handler.sa_mask) < 0)
		ServDieWithError("sigfillset failed.", 0);
		
	MS_signals.handler.sa_flags = 0;

	// Zero out signal log
	for (i = 0; i <= 65; i++) MS_signals.signalLog[i] = 0;
	MS_signals.lastSignalReceived = 0;
	MS_signals.shutdownInitiated = 0;
	
	 
	// --------------------------------------------------
	// SIGINT :: Interrupt (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGINT, &MS_signals.handler, 0) < 0) 
		ServDieWithError("sigfillset setup SIGINT failed.", 0);

	// --------------------------------------------------
	// SIGILL :: Illegal instruction (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGILL, &MS_signals.handler, 0) < 0) 
		ServDieWithError("sigfillset setup SIGILL failed.", 0);

	// --------------------------------------------------
	// SIGABRT :: Abort (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGABRT, &MS_signals.handler, 0) < 0) 
		ServDieWithError("sigfillset setup SIGABRT failed.", 0);

	// --------------------------------------------------
	// SIGHUP :: Hangup (POSIX).
	// --------------------------------------------------
	if (sigaction(SIGHUP, &MS_signals.handler, 0) < 0)
		ServDieWithError("sigfillset setup SIGHUP failed.", 0);

	// --------------------------------------------------
	// SIGSEGV :: Segmentation violation (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGSEGV, &MS_signals.handler, 0) < 0)
		ServDieWithError("sigfillset setup SIGSEGV failed.", 0);

	// --------------------------------------------------
	// SIGPIPE :: Broken Pipe
	// --------------------------------------------------
	if (sigaction(SIGPIPE, &MS_signals.handler, 0) < 0)
		ServDieWithError("sigfillset setup SIGPIPE failed.", 0);

	// --------------------------------------------------
	// SIGSTKFLT :: Stack fault. Signal doesn't exist for OpenBSD
	// --------------------------------------------------
	if (sigaction(SIGSTKFLT, &MS_signals.handler, 0) < 0)
		ServDieWithError("sigfillset setup SIGSTKFLT failed.", 0);

}
