/**
 * @file	info.h
 * @author  Zach Habersang
 * @brief 	Linux System Information Header
 *
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#ifndef _INFO_H_
#define _INFO_H_ 1

#include <sys/utsname.h> // struct uname_s

struct process {
	struct process *next;
	struct process *previous;

	pid_t pid;
	char *name;
	float amount;
	float totalmem;
	
	// User and kernel times are in hundredths of seconds
	unsigned long user_time;
	unsigned long total;
	unsigned long kernel_time;
	unsigned long previous_user_time;
	unsigned long previous_kernel_time;
	unsigned long total_cpu_time;
	unsigned int vsize;
	unsigned int rss;
	unsigned int time_stamp;
	unsigned int counted;
	unsigned int changed;
};

struct sorted_process {
	struct sorted_process *greater;
	struct sorted_process *less;
	struct process *proc;
};

#define MAX_CPU_FREQ_COUNT 10 // use a relatively large number

struct information 
{	
	struct utsname uname_s; // about the system

	float *freq; // store all of the freq of the cpus 

	double uptime;

	// memory information in kilobytes 
	unsigned long long mem, memeasyfree, memfree, memmax, swap, swapmax;
	unsigned long long bufmem, buffers, cached;

	unsigned short procs;
	unsigned short run_procs;

	float *cpu_usage;
	unsigned int cpu_count;
	int cpu_avg_samples;

	int net_avg_samples;

	float loadavg[3];

	// struct usr_info users;
	// struct gateway_info gw_info;
	// struct dns_data nameserver_info;
	
	struct process *cpu[10];
	struct process *memu[10];
	struct process *time[10];

	struct process *first_process;
	unsigned long looped;
};

#define CPU_SAMPLE_COUNT 2
struct cpu_info {
	unsigned long long cpu_user;
	unsigned long long cpu_system;
	unsigned long long cpu_nice;
	unsigned long long cpu_idle;
	unsigned long long cpu_iowait;
	unsigned long long cpu_irq;
	unsigned long long cpu_softirq;
	unsigned long long cpu_steal;
	unsigned long long cpu_total;
	unsigned long long cpu_active_total;
	unsigned long long cpu_last_total;
	unsigned long long cpu_last_active_total;
	double cpu_val[CPU_SAMPLE_COUNT];
};



int update_uptime(void);
char *get_formatted_uptime(char *buf);
int update_meminfo(void);
int update_total_processes(void);
void get_cpu_count(void);
int update_stat(void);
int update_load_average(void);

#define CPUFREQ_PREFIX "/sys/devices/system/cpu"
#define CPUFREQ_POSTFIX "cpufreq/scaling_cur_freq"


// ---------------------------
// top
// ---------------------------

#include <dirent.h> // readdir, directory handling stuff in update_process_table

#include <sys/stat.h> // for open()
#include <fcntl.h> // for open()

#include <limits.h> // ULONG_MAX

#define BUFFER_LEN 1024 // really shouldn't use this, but need for scanf

#define PROCFS_TEMPLATE "/proc/%d/stat"
#define PROCFS_TEMPLATE_MEM "/proc/%d/statm"
#define PROCFS_TEMPLATE_IO "/proc/%d/io"
#define PROCFS_CMDLINE_TEMPLATE "/proc/%d/cmdline"

#define MAX_SP 10 // max number of sorted processes

int update_top(void);
struct process *get_first_process(void);

// top :: pointer to head of process list
int process_find_top(struct process **cpu, struct process **mem, struct process **ptime);

static unsigned long long calc_cpu_total(void);
static int update_process_table(void);
static struct process *find_process(pid_t pid);
static struct process *new_process(int p);
static int calculate_stats(struct process *proc);
static int process_parse_stat(struct process *proc);
inline static void calc_cpu_each(unsigned long long total);
static void process_cleanup(void);
static void delete_process(struct process *p);
static struct sorted_process *malloc_sp(struct process *proc);
static void free_sp(struct sorted_process *sp);
static int insert_sp_element(struct sorted_process *sp_cur, struct sorted_process **p_sp_head, struct sorted_process **p_sp_tail, int max_elements, int compare_funct(struct process *, struct process *));
static void sp_acopy(struct sorted_process *sp_head, struct process **ar, int max_size);

static int compare_cpu(struct process *a, struct process *b);
static int compare_mem(struct process *a, struct process *b);
static int compare_time(struct process *a, struct process *b);



#endif
