/**
 * @file	info.c
 * @author  Zach Habersang
 * @brief 	Linux System Information, based on conky
 *
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

// note: need to free info.cpu_usage and freq
 
#include "moncellsd.h"
#include "info.h"

extern struct information info; // global system information instance


// ##############################################################################################
// ############################################################################################## MAIN
// ##############################################################################################

/* --------------------------------------------------
 *  THIS MAIN IS FOR TESTING ONLY!
 * --------------------------------------------------
 */
#if 0
int main(void)
{
	//
	// uptime
	//
	
	char buf[128];
	update_uptime();
	printf("Uptime: %s\n", get_formatted_uptime(buf)); 
	
	//
	// mount checking
	//
	
	char *fs = "/media/mmc2";
	if (check_mount(fs))
		printf("FS is mounted!!: %s\n", fs);
	else
		printf("NOT mounted!!: %s\n", fs);

	//
	// memory
	//
	
	
	update_meminfo();
	printf("Memory used %lluMiB of %lluMiB :: %i%%\n", (info.mem/1024), (info.memmax/1024), get_memory_per_usage()); 
	printf("Swap used %lluMiB of %lluMiB :: %i%%\n", (info.swap/1024), (info.swapmax/1024), get_swap_per_usage());
	
	//info.mem = info.memmax - info.memfree;
	//info.memeasyfree = info.memfree;
	//info.swap = info.swapmax - info.swap;
	//info.bufmem = info.cached + info.buffers;
	
	//
	// cpu procs
	//
	update_total_processes();
	printf("Number of Processes: %hu\n", info.procs);
	
	//
	// cpu usage and num cpu
	// :: Upates: run_procs,
	update_stat(); 
	printf("::CPU Stat:: CPU Count: %u :: Running Procs: %hu :: Usage: %f\n", info.cpu_count, info.run_procs, info.cpu_usage[1]);
	
	//
	// load averages
	//
	update_load_average();
	printf("Loadavg: %f %f %f\n", info.loadavg[0], info.loadavg[1], info.loadavg[2]);
	
	//
	// top
	//
	update_top();
	printf("Top #1: %s %f %f\n", info.cpu[0]->name, info.cpu[0]->amount, info.cpu[0]->totalmem);
	printf("Top #2: %s %f %f\n", info.cpu[1]->name, info.cpu[1]->amount, info.cpu[1]->totalmem);
	printf("Top #3: %s %f %f\n", info.cpu[2]->name, info.cpu[2]->amount, info.cpu[2]->totalmem);
	printf("Top #4: %s %f %f\n", info.cpu[3]->name, info.cpu[3]->amount, info.cpu[3]->totalmem);
	printf("Top #5: %s %f %f\n\n", info.cpu[4]->name, info.cpu[4]->amount, info.cpu[4]->totalmem);
	
	//
	// uname system information
	//
	update_uname();
	printf("SYSTEM INFO: %s | %s | %s | %s | %s\n", info.uname_s.sysname, info.uname_s.nodename, info.uname_s.release, info.uname_s.version, info.uname_s.machine);
	// Linux   | Nokia-N810-43-7 | 2.6.21-omap1 | #2 Tue Oct 14 11:25:25 EEST 2008 | armv6l
	// sysinfo | nodename        | release      | version                          | machine
	// OS      | network name    | OS Release   | OS Version                       | Hardware identifier
	
	//
	// cpu freq
	//
	if (get_freq())
		printf("CPU FREQ: %f\n", info.freq[0]);

	
	
	return EXIT_SUCCESS;
}
#endif

// ##############################################################################################
// ############################################################################################## uname system information
// ##############################################################################################

int update_uname(void)
{
	return uname(&info.uname_s);
}

// ##############################################################################################
// ############################################################################################## uptime
// ##############################################################################################


/**
 * Use sysinfo or proc to update uptime
 * 
 */
int update_uptime(void)
{
	int retval = 1;
	FILE *fp;

	if (!(fp = fopen("/proc/uptime", "r")))
	{
		// use sysinfo() if /proc doesn't work
		struct sysinfo s_info;
		if (!(sysinfo(&s_info) == 0))
			retval = -1;
		
		info.uptime = (double) s_info.uptime;
		return;
	}
	
	// use /proc/uptime to get uptime
	else
	{
		if (!(fscanf(fp, "%lf", &info.uptime)))
			retval = -1;
		fclose(fp);
	}
	
	return retval;
}

char *get_formatted_uptime(char *buf)
{
	
	double unf_days = (info.uptime / (60*60*24)); // seconds to days (unformatted)
	int days = (int)trunc(unf_days);
	
	double unf_hours = ((unf_days - days) * 24);
	int hours = (int)trunc(unf_hours);
	
	double unf_minutes = ((unf_hours - hours) * 60);
	int minutes = (int)trunc(unf_minutes);
	
	snprintf(buf , 128, "%id %ih %im", days, hours, minutes);
	return buf;
}


// ##############################################################################################
// ############################################################################################## mount checking
// ##############################################################################################

/**
 * Use /etc/mtab to check if filesystem is mounted
 * @param s The device to check if is mounted
 * 
 */
int check_mount(char *s)
{
	int ret = 0;
	FILE *mtab = fopen("/etc/mtab", "r");

	if (mtab) 
	{
		char buf1[256], buf2[128];

		while (fgets(buf1, 256, mtab)) 
		{
			sscanf(buf1, "%*s %128s", buf2); // %*s means ignore the first string
			if (!strcmp(s, buf2)) 
			{
				ret = 1; // is mounted
				break;
			}
		}
		fclose(mtab);
	} 
	
	else
		ServERR("Could not open mtab");
		
	return ret;
}


// ##############################################################################################
// ############################################################################################## memory
// ##############################################################################################


/**
 * Update all memory using /proc/meminfo
 *
 */
int update_meminfo(void)
{
	int retval = 1;
	FILE *meminfo_fp;
	char buf[256];

	info.mem = info.memmax = info.swap = info.swapmax = info.bufmem =
		info.buffers = info.cached = info.memfree = info.memeasyfree = 0;

	// open /proc/meminfo, fail if can't open
	if (!(meminfo_fp = fopen("/proc/meminfo", "r")))
	{
		retval = ServERR("Failed to open /proc/meminfo");
		return retval;
	}
	
	// read each line of /proc/meminfo
	while (!feof(meminfo_fp)) {
		
		// break
		if (fgets(buf, 255, meminfo_fp) == NULL)
			break;

		// TOTAL MEMORY
		if (strncmp(buf, "MemTotal:", 9) == 0) 
			sscanf(buf, "%*s %llu", &info.memmax);
		
		// FREE MEMORY
		else if (strncmp(buf, "MemFree:", 8) == 0)
			sscanf(buf, "%*s %llu", &info.memfree);
		
		// TOTAL SWAP
		else if (strncmp(buf, "SwapTotal:", 10) == 0)
			sscanf(buf, "%*s %llu", &info.swapmax);
		
		// FREE SWAP
		else if (strncmp(buf, "SwapFree:", 9) == 0)
			sscanf(buf, "%*s %llu", &info.swap);
		
		// BUFFERED MEMORY
		else if (strncmp(buf, "Buffers:", 8) == 0)
			sscanf(buf, "%*s %llu", &info.buffers);
		
		// CACHED MEMORY
		else if (strncmp(buf, "Cached:", 7) == 0)
			sscanf(buf, "%*s %llu", &info.cached);
	}

	info.mem = info.memmax - info.memfree;
	info.memeasyfree = info.memfree;
	info.swap = info.swapmax - info.swap;

	info.bufmem = info.cached + info.buffers;


	fclose(meminfo_fp);
	return retval;
}

int get_memory_per_usage()
{
	return (int)(((float)info.mem / (float)info.memmax) * 100);
}

int get_swap_per_usage()
{
	return (int)(((float)info.swap / (float)info.swapmax) * 100);
}

// ##############################################################################################
// ############################################################################################## procs
// ##############################################################################################


int update_total_processes(void)
{
	int retval = 1;
	FILE *fp;

	if (!(fp = fopen("/proc/loadavg", "r"))) 
	{
		// use sysinfo()
		struct sysinfo s_info;
		if (!(sysinfo(&s_info) == 0))
			retval = -1;
		info.procs = s_info.procs;
		return;
	}
	
	// use /proc/loadavg
	else
	{
		if (!(fscanf(fp, "%*f %*f %*f %*d/%hu", &info.procs)))
			retval = -1;
		fclose(fp);
	}
	
	return retval;
}


// ##############################################################################################
// ############################################################################################## cpu stats
// ##############################################################################################


/**
 * Return system frequency in MHz (use divisor=1) or GHz (use divisor=1000) 
 *
 */
int get_freq()
{
	FILE *f;
	int retval = 1;
	char frequency[32];
	char s[256];
	int c; // cpu counter
	
	int gotSys = 0; // flag that says we don't need to use /proc

	if (!info.cpu_count)
		get_cpu_count(); // gives us info.cpu_count
	
	info.freq = malloc(sizeof(float) * info.cpu_count);
	
	// get cpu freq using /sys
	for (c = 0; c < info.cpu_count; c++)
	{
		char current_freq_file[128];
		snprintf(current_freq_file, 127, "%s/cpu%d/%s", CPUFREQ_PREFIX, c, CPUFREQ_POSTFIX);
	
		f = fopen(current_freq_file, "r");
		
		if (!f)
			continue;
	
		else 
		{
			gotSys = 1; // don't use /proc!
			
			// if there's a cpufreq /sys node, read the current frequency from, this node and divide by 1000 to get Mhz
			if (fgets(s, sizeof(s), f)) 
			{
				s[strlen(s) - 1] = '\0';
				info.freq[c] = (atof(s) / 1000); // Mhz
			}
			fclose(f);
			
			c++;
		}
	}
	
	// get out of here if we accessed /sys correctly
	if (gotSys)
		return retval;
	
	// use /proc
	else
	{
		c = 0; // set cpu count back to 0
		f = fopen("/proc/cpuinfo", "r");
		if (!f) 
		{
			retval = -1;
			return retval;
		}

		// read the file
		while (fgets(s, sizeof(s), f) != NULL) 
		{
			if (c == info.cpu_count)
				break;
			
			// search for the "cpu MHz"
			if (strncmp(s, "cpu MHz", 7) == 0) 
			{
				strcpy(frequency, strchr(s, ':') + 2);
				frequency[strlen(frequency) - 1] = '\0'; // strip \n
				info.freq[c] = atof(frequency); // this already comes in MHz!
				c++;
			}
		}
	}
	
	// didn't find any "cpu MHz"
	if (!c)
		retval = -1;
		
	return retval;
}


void get_cpu_count(void)
{
	FILE *stat_fp;
	static int rep = 0;
	char buf[256];

	// only call this function once
	if (info.cpu_usage)
		return;

	// open /proc/stat
	if (!(stat_fp = fopen("/proc/stat", "r")))
		return;

	info.cpu_count = 0;

	// count number of cpus
	while (!feof(stat_fp)) 
	{
		if (fgets(buf, 255, stat_fp) == NULL)
			break;

		if (strncmp(buf, "cpu", 3) == 0 && isdigit(buf[3])) 
			info.cpu_count++;
	}
	info.cpu_usage = malloc((info.cpu_count + 1) * sizeof(float));
	
	info.cpu_avg_samples = 2;

	fclose(stat_fp);
}


int update_stat(void)
{
	int retval = 1;
	
	FILE *stat_fp;
	static struct cpu_info *cpu = NULL;
	char buf[256];
	int i;
	unsigned int idx;
	double curtmp;
	const char *stat_template = NULL;
	unsigned int malloc_cpu_size = 0;

	// add check for !info.cpu_usage since that mem is freed on a SIGUSR1
	if (!info.cpu_usage)
		get_cpu_count();

	// note to zach: took out stat choosing, using just the long stat
	// #define TMPL_LONGSTAT "%*s %llu %llu %llu %llu %llu %llu %llu %llu" ..used by kernel 2.6 and greater
	// #define TMPL_SHORTSTAT "%*s %llu %llu %llu %llu"                    ..used by kernel 2.5 or lower
	// ---------------------------------------------------------------------
	if (!stat_template)
		stat_template = "%*s %llu %llu %llu %llu %llu %llu %llu %llu";

	// init cpu_info structure
	if (!cpu) {
		malloc_cpu_size = (info.cpu_count + 1) * sizeof(struct cpu_info);
		cpu = malloc(malloc_cpu_size);
		memset(cpu, 0, malloc_cpu_size);
	}

	// open /proc/stat
	if (!(stat_fp = fopen("/proc/stat", "r"))) {
		info.run_procs = 0;
		if (info.cpu_usage) memset(info.cpu_usage, 0, info.cpu_count * sizeof(float));
		retval = -1;
		return retval;
	}

	idx = 0;
	
	// loop through /proc/stat
	while (!feof(stat_fp)) {
		if (fgets(buf, 255, stat_fp) == NULL)
			break;

		// set info.run_procs here
		if (strncmp(buf, "procs_running ", 14) == 0)
			sscanf(buf, "%*s %hu", &info.run_procs);


		// make cpu calculations here e.g. info.cpu_usage
		else if (strncmp(buf, "cpu", 3) == 0) 
		{
			// detect e.g. CPU0 or CPU1 etc...
			if (isdigit(buf[3])) idx = atoi(&buf[3]) + 1;
			
			// first CPU line
			else idx = 0;
	
			// get cpu data
			sscanf(buf, stat_template, &(cpu[idx].cpu_user),
				&(cpu[idx].cpu_nice), &(cpu[idx].cpu_system),
				&(cpu[idx].cpu_idle), &(cpu[idx].cpu_iowait),
				&(cpu[idx].cpu_irq), &(cpu[idx].cpu_softirq),
				&(cpu[idx].cpu_steal));

			// add up data to "cpu_total"
			cpu[idx].cpu_total = cpu[idx].cpu_user + cpu[idx].cpu_nice +
				cpu[idx].cpu_system + cpu[idx].cpu_idle +
				cpu[idx].cpu_iowait + cpu[idx].cpu_irq +
				cpu[idx].cpu_softirq + cpu[idx].cpu_steal;

			// calculate "active_total"
			cpu[idx].cpu_active_total = cpu[idx].cpu_total - (cpu[idx].cpu_idle + cpu[idx].cpu_iowait);


			// calculate "cpu_val"
			cpu[idx].cpu_val[0] = (cpu[idx].cpu_active_total - cpu[idx].cpu_last_active_total) / (float)(cpu[idx].cpu_total - cpu[idx].cpu_last_total);
			
			
			// add up "cpu_val" values
			curtmp = 0;
			for (i = 0; i < info.cpu_avg_samples; i++) curtmp = curtmp + cpu[idx].cpu_val[i];

			// finally, get "cpu_usage"
			info.cpu_usage[idx] = curtmp / info.cpu_avg_samples;

			// need this for next iteration
			cpu[idx].cpu_last_total = cpu[idx].cpu_total;
			cpu[idx].cpu_last_active_total = cpu[idx].cpu_active_total;

			// calculate more "cpu_val"
			for (i = info.cpu_avg_samples - 1; i > 0; i--) cpu[idx].cpu_val[i] = cpu[idx].cpu_val[i - 1];
		}
	}
	fclose(stat_fp);
	return retval;
}



// ##############################################################################################
// ############################################################################################## loadavg
// ##############################################################################################


int update_load_average(void)
{
	int retval = 1;
	FILE *fp;
	
	if (!(fp = fopen("/proc/loadavg", "r"))) 
	{
		// use getloadavg() from stdlib.h
		double v[3];
		if (getloadavg(v, 3) < 0)
			retval = -1;
		info.loadavg[0] = (float) v[0];
		info.loadavg[1] = (float) v[1];
		info.loadavg[2] = (float) v[2];
	}
	
	// use /proc/loadavg
	else
	{
		if(!(fscanf(fp, "%f %f %f", &info.loadavg[0], &info.loadavg[1], &info.loadavg[2])))
			retval = -1;
		fclose(fp);
	}
	return retval;
}


// ##############################################################################################
// ############################################################################################## top
// ##############################################################################################

static struct process *first_process = 0;
static unsigned long long previous_total = 0;
static unsigned long g_time = 0;

// maximum size of individual text buffers, ie $exec buffer size .. for top
#define DEFAULT_TEXT_BUFFER_SIZE 1024
unsigned int text_buffer_size = DEFAULT_TEXT_BUFFER_SIZE;
int cpu_separate = 0;

#define MIN(a,b) (((a) < (b)) ? (a) : (b))

int update_top(void)
{
	int retval = 1;
	if (process_find_top(info.cpu, info.memu, info.time) < 0)
		retval = -1;
	else 
		info.first_process = get_first_process();
	return retval;
}

struct process *get_first_process(void)
{
	return first_process;
}

/**
 * Get a sorted list of the top cpu hogs and top mem hogs.
 * Results are stored in the cpu,mem arrays in decreasing order[0-9].
 */
int process_find_top(struct process **cpu, struct process **mem, struct process **ptime)
{
	int top_cpu = 1, top_mem = 0, top_time = 0; // note to zach: only enabled top_cpu because only plan to use that
	
	struct sorted_process *spc_head = NULL, *spc_tail = NULL, *spc_cur = NULL; // sorted process cpu
	struct sorted_process *spm_head = NULL, *spm_tail = NULL, *spm_cur = NULL; // sorted process memory
	struct sorted_process *spt_head = NULL, *spt_tail = NULL, *spt_cur = NULL; // sorted process time

	struct process *cur_proc = NULL;
	unsigned long long total = 0;

	if (!top_cpu && !top_mem && !top_time)
		return -1;
	
	// calculate the total of the processor 
	if ((total = calc_cpu_total()) < 0)
		return -1;
	
	// update the table with process list 
	if (update_process_table() < 0)
		return -1;
	
	calc_cpu_each(total);		// and then the percentage for each task 
	process_cleanup();			// cleanup list from exited processes 

	cur_proc = first_process;

	while (cur_proc != NULL) 
	{
		
		if (top_cpu) 
		{
			spc_cur = malloc_sp(cur_proc);
			insert_sp_element(spc_cur, &spc_head, &spc_tail, MAX_SP, &compare_cpu);
		}
		
		if (top_mem) 
		{
			spm_cur = malloc_sp(cur_proc);
			insert_sp_element(spm_cur, &spm_head, &spm_tail, MAX_SP, &compare_mem);
		}
		
		if (top_time) 
		{
			spt_cur = malloc_sp(cur_proc);
			insert_sp_element(spt_cur, &spt_head, &spt_tail, MAX_SP, &compare_time);
		}

		cur_proc = cur_proc->next;
	}

	if (top_cpu)
		sp_acopy(spc_head, cpu, MAX_SP);
	if (top_mem)
		sp_acopy(spm_head, mem, MAX_SP);
	if (top_time)
		sp_acopy(spt_head, ptime, MAX_SP);

	return 0;
}


/**
 * Calculate cpu total
 * 
 */
static unsigned long long calc_cpu_total(void)
{
	unsigned long long total = 0;
	unsigned long long t = 0;
	int rc;
	int ps;
	char line[BUFFER_LEN] = { 0 };
	unsigned long long cpu = 0;
	unsigned long long niceval = 0;
	unsigned long long systemval = 0;
	unsigned long long idle = 0;
	unsigned long long iowait = 0;
	unsigned long long irq = 0;
	unsigned long long softirq = 0;
	unsigned long long steal = 0;
	
	// note to zach: took out stat choosing, using just the long stat
	// #define TMPL_SHORTPROC "%*s %llu %llu %llu %llu"                    ..used by kernel 2.5 or lower
	// #define TMPL_LONGPROC "%*s %llu %llu %llu %llu %llu %llu %llu %llu" ..used by kernel 2.6 and greater
	// ---------------------------------------------------------------------
	const char *template = "%*s %llu %llu %llu %llu %llu %llu %llu %llu";


	// read /proc/stat, return if can't read
	ps = open("/proc/stat", O_RDONLY);
	rc = read(ps, line, sizeof(line));
	close(ps);
	if (rc < 0)
		return -1;

	// parse line
	sscanf(line, template, &cpu, &niceval, &systemval, &idle, &iowait, &irq, &softirq, &steal);
	total = cpu + niceval + systemval + idle + iowait + irq + softirq + steal;

	t = total - previous_total;
	previous_total = total;

	return t;
}

/**
 * Update process table
 */
static int update_process_table(void)
{
	DIR *dir;
	struct dirent *entry;

	if (!(dir = opendir("/proc")))
		return -1;

	++g_time;

	// get list of processes from /proc directory
	while ((entry = readdir(dir))) 
	{
		pid_t pid;

		// problem reading list of processes
		if (!entry)
		{
			closedir(dir);
			return -1;
		}

		// get pid directory name
		if (sscanf(entry->d_name, "%d", &pid) > 0) 
		{
			struct process *p;

			p = find_process(pid);
			if (!p) {
				p = new_process(pid);
			}

			// compute each process cpu usage
			calculate_stats(p);
		}
	}

	closedir(dir);

	return 0;
}

/**
 * Use first_process pointer to find a process with id -> pid
 */
static struct process *find_process(pid_t pid)
{
	struct process *p = first_process;

	while (p) 
	{
		if (p->pid == pid)
			return p;

		p = p->next;
	}
	return 0;
}

/** 
 * Create a new process object and insert it into the process list
 */
static struct process *new_process(int p)
{
	struct process *proc;
	proc = (struct process *) malloc(sizeof(struct process));

	// clean up memory first
	memset(proc, 0, sizeof(struct process));

	// do stitching necessary for doubly linked list
	proc->name = 0;
	proc->previous = 0;
	proc->next = first_process;
	if (proc->next) {
		proc->next->previous = proc;
	}
	first_process = proc;

	proc->pid = p;
	proc->time_stamp = 0;
	proc->previous_user_time = ULONG_MAX;
	proc->previous_kernel_time = ULONG_MAX;
	proc->counted = 1;

	return proc;
}


/**
 * Get process structure for process pid
 * 
 * conky note:
 * This function seems to hog all of the CPU time.
 * I can't figure out why - it doesn't do much. 
 */
static int calculate_stats(struct process *proc)
{
	int rc;

	// compute each process cpu usage by reading /proc/<proc#>/stat
	rc = process_parse_stat(proc);
	if (rc)
		return 1;

	return 0;
}

/**
 * Extract information from /proc.
 * These are the guts that extract information out of /proc.
 * 
 */
static int process_parse_stat(struct process *proc)
{
	struct information *cur = &info;
	char line[BUFFER_LEN] = { 0 }, filename[BUFFER_LEN], procname[BUFFER_LEN];
	int ps;
	unsigned long user_time = 0;
	unsigned long kernel_time = 0;
	int rc;
	char *r, *q;
	int endl;
	int nice_val;
	char *lparen, *rparen;

	snprintf(filename, sizeof(filename), PROCFS_TEMPLATE, proc->pid);

	ps = open(filename, O_RDONLY);
	if (ps < 0)
		return 1;

	// mark process as up-to-date.
	proc->time_stamp = g_time;

	rc = read(ps, line, sizeof(line));
	close(ps);
	if (rc < 0)
		return 1;

	// extract cpu times from data in /proc filesystem
	lparen = strchr(line, '(');
	rparen = strrchr(line, ')');
	if(!lparen || !rparen || rparen < lparen)
		return 1; // this should not happen

	rc = MIN((unsigned)(rparen - lparen - 1), sizeof(procname) - 1);
	strncpy(procname, lparen + 1, rc);
	procname[rc] = '\0';
	rc = sscanf(rparen + 1, "%*s %*s %*s %*s %*s %*s %*s %*s %*s %*s %*s %lu %lu %*s %*s %*s %d %*s %*s %*s %u %u", &proc->user_time, &proc->kernel_time, &nice_val, &proc->vsize, &proc->rss);
	if (rc < 5)
		return 1;
	
	// remove any "kdeinit: "
	if (procname == strstr(procname, "kdeinit")) 
	{
		// get filename
		snprintf(filename, sizeof(filename), PROCFS_CMDLINE_TEMPLATE, proc->pid);

		// open
		ps = open(filename, O_RDONLY);
		if (ps < 0)
			return 1;

		// read
		endl = read(ps, line, sizeof(line));
		close(ps);

		// null terminate the input
		line[endl] = 0;
		
		// account for "kdeinit: "
		if ((char *) line == strstr(line, "kdeinit: "))
			r = ((char *) line) + 9;
		else
			r = (char *) line;

		q = procname;
		
		// stop at space
		while (*r && *r != ' ')
			*q++ = *r++;
			
		*q = 0;
	}
	
	
	if (proc->name)
		free(proc->name);

	proc->name = (char *)strndup(procname, text_buffer_size);
	proc->rss *= getpagesize();

	if (!cur->memmax)
		update_total_processes();

	proc->total_cpu_time = proc->user_time + proc->kernel_time;
	proc->totalmem = (float)(((float) proc->rss / cur->memmax) / 10);
	
	// zach: commented because the if statement passes the first time always :|
	if (proc->previous_user_time == ULONG_MAX) {
		proc->previous_user_time = proc->user_time;
	}
	if (proc->previous_kernel_time == ULONG_MAX) {
		proc->previous_kernel_time = proc->kernel_time;
	}

	// store the difference of the user_time
	user_time = proc->user_time - proc->previous_user_time;
	kernel_time = proc->kernel_time - proc->previous_kernel_time;

	// backup the process->user_time for next time around
	proc->previous_user_time = proc->user_time;
	proc->previous_kernel_time = proc->kernel_time;

	// store only the difference of the user_time here...
	proc->user_time = user_time;
	proc->kernel_time = kernel_time;

	return 0;
}

/**
 * Calculate each processes cpu		
 */
inline static void calc_cpu_each(unsigned long long total)
{
	struct process *p = first_process;

	// loop through the linked list and calculate each cpu_usage
	while (p) {
		p->amount = 100.0 * (cpu_separate ? info.cpu_count : 1) * (p->user_time + p->kernel_time) / (float) total;
		p = p->next;
	}
}


/**
 * Strip dead process entries
 * 
 */
static void process_cleanup(void)
{
	struct process *p = first_process;

	while (p) 
	{
		struct process *current = p;
		p = p->next;
		
		// delete processes that have died
		if (current->time_stamp != g_time)
			delete_process(current);
	}
}


/**
 * Destroy and remove a process  
 * 
 */
static void delete_process(struct process *p)
{
	// maintain doubly linked list
	if (p->next)
		p->next->previous = p->previous;
	if (p->previous)
		p->previous->next = p->next;
	else
		first_process = p->next;

	// free from heap
	if (p->name)
		free(p->name);
		
	free(p);
}

/**
 * Create a new sp_process structure
 * 
 */
static struct sorted_process *malloc_sp(struct process *proc)
{
	struct sorted_process *sp;
	sp = malloc(sizeof(struct sorted_process));
	sp->greater = NULL;
	sp->less = NULL;
	sp->proc = proc;
	return sp;
}

/** 
 * Free a sp_process structure
 * 
 */
static void free_sp(struct sorted_process *sp)
{
	free(sp);
}

/** 
 * Insert this process into the list in a sorted fashion,
 * or destroy it if it doesn't fit on the list 
 * 
 */
static int insert_sp_element(struct sorted_process *sp_cur, struct sorted_process **p_sp_head, struct sorted_process **p_sp_tail, int max_elements, int compare_funct(struct process *, struct process *))
{
	struct sorted_process *sp_readthru = NULL, *sp_destroy = NULL;
	int did_insert = 0, x = 0;

	if (*p_sp_head == NULL) 
	{
		*p_sp_head = sp_cur;
		*p_sp_tail = sp_cur;
		return 1;
	}
	
	for (sp_readthru = *p_sp_head, x = 0; sp_readthru != NULL && x < max_elements; sp_readthru = sp_readthru->less, x++) 
	{
		if (compare_funct(sp_readthru->proc, sp_cur->proc) > 0 && !did_insert) 
		{
			// sp_cur is bigger than sp_readthru so insert it before sp_readthru
			sp_cur->less = sp_readthru;
			
			if (sp_readthru == *p_sp_head)
				*p_sp_head = sp_cur; // insert as the new head of the list
			
			else 
			{
				// insert inside the list
				sp_readthru->greater->less = sp_cur;
				sp_cur->greater = sp_readthru->greater;
			}
			
			sp_readthru->greater = sp_cur;
			
			// element was inserted, so increase the counter
			did_insert = ++x;
		}
	}
	
	if (x < max_elements && sp_readthru == NULL && !did_insert) 
	{
		// sp_cur is the smallest element and list isn't full, so insert at the end
		(*p_sp_tail)->less = sp_cur;
		sp_cur->greater = *p_sp_tail;
		*p_sp_tail = sp_cur;
		did_insert = x;
	} 
	
	else if (x >= max_elements) 
	{
		// We inserted an element and now the list is too big by one. Destroy the smallest element
		sp_destroy = *p_sp_tail;
		*p_sp_tail = sp_destroy->greater;
		(*p_sp_tail)->less = NULL;
		free_sp(sp_destroy);
	}
	
	if (!did_insert)
		// sp_cur wasn't added to the sorted list, so destroy it
		free_sp(sp_cur);
	
	return did_insert;
}


/**
 * Copy the procs in the sorted list to the array, and destroy the list 
 * 
 */
static void sp_acopy(struct sorted_process *sp_head, struct process **ar, int max_size)
{
	struct sorted_process *sp_cur, *sp_tmp;
	int x;

	sp_cur = sp_head;
	for (x = 0; x < max_size && sp_cur != NULL; x++) 
	{
		ar[x] = sp_cur->proc;
		sp_tmp = sp_cur;
		sp_cur = sp_cur->less;
		free_sp(sp_tmp);
	}
}

/** 
 * Cpu comparison function for insert_sp_element 
 * 
 */
static int compare_cpu(struct process *a, struct process *b)
{
	if (a->amount < b->amount)
		return 1;
	else if (a->amount > b->amount)
		return -1;
	else
		return 0;
}

/** 
 * Mem comparison function for insert_sp_element 
 * 
 */
static int compare_mem(struct process *a, struct process *b)
{
	if (a->totalmem < b->totalmem)
		return 1;
	else if (a->totalmem > b->totalmem)
		return -1;
	else
		return 0;
}

/** 
 * CPU time comparision function for insert_sp_element 
 * 
 */
static int compare_time(struct process *a, struct process *b)
{
	return b->total_cpu_time - a->total_cpu_time;
}
