/**
 * @file	moncells.c
 * @author  Zach Habersang
 * @brief 	System Monitor Application
 * 
 * Copyright (C) 2008 Zach Habersang
 *
 * @section LICENSE
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include "moncells.h"

liqcell *moncells_create();
int ResolveHostName(char hostname[]);
void ClientSignalHandler(int signal);
void MC_signals_init(void (*func)(int signal));

struct {
	unsigned int signalLog[64];

	unsigned int signalReceived:1;
	int lastSignalReceived;

	unsigned int shutdownInitiated:1;

	struct sigaction handler;
} MC_signals;	   

	   
/** Shuts down monitor, logs the shutdown, closes canvas, close the app */
void closeall()
{
	// Check if app is already closing
	static int alreadyclosing = 0;
	if (alreadyclosing++) return;
	
	// close canvas
	liqapp_log("%s closing", APP_NAME);
	liqapp_log("Closing canvas");
	liqcanvas_close();
	
	// close app
	liqapp_log("closing app");
	liqapp_close();
	liqapp_log("goodbye.");
	alreadyclosing=0;
}


/** 
 * Print error message, close the socket, and exit with an error.
 * @param errorMsg The error message to print
 * @return int Success or Failure 
 */
int ClntDieWithError(char *errorMsg)
{
	syslog(LOG_ERR, errorMsg);
	closeall();
	return liqapp_errorandfail(-1, errorMsg);
}


/** 
 * ------------------------
 * Monitor Application Main 
 * ------------------------
 */
int main (int argc, char* argv[])
{
	// set signals
	MC_signals_init(NULL);
	
	// init app
	if(liqapp_init(argc, argv , APP_NAME, APP_VERSION) != 0)
		return ClntDieWithError("liqapp_init failed");

	// init canvas, 800x480 resolution, fullscreen :: NIT Settings
	if(0 != liqcanvas_init(800, 480, 1))
		return ClntDieWithError("canvas init");
		
	// build app	
	liqcell *moncells = moncells_create();
	
	// ----------------------------------------
	// START LIQBASE GUI APP EVENT LOOP
	// ----------------------------------------
		
	if (moncells) 
	{
		liqapp_log("Starting global Monitor Application!");
		liqcell_easyrun(moncells);
		
		// close app
		liqapp_log("Monitor Application Run Complete!");
		liqcell_release(moncells);		
	}

	// shutdown app
	closeall();
	return EXIT_SUCCESS;
}


/* 
 * --------------------------------------------
 * Widget Handling Functions
 * --------------------------------------------
 */

static int widget_click(liqcell *self, liqcellclickeventargs *args)
{
	args->newdialogtoopen = liqcell_getcontent(self);
	return 1;
}



static int hostnames_show_button_click(liqcell *self,liqcellclickeventargs *args, liqcell *app)
{
	liqcell *hostnames = liqcell_quickcreatevis("hostnames", "hostnames", 0, 0, -1, -1);
	
	if(hostnames)
	{
		liqcell_easyrun(hostnames);
		liqcell_release(hostnames);
	}
}


/**
 * ----------------------------------------
 * Monitor Application Running Function
 * ----------------------------------------
 * Creates the monitor_universe (loads all widgets) and runs liqbase GUI loop
 * 
 */
liqcell *moncells_create()
{
	liqcell *self = liqcell_quickcreatewidget(APP_NAME, "form", 800, 480);

	if(self)
	{
		// Add meta data
		liqcell *meta = mkmeta_group(
						mkmeta_title(        APP_NAME),
						mkmeta_description(  APP_DESCRIPT),
						mkmeta_author(       APP_AUTHOR),
						mkmeta_version(      APP_VERSION),
						NULL);
		liqcell_child_append(self,meta);
		
		// create titlebar
		liqcell *title = liqcell_quickcreatevis("title", "label", 0,0, 800, MONITOR_TITLEBAR_HEIGHT);
		liqcell_setfont(title, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (42), 0));
		liqcell_setcaption(title, MONITOR_TITLEBAR_CAPTION);
		liqcell_propsets(title, "textcolor", CYAN);
		liqcell_propsets(title, "backcolor", DARK);
		liqcell_propseti(title, "textaligny", 2);
		liqcell_propseti(title, "textalign", 2);
		liqcell_child_append(self, title);
		
		// create and insert body
		liqcell *body = liqcell_quickcreatevis("body", NULL, 0, MONITOR_TITLEBAR_HEIGHT - 15, self->w, self->h - MONITOR_TITLEBAR_HEIGHT - 55);
		liqcell_child_append(self, body);
		
		// try to monitor local machine, enter cell into body if successful
		char *hostname = LOCALHOST;
		Monitor_Cell_Init(self, hostname);
		
		// autoconnect to the names listed in "hostnames.conf"
		if (autoconnect_hostnames(self) < 0)
			liqapp_log("!-- UNABLE TO AUTOCONNECT TO HOSTNAMES IN HOSTNAMES.CONF --!"); 
		
		// add input box
		liqcell *ipbar = liqcell_quickcreatevis("main_ipbar", "ipbar", 250, 425, 600, 50);
		liqcell_child_append(self, ipbar);
		
		// add hostnames button
		liqcell *hostnames = liqcell_quickcreatevis("hostnames_button", "commandbutton", 50, 425, 200, 50);
		liqcell_handleradd_withcontext(hostnames, "click", hostnames_show_button_click, self);
	    liqcell_setfont(hostnames, liqfont_cache_getttf("/usr/share/fonts/nokia/nosnb.ttf", (24), 0));
		liqcell_setcaption(hostnames, "Edit Hostnames");
		liqcell_propsets(hostnames, "backcolor", BLACK);
		liqcell_propsets(hostnames, "textcolor", CYAN);
		liqcell_propseti(hostnames, "textalign", 2);
		liqcell_propseti(hostnames, "textaligny", 2);
		liqcell_propseti(hostnames, "lockaspect", 1);
		liqcell_child_append(self, hostnames);
	
		// set background
		liqcell_propsets(self, "backcolor", DARK);
		
		return self;
	}
	return NULL;
}

/** Read and connect to hosts in "hostnames.conf" file */
int autoconnect_hostnames(liqcell *self)
{
	int retval = 0;
	FILE *hnconf = fopen(HOSTNAME_FILE, "r");

	if (hnconf) 
	{
		char buf[256], hostname[128];

		while (fgets(buf, 256, hnconf)) 
		{
			sscanf(buf, "%128s", hostname);
			Monitor_Cell_Init(self, hostname);
		}
		fclose(hnconf);
	} 
	else
		retval = -1;
		
	return retval;
}

/** 
 * Monitor Cell Init 
 * @param self The application liqcell with the child "body"
 * @param hostname The hostname string for the moncell client to connect a server
 **/
void Monitor_Cell_Init(liqcell *self, char *hostname)
{
	int servsock;
	
	// 0 is a failure to resolve hostname
	if ((servsock = ResolveHostName(hostname)) < 0)
	{
		liqapp_log("Did not init Monitor Cell for: %s", hostname);
		return;
	}
	
	else
	{
		liqapp_log("Successfully Connected To: %s", hostname);
		
		liqcell *m; // moncell 
		liqcell *wrap; // wrapper
		char wrap_name[32];
		snprintf(wrap_name, 32, "%s_wrap", hostname); // build wrap string
	
		// get body
		liqcell *body = liqcell_child_lookup(self, "body");
	
		// build new moncell
		m = liqcell_quickcreatevis(hostname, "moncell", 0, 0, 0, 0);
		
		// set needed props, hostname and socket
		liqcell_propsets(m, "hostname", hostname);
		liqcell_propseti(m, "servsock", servsock);
		
		// wrap moncell, needed for click handler to work
		wrap = liqcell_quickcreatevis(wrap_name ,NULL, 0, 0, 0, 0);
		liqcell_handleradd(wrap, "click", widget_click);
		liqcell_setcontent(wrap, m);
		liqcell_propseti(wrap, "lockaspect", 1);
		
		// add wrapper
		liqcell_child_insert(body, wrap);
	
		// rearrange and set dirty
		liqcell_child_arrange_easytile(body);
		liqcell_setdirty(self, 1);
	}
}

/** 
 * Check validity of a hostname in dotted-quad or domain name representation 
 * @param hostname Hostname to connect to 
 * @return int Socket Descriptor or FAILURE
 */
int ResolveHostName(char hostname[])
{
	int servsock;
    struct sockaddr_in servAddr;
	struct hostent *host;
	
	fd_set sockSet;
	struct timeval selTimeout;
	
	long arg;
	
	// check hostname
	if ((host = gethostbyname(hostname)) == NULL)
	{
		liqapp_log("!-- UNABLE TO RESOLVE HOSTNAME: %s", hostname); 
		return -1;
	}
	
    // init client socket
    if ((servsock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0)
	{
		liqapp_log("socket() failed", servsock);
		return -1;
	}
	
	// set address
    memset(&servAddr, 0, sizeof(servAddr));
    servAddr.sin_family = AF_INET;
	servAddr.sin_addr.s_addr = *((unsigned long *) host->h_addr_list[0]);
    servAddr.sin_port = htons(PORT);
	
	// arrange for nonblocking I/O and SIGIO delivery
	if (fcntl(servsock, F_SETFL, O_NONBLOCK) < 0) 
	{
		liqapp_log("Unable to put client socket into nonblocking/async mode");
		return -1;
	}
	// Connect client to server, return if can't connect
    if (connect(servsock, (const void *)&servAddr, sizeof(servAddr)) < 0)
	{
		// ------------------------------------
		// use select() to connect with timeout
		// ------------------------------------
		
		FD_ZERO(&sockSet); // reset struct for socket insert
		FD_SET(servsock, &sockSet);
		
		selTimeout.tv_sec = CLIENT_CONNECT_TIMEOUT_SECONDS;
		selTimeout.tv_usec = 0; // 500,000 millisec. == 0.5 sec.
		
		if (select(servsock + 1, &sockSet, NULL, NULL, &selTimeout) == 0)
		{
			liqapp_log("Network Monitor Client :: NONFATAL ERROR :: select() failed to: %s :: SYSTEM ERROR: %s", hostname, NonFatalSysError()); // NON-FATAL ERROR
			return -1;
		}
		else
			liqapp_log("Selected successfully to %s", hostname);
	}
	else
		liqapp_log("Connected successfully to %s", hostname);
		
		
	// set blocking again
	arg = fcntl(servsock, F_GETFL, NULL);
	arg &= (~O_NONBLOCK);
	fcntl(servsock, F_SETFL, arg);
	
	return servsock;
}

/**
 * Get a system error
 * @return char* The system error
 */
char *NonFatalSysError()
{
	char *syserror = strerror(errno); // Get latest system error
	if(!syserror) syserror="*UNKNOWN ERROR*";
	return syserror;
}

/**
 * Handle signals
 * @param int Signal received
 */
void ClientSignalHandler(int signal) 
{
	MC_signals.signalLog[signal]++;
	MC_signals.lastSignalReceived = signal;

	// -------------------------------------------
	// WRITE WHAT EACH SIGNAL DOES HERE!!!!
	// -------------------------------------------
	switch(signal) 
	{
		case SIGINT:
			//if (MC_signals.signalLog[SIGINT] >= 2) 
			ClntDieWithError("SIGINT Received");
			break;
		
		case SIGIO: // Socket ready for I/O :: Default: Ignore
		
		case SIGPIPE: // Attempt to write to a closed socket :: Default: Termination
		
		case SIGALRM: // Expiration of an alarm timer :: Default: Termination
		
		case SIGCHLD: // Child process exit :: Default: Ignore
		
		case SIGABRT: // Abort
		
		case SIGHUP: // Hangup

		default:
			MC_signals.signalReceived = 1;
			char buf[128];
			snprintf(buf , 128, "Received Signal (%d)", signal);
			ClntDieWithError(buf);
	}
}

/** Initialize signal handlers */
void MC_signals_init(void (*func)(int signal)) 
{
	unsigned int i = 0;

	// Assign function provided or default local function
	if (func == NULL)
		func = ClientSignalHandler;

	// Setup signal handlers
	MC_signals.handler.sa_handler = func;

	// Initialize set to full
	if (sigfillset(&MC_signals.handler.sa_mask) < 0)
		ClntDieWithError("sigfillset failed.");
		
	MC_signals.handler.sa_flags = 0;

	// Zero out signal log
	for (i = 0; i <= 65; i++) MC_signals.signalLog[i] = 0;
	MC_signals.lastSignalReceived = 0;
	MC_signals.shutdownInitiated = 0;
	
	 
	// --------------------------------------------------
	// SIGINT :: Interrupt (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGINT, &MC_signals.handler, 0) < 0) 
		ClntDieWithError("sigfillset setup SIGINT failed.");

	// --------------------------------------------------
	// SIGILL :: Illegal instruction (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGILL, &MC_signals.handler, 0) < 0) 
		ClntDieWithError("sigfillset setup SIGILL failed.");

	// --------------------------------------------------
	// SIGABRT :: Abort (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGABRT, &MC_signals.handler, 0) < 0) 
		ClntDieWithError("sigfillset setup SIGABRT failed.");

	// --------------------------------------------------
	// SIGHUP :: Hangup (POSIX).
	// --------------------------------------------------
	if (sigaction(SIGHUP, &MC_signals.handler, 0) < 0)
		ClntDieWithError("sigfillset setup SIGHUP failed.");

	// --------------------------------------------------
	// SIGSEGV :: Segmentation violation (ANSI).
	// --------------------------------------------------
	if (sigaction(SIGSEGV, &MC_signals.handler, 0) < 0)
		ClntDieWithError("sigfillset setup SIGSEGV failed.");

	// --------------------------------------------------
	// SIGPIPE :: Broken Pipe
	// --------------------------------------------------
	if (sigaction(SIGPIPE, &MC_signals.handler, 0) < 0)
		ClntDieWithError("sigfillset setup SIGPIPE failed.");

	// --------------------------------------------------
	// SIGSTKFLT :: Stack fault. Signal doesn't exist for OpenBSD
	// --------------------------------------------------
	if (sigaction(SIGSTKFLT, &MC_signals.handler, 0) < 0)
		ClntDieWithError("sigfillset setup SIGSTKFLT failed.");

}
